/*
Copyright 1990-2001 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/


package com.sun.iiim;

import java.awt.Dimension;
import java.text.AttributedString;
import java.text.AttributedCharacterIterator;

/**
 * This class represents InputMethod's lookup event.
 */
public class IIIMLookupEvent extends IIIMEvent {

    private int master;
    private int choicePerWindow;
    private Dimension rowColum;
    private int direction;
    private int labelOwner;
    private int firstCandidate;
    private int lastCandidate;
    private int currentCandidate;
    private String[] candidates;
    private AttributedCharacterIterator[] candidateIterator;
    private String[] labelList;
    private AttributedCharacterIterator[] labelIterator;
    private int processType;
    private int indexPage;
    private String title;
    private AttributedCharacterIterator titleIterator;

    //
    // The first argument of each constructor type may be redundant, but
    // it will allow the event with the same kind of aruguments
    // and different type of event.
    //

    /**
     *  Constructor for START event
     *
     * Each parameter represents IIIMPv1.2 IM_LOOKUP_CHOICE_START
     * opration's parameter.
     */
    public IIIMLookupEvent(int type, int master, int choicePerWindow,
			   Dimension rowColum, int direction, int labelOwner) {
	super(LOOKUP);
	if (type != START) {
	    throw new IllegalArgumentException();
	}
	this.type = START;
	this.master = master;
	this.choicePerWindow = choicePerWindow;
	this.rowColum = rowColum;
	this.direction = direction;
	this.labelOwner = labelOwner;
    }

    /**
     * constructor for DONE event
     */
    public IIIMLookupEvent(int type) {
	super(LOOKUP);
	if (type != DONE) {
	    throw new IllegalArgumentException();
	}
	this.type = DONE;
    }

    /**
     * Constructor for DRAW event
     *
     * Each parameter represents IIIMPv1.2 IM_LOOKUP_DRAW
     * operation's parameter.
     *
     */
    public IIIMLookupEvent(int type, int firstCandidate, int lastCandidate,
			   int currentCandidate, String[] candidates,
			   String[] labelList, String title) {
	super(LOOKUP);
	if (type != DRAW) {
	    throw new IllegalArgumentException();
	}
	this.type = DRAW;
	this.firstCandidate = firstCandidate;
	this.lastCandidate = lastCandidate;
	this.currentCandidate = currentCandidate;
	this.candidates = candidates;
	this.labelList = labelList;
	this.title = title;
    }

    /**
     * Constructor for DRAW event
     *
     * Each parameter represents IIIMPv1.2 IM_LOOKUP_DRAW
     * operation's parameter.
     */
    public IIIMLookupEvent(int type, int firstCandidate, int lastCandidate,
			   int currentCandidate,
			   AttributedCharacterIterator[] candidates,
			   AttributedCharacterIterator[] labelList,
			   AttributedCharacterIterator title) {
	super(LOOKUP);
	if (type != DRAW) {
	    throw new IllegalArgumentException();
	}
	this.type = DRAW;
	this.firstCandidate = firstCandidate;
	this.lastCandidate = lastCandidate;
	this.currentCandidate = currentCandidate;
	candidateIterator = candidates;
	labelIterator = labelList;
	titleIterator = title;
    }

    /**
     * Constructor for PROCESS event
     */
    public IIIMLookupEvent(int type, int processType,
			   int indexPage) {
	super(LOOKUP);
	if (type != PROCESS) {
	    throw new IllegalArgumentException();
	}
	this.type = PROCESS;
	this.processType = processType;
	this.indexPage = indexPage;
    }

    public final static int START   = 0;
    public final static int DONE    = 1;
    public final static int DRAW    = 2;
    public final static int PROCESS = 3;

    public final static int SERVER_MASTER = 1;
    public final static int CLIENT_MASTER = 2;
    
    public final static int DRAW_HORIZONTALLY = 0;
    public final static int DRAW_VERTICALLY   = 1;

    public final static int SERVER_LABEL = 0;
    public final static int CLIENT_LABEL = 1;

    public final static int PROCESS_INDEX = 0;
    public final static int PROCESS_PAGE = 1;

    public final static int NEXT_PAGE  = 0;
    public final static int PREV_PAGE  = 1;
    public final static int FIRST_PAGE = 2;
    public final static int LAST_PAGE  = 3;

    /**
     * Each type means the following methods return valid value
     *
     *   type: | methods return valid value
     *   ------+-----------------------------
     *   START 
     *       	int getMaster()
     * 		int getChoicePerWindow()
     *		Dimension getRowColum()
     *		int getDirection()
     *		int getLabelOwner()
     *
     *   DRAW
     *		int getFirstCandidate()
     *		int getLastCandidate()
     *		int getCurrentCandidate()
     *		String[] getCandidates()
     *		String[] getLabelList() - if getLableOwner is SERVER_LABEL
     *		String getTitle()
     *		AttributedCharacterIterator[] getCandidateIterator()
     *		AttributedCharacterIterator[] getLabelIterator()
     *		AttributedCharacterIterator getTitleIterator()
     *
     *   PROCESS
     *		int getProcessType()
     *		int getProcessIndexPage()
     *
     */

    // START event methods

    /**
     * @return SERVER_MASTER or CLIENT_MASTER
     */
    public int getMaster() {
	return master;
    }

    /**
     * @return # of candidates per window
     */
    public int getChoicePerWindow() {
	return choicePerWindow;
    }

    /**
     * @return row / colum of candidates in window
     */
    public Dimension getRowColum() {
	return rowColum;
    }

    /**
     * @return DRAW_HORIAONTALLY or DRAW_VERTICALLY
     */
    public int getDirection() {
	return direction;  
    }

    /**
     * @return SERVER_LABEL or CLIENT_LABEL
     */
    public int getLabelOwner() {
	return labelOwner;
    }

    // DRAW event methods

    /**
     * @return index of first candidate
     */
    public int getFirstCandidate() {
	return firstCandidate;
    }

    /**
     * @return index of last candidate
     */
    public int getLastCandidate() {
	return lastCandidate;
    }

    /**
     * @return index of current candidate
     */
    public int getCurrentCandidate() {
	return currentCandidate;
    }

    /**
     * @return candidate list as String array
     */
    public String[] getCandidates() {
	if (candidates == null) {
	    String[] sa = new String[candidateIterator.length];
	    for (int i = 0; i < sa.length; i++) {
		sa[i] = iteratorToString(candidateIterator[i]);
	    }
	    candidates = sa;
	}
	return candidates;
    }

    /**
     * @return candidate list as AttributedCharacterIterator array
     */
    public AttributedCharacterIterator[] getCandidateIterator() {
	if (candidateIterator == null) {
	    AttributedCharacterIterator[] acia =
		new AttributedCharacterIterator[candidates.length];
	    for (int i = 0; i < acia.length; i++) {
		acia[i] = new AttributedString(candidates[i]).getIterator();
	    }
	    candidateIterator = acia;
	}
	return candidateIterator;
    }

    /**
     * @return label list as String array
     */
    public String[] getLabelList() {
	if (labelList == null) {
	    String[] sa = new String[labelIterator.length];
	    for (int i = 0; i < sa.length; i++) {
		sa[i] = iteratorToString(labelIterator[i]);
	    }
	    candidates = sa;
	}
	return labelList;
    }

    /**
     * @return label list as AttributedCharacterIterator array
     */
    public AttributedCharacterIterator[] getLabelIterator() {
	if (labelIterator == null) {
	    AttributedCharacterIterator[] acia =
		new AttributedCharacterIterator[labelList.length];
	    for (int i = 0; i < acia.length; i++) {
		acia[i] = new AttributedString(labelList[i]).getIterator();
	    }
	    labelIterator = acia;
	}
	return labelIterator;
    }

    /**
     * @return window title
     */
    public String getTitle() {
	if (title == null) {
	    title = iteratorToString(titleIterator);
	}
	return title;
    }

    /**
     * @return window title as AttributedCharacterIterator
     */
    public AttributedCharacterIterator getTitleIterator() {
	if (titleIterator == null) {
	    titleIterator = new AttributedString(title).getIterator();
	}
	return titleIterator;
    }

    // PROCESS event methods

    /**
     * @return process event type PROCESS_INDEX or PROCESS_PAGE
     */
    public int getProcessType() {
	return processType;
    }

    /**
     * @return candidate index if process type is PROCESS_INDEX
     *         otherwise (process type is PROCESS_PAGE) returns candidate page
     */
    public int getProcessIndexPage() {
	return indexPage;
    }
}
