/*
 * Program to control ICOM radios
 *
 * Tables and chairs
 */
#include "icom.h"
#include <stdio.h>

/*
 * Keyboard mode commands
 */
struct cmdtable cmd[] = {
	{"agc",		C_AGC,		"AGC fast/slow"},
	{"am",		C_MODEG,	"set mode AM"},
	{"ant",		C_ANT,		"select antenna"},
	{"apf",		C_APF,		"APF on/off"},
	{"atten",	C_ATTEN,	"set attenuator"},
	{"band",	C_BAND,		"set band limits"},
	{"bank",	C_BANK,		"load bank/name"},
	{"cap",		C_PROBE,	"query capability"},
	{"change",	C_CHANGE,	"swap main and sub VFOs"},
	{"chan",	C_CHAN,		"read channel"},
	{"clear",	C_CLEAR,	"clear channel"},
	{"comp",	C_COMP,		"set VFO compensation"},
	{"cw",		C_MODEG,	"set mode CW"},
	{"dial",	C_DIAL,		"set dial tuning step"},
	{"down",	C_DOWN,		"step down"},
	{"duplex",	C_DUPLEX,	"set transmit duplex"},
	{"fm",		C_MODEG,	"set mode FM"},
	{"freq",	C_FREQ,		"set frequency"},
	{"key",		C_KEY,		"send CW message"},
	{"load",	C_LOAD,		"load channel/name"},
	{"lsb",		C_MODEG,	"set mode LSB"},
	{"meter",	C_METER,	"read S meter and squelch"},
	{"misc",	C_MISC,		"miscellaneous control"},
	{"mode",	C_MODE,		"set mode/BFO compensation"},
	{"nb",		C_NB,		"NB on/off"},
	{"offset",	C_OFFSET,	"set VFO offset"},
	{"pad",		C_KEYPAD,	"switch to keypad mode"},
	{"peak",	C_APFC,		"APF control"},
	{"power",	C_POWER,	"power on/off"},
	{"quit",	C_QUIT,		"exit program"},
	{"radio",	C_RADIO,	"select radio"},
	{"rate",	C_RATE,		"set tuning rate"},
	{"restore",	C_RESTORE,	"restore channels"},
	{"rtty",	C_MODEG,	"set mode RTTY"},
	{"save",	C_SAVE,		"save channels"},
	{"say",		C_ANNC,		"announce control"},
	{"scan",	C_SCAN,		"scan control"},
	{"shift",	C_SHIFT,	"IF shift control"},
	{"simplex",	C_SMPLX,	"receive on transmit frequency"},
	{"split",	C_SPLIT,	"toggle split mode with offset"},
	{"squelch",	C_SQUELCH,	"squelch control"},
	{"step",	C_STEP,		"set tuning step"},
	{"trace",	C_DEBUG,	"trace CI-V messages"},	
	{"up",		C_UP,		"step up"},
	{"usb",		C_MODEG,	"set mode USB"},
	{"verbose",	C_VERB,		"set verbose"},
	{"vfo",		C_VFO,		"VFO control"},
	{"vol",		C_VOLUME,	"volume control"},
	{"write",	C_WRITE,	"write channel"},
#ifdef AUDIO
	{"gain",	C_GAIN,		"adjust output level"},
	{"mute",	C_MUTE,		"mute output (toggle)"},
	{"port",	C_PORT,		"select input port"},
#endif /* AUDIO */
	{"\0",		C_FREQX,	"set VFO frequency"}
};

/*
 * Keypad mode commands
 */
struct cmdtable key[] = {
	{"\\",		C_ERASE,	"erase input"},
	{"/", 		C_SMPLX,	"receive on transmit frequency"},
	{"=",		C_CHANGE,	"swap main and sub VFOs"},
	{"[a",		C_UP,		"UP step up"},
	{"[b",		C_DOWN,		"DOWN step down"},
	{"[c",		C_RUP,		"RIGHT tuning rate up"},
	{"[d",		C_RDOWN,	"LEFT tuning rate down"},
	{"]",		C_SPLIT,	"toggle split mode with offset"},
	{"a",		C_ANT,		"select antenna"},
	{"c",		C_CHAN,		"read channel"},
	{"f",		C_MODEG,	"set mode FM"},
	{"i",		C_SHIFT,	"IF shift control"},
	{"l",		C_MODEG,	"set mode LSB"},
	{"m",		C_MODEG,	"set mode AM"},
	{"p",		C_APFC,		"APF control"},
	{"q",		C_KEYBD,	"switch to keyboard mode"},
	{"s",		C_SQUELCH,	"squelch control"},
	{"t",		C_STEP,		"set tuning step"},
	{"u",		C_MODEG,	"set mode USB"},
	{"v",		C_VOLUME,	"volume control"},
	{"w",		C_WRITE,	"write channel"},
	{"x",		C_CLEAR,	"clear channel"},
#ifdef AUDIO
	{"g",		C_GAIN,		"adjust output gain"},
	{"o",		C_MUTE,		"mute output (toggle)"},
	{"p",		C_PORT,		"select input port"},
#endif /* AUDIO */
	{"\0",		C_FREQX,	"set VFO frequency"}
};

/*
 * Trace control subcommands
 */
struct cmdtable dbx[] = {
	{"all",		P_TRACE|P_ERMSG, "trace bus, packets"},
	{"bus",		P_TRACE,	"trace bus"},
	{"none",	0x0,		"trace none"},
	{"packet",	P_ERMSG,	"trace packet errors"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * R8500/R9000 initialization subcommands
 */
struct cmdvec p8500[] = {
	{"agc",		{V_TOGL, S_AGC1, FI}, 	"slow"}, 
	{"apf",		{V_TOGL, S_APF0, FI}, 	"off"}, 
	{"atten",	{V_ATTEN, S_AT00, FI},	"off"},
	{"bank",	{V_SMEM, 0xa0, 0, FI},	"0"},   
	{"dial",	{V_DIAL, 0, FI},	"10 Hz"},
	{"nb",		{V_TOGL, S_NB0, FI}, 	"off"}, 
	{"peak",	{V_WRCTL, S_WRAP, 1, 2 << 4 | 8, FI}, "0"}, 
	{"power",	{V_POWER, S_ON, FI},	"on"}, 
	{"say",		{V_ANNC, S_ALL, FI},	"all"},
	{"scan",	{V_SCAN, 0, FI},	"stop"},
	{"shift",	{V_WRCTL, S_WRSH, 1, 2 << 4 | 8, FI}, "0"}, 
	{"squelch",	{V_WRCTL, S_WRSQ, 0, 0, FI}, "0"}, 
	{"volume",	{V_WRCTL, S_WRAF, 0xa0, 0, FI}, "0"},
	{"\0",		{C_ERROR},		""}
};

/*
 * 775/781 initialization subcommands
 */
struct cmdvec p775[] = {
	{"ant",		{V_SANT, S_ANT1, FI},	"1"},
	{"dial",	{V_DIAL, 0, FI},	"10 Hz"},
	{"split",	{V_SPLIT, S_OFF, FI},	"nosplit"},
	{"vfo",		{V_SVFO, S_DWOFF, FI},	"nowatch"},
	{"\0",		{C_ERROR},		""}
};

/*
 * "verbose" subcommands
 */
struct cmdtable verbx[] = {
	{"on",		P_VERB,		"verbose"},
	{"off",		0,		"terse"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * "vfo" subcommands
 */
struct cmdtable vfo[] = {
	{"a",		S_VFOA,		"select vfo a"},
	{"b",		S_VFOB,		"select vfo b"},
	{"btoa",	S_BTOA,		"vfo a <- vfo b"},
	{"equal",	S_EQUAL,	"main -> sub"},
	{"mainband",	S_MBAND,	"access main band"},
	{"nowatch",	S_DWOFF,	"dual watch off"},
	{"subband",	S_SBAND,	"access sub band"},
	{"swap",	S_XCHNG,	"main <-> sub"},
	{"watch",	S_DWON,		"dual watch on"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * "scan" subcommands
 */
struct cmdtable scan[] = {
	{"10",		S_DF10,		"delta-f 10 kHz"},
	{"2.5",		S_DF2,		"delta-f 2.5 kHz"},
	{"20",		S_DF20,		"delta-f 20 kHz"},
	{"5",		S_DF5,		"delta-f 5 kHz"},
	{"50",		S_DF50,		"delta-f 50 kHz"},
	{"deltaf",	S_DFST,		"delta-f scan"},
	{"fdeltaf",	S_FDST,		"fine delta-f scan"},
	{"fix",		S_FIX,		"fix center frequency"},
	{"fprog",	S_FPST,		"fine program scan"},
	{"mem",		S_MSST,		"memory scan"},
	{"memauto",	S_AMST,		"auto write scan"},
	{"memchan",	S_MEMN,		"memory channel scan number"},
	{"memnum",	S_SNST,		"memory scan number"},
	{"memsel",	S_SMST,		"selected mode memory scan"},
	{"noresume",	S_SROFF,	"scan resume off"},
	{"notresume",	S_SRNOT,	"scan resume never"},
	{"novsc",	S_VSOFF,	"VSC off"},
	{"prog",	S_PSST,		"program scan"},
	{"resumea",	S_SRA,		"scan resume a (delay)"},
	{"resumeb",	S_SRB,		"scan resume b"},
	{"skip",	S_DSBM,		"disable memory channel"},
	{"start",	S_START,	"scan"},
	{"stop",	S_OFF,		"stop scan"},
	{"pstop",	S_PXST,		"priority scan"},
	{"unfix",	S_UNFIX,	"unfix center frequency"},
	{"unskip",	S_ENBM,		"enable memory channel"},
	{"vsc",		S_VSON,		"VSC on"},
	{"\0",		C_ERROR,	"invalid argument"}
};
 
/*
 * "split" subcommands
 */
struct cmdtable split[] = {
	{"+",		S_DUPP,		"select +duplex"},
	{"-",		S_DUPM,		"select -duplex"},
	{"cancel",	S_DUPOF,	"cancel duplex"},
	{"nosplit",	S_OFF,		"split off"},
	{"split",	S_ON,		"split on"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * "atten" subcommands
 */
struct cmdtable atten[] = {
	{"0",		S_AT00,		"off"},
	{"10",		S_AT10,		"10 dB"},
	{"20",		S_AT20,		"20 dB"},
	{"30",		S_AT30,		"30 dB"},
	{"off",		S_AT00,		"off"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * "ant" subcommands
 */
struct cmdtable ant[] = {
	{"1",		S_ANT1,		"antenna 1"},
	{"2",		S_ANT2,		"antenna 2"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * "say" subcommands
 */
struct cmdtable annc[] = {
	{"all",		S_ALL,		"announce all"},
	{"freq",	S_FREQ,		"announce freq"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * "power" subcommands
 */
struct cmdtable power[] = {
	{"off",		S_OFF,		"power off"},
	{"on",		S_ON,		"power on"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * "agc" subcommands
 */
struct cmdtable agc[] = {
	{"slow",	S_AGC0,		"AGC slow"},
	{"fast",	S_AGC1,		"AGC fast"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * "nb" subcommands
 */
struct cmdtable nb[] = {
	{"off",		S_NB0,		"NB off"},
	{"on",		S_NB1,		"NB on"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * "apf" subcommands
 */
struct cmdtable apf[] = {
	{"off",		S_APF0,		"APF off"},
	{"on",		S_APF1,		"APF on"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * "dial" subcommands
 */
struct cmdtable diala[] = {
	{".01",		0,		"10 Hz"},
	{".05",		1,		"50 Hz"},
	{".1",		2,		"100 Hz"},
	{"1",		3,		"1 kHz"},
	{"2.5",		4,		"2.5 kHz"},
	{"5",		5,		"5 kHz"},
	{"9",		6,		"9 kHz"},
	{"10",		7,		"10 kHz"},
	{"12.5",	8,		"12.5 kHz"},
	{"20",		9,		"20 kHz"},
	{"25",		10,		"25 kHz"},
	{"100",		11,		"100 kHz"},
	{"1000",	12,		"1 MHz"},
	{"\0",		C_ERROR,	"invalid argument"}
};
struct cmdtable dialb[] = {
	{".01",		0,		"10 Hz"},
	{"1",		1,		"1 kHz"},
	{"2",		2,		"2 kHz"},
	{"3",		3,		"3 kHz"},
	{"4",		4,		"4 kHz"},
	{"5",		5,		"5 kHz"},
	{"6",		6,		"6 kHz"},
	{"7",		7,		"7 kHz"},
	{"8",		8,		"8 kHz"},
	{"9",		9,		"9 kHz"},
	{"10",		10,		"10 kHz"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * "misc" subcommands
 */
struct cmdtable misc[] = {
	{"ctss",	S_CTSS,		"read CTSS"},
	{"dcs",		S_DCS,		"read DCS"},
	{"dtmf",	S_DTMF,		"read DTMF"},
	{"freq",	S_NXFM,		"next frequency"},
	{"id",		S_RDID,		"read ID"},
	{"local",	S_LCL,		"local control"},
	{"nosearch",	S_SMOFF,	"disable search"},
	{"nospeaker",	S_SPOFF,	"mute speaker"},
	{"notape",	S_TPOFF,	"disable tape recorder"},
	{"nowindow",	S_5OFF,		"disable search window"},
	{"opto",	S_OPTO,		"read OPTO"},
	{"remote",	S_RMT,		"remote control"},
	{"search",	S_SMON,		"enable search"},
	{"speaker",	S_SPON,		"unmute speaker"},
	{"tape",	S_TPON,		"enable tape recorder"},
	{"window",	S_5ON,		"enable search window"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * Radio identifier decode
 */
struct cmdtable ident[] = {
	{"1271",	0x24,		"1271 UHF Transceiver"},
	{"1275",	0x18,		"1275 UHF Transceiver"},
	{"271",		0x20,		"271 VHF Transceiver"},
	{"275",		0x10,		"275 VHF Transceiver"},
	{"375",		0x12,		"375 VHF Transceiver"},
	{"471",		0x22,		"471 UHF Transceiver"},
	{"475",		0x14,		"475 UHF Transceiver"},
	{"575",		0x16,		"575 VHF Transceiver"},
	{"575",		0x28,		"725 HF Transceiver"},
	{"706",		0x4e,		"706 HF Transceiver"},
	{"726",		0x30,		"726 HF Transceiver"},
	{"735",		0x04,		"735 HF Transceiver"},
	{"751",		0x1c,		"751 HF Transceiver"},
	{"761",		0x1e,		"761 HF Transceiver"},
	{"765",		0x2c,		"765 HF Transceiver"},
	{"775",		0x46,		"775 HF Transceiver"},
	{"781",		0x26,		"781 HF Transceiver"},
	{"970",		0x2e,		"970 HF Transceiver"},
	{"R7000",	0x08,		"R7000 VHF/UHF Receiver"},
	{"R71",		0x1A,		"R71 HF Receiver"},
	{"R7100",	0x34,		"R7100 VHF/UHF Receiver"},
	{"R72",		0x32,		"R72 HF Receiver"},
	{"R8500",	0x4a,		"R8500 HF/VHF/UHF Receiver"},
	{"R9000",	0x2a,		"R9000 VHF/UHF Receiver"},
	{"\0",		C_ERROR,	"unknown radio"}
};

/*
 * Broadband HF/VHF/UHF receiver mode decode (R7000)
 *
 * Note that panel FM mode is really wideband FM and panel FMn mode is
 * really FM mode on other radios. Clearing a channel changes the mode
 * to 0xff, but does not change the frequency. Reactivate the channel
 * by loading valid mode.
 */
static struct cmdtable mode2[] = {
	{"AM",		0x0002,		"AM"},
	{"m",		0x0002,		"AM (keypad)"},
	{"FMn",		0x8205,		"FM"},
	{"WFM",		0x0005,		"WFM"},
	{"f",		0x8205,		"FM (keypad)"},
	{"USB",		0x8005,		"USB"},
	{"u",		0x8005,		"USB (keypad)"},
	{"\0",		C_ERROR,	"invalid mode"}
};

/*
 * Broadband MF/HF/VHF/UHF receiver mode decode (R8500/R9000)
 */
static struct cmdtable mode3[] = {
	{"LSB",		0x0001,		"LSB"},
	{"l",		0x0001,		"LSB (keypad)"},
	{"USB",		0x8101,		"USB"},
	{"u",		0x8101,		"USB (keypad)"},
	{"AM",		0x8202,		"AM"},
	{"m",		0x8202,		"AM (keypad)"},
	{"AMn",		0x8302,		"AM narrow"},
	{"AMw",		0x8102,		"AM wide"},
	{"CW",		0x8103,		"CW"},
	{"CWn",		0x8203,		"CW narrow"},
	{"FM",		0x8105,		"FM"},
	{"f",		0x8105,		"FM (keypad)"},
	{"FMn",		0x8205,		"FM narrow"},
	{"WFM",		0x8106,		"WFM"},
	{"\0",		C_ERROR,	"invalid mode"}
};

/*
 * Narrowband MF/HF/VHF/UHF transceiver and receiver mode decode
 */
static struct cmdtable mode1[] = {
	{"AM",		0x0002,		"AM"},
	{"AMn",		0x8202,		"AM narrow"},
	{"AMw",		0x8102,		"AM wide"},
	{"m",		0x0002,		"AM (keypad)"},
	{"CW",		0x0003,		"CW"},
	{"CWn",		0x8203,		"CW narrow"},
	{"CWw",		0x8103,		"CW wide"},
	{"FM",		0x0005,		"FM"},
	{"FMn",		0x8205,		"FM narrow"},
	{"FMw",		0x8105,		"FM wide"},
	{"f",		0x0005,		"FM (keypad)"},
	{"LSB",		0x0000,		"LSB"},
	{"LSBn",	0x8200,		"LSB narrow"},
	{"LSBw",	0x8100,		"LSB wide"},
	{"l",		0x0000,		"LSB (keypad)"},
	{"RTTY",	0x0004,		"RTTY"},
	{"RTTYn",	0x8204,		"RTTY narrow"},
	{"RTTYw",	0x8104,		"RTTY wide"},
	{"USB",		0x0001,		"USB"},
	{"USBn",	0x8201,		"USB narrow"},
	{"USBw",	0x8101,		"USB wide"},
	{"u",		0x0001,		"USB (keypad)"},
	{"\0",		C_ERROR,	"invalid mode"}
};

/*
 * Radio control initialization
 */
struct namestruct name[] = {
	{"1271",  0x24,	32,  NULL,  mode1, 0,	NULL},
	{"1275",  0x18,	32,  NULL,  mode1, 0,	NULL},
	{"271",	  0x20,	32,  NULL,  mode1, 0,	NULL},
	{"275",	  0x10,	101, NULL,  mode1, 0,	NULL},
	{"375",	  0x12,	101, NULL,  mode1, 0,	NULL},
	{"471",	  0x22,	32,  NULL,  mode1, 0,	NULL},
	{"475",	  0x14,	101, NULL,  mode1, 0,	NULL},
	{"575",	  0x16,	101, NULL,  mode1, 0,	NULL},
	{"706",   0x4e, 10,  NULL,  mode1, 0,   NULL},
 	{"725",	  0x28,	26,  NULL,  mode1, 0,	NULL},
	{"726",	  0x30,	26,  NULL,  mode1, 0,	NULL},
	{"735",	  0x04,	12,  NULL,  mode1, F_735, NULL},
	{"751",	  0x1c,	32,  NULL,  mode1, 0,	NULL},
	{"761",	  0x1e,	32,  NULL,  mode1, 0,	NULL},
	{"765",	  0x2c,	32,  NULL,  mode1, 0,	NULL},
	{"775",	  0x46,	101, p775,  mode1, F_DIAL, NULL},
	{"781",	  0x26,	101, p775,  mode1, 0,	NULL},
	{"970",	  0x2e,	101, NULL,  mode1, 0,	NULL},
	{"R7000", 0x08,	99,  NULL,  mode2, 0,	NULL},
	{"R71",	  0x1A,	32,  NULL,  mode1, 0,	NULL},
	{"R7100", 0x34,	99,  NULL,  mode2, 0,	NULL},
	{"R72",	  0x32,	101, NULL,  mode1, 0,	NULL},
	{"R8500", 0x4a,	800, p8500, mode3, F_CHZ, NULL},
	{"R9000", 0x2a,	99,  p8500, mode3, F_CHZ, NULL},
	{"\0",	  0x0,	0,   0,    0,     0,	 NULL}  /* end */
};
