/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: Song.h,v 1.20 2004/06/30 16:13:13 comix Exp $
 *
 */

#ifndef SONG_H
#define SONG_H


#include <string>
using std::string;

#include <vector>
using std::vector;

#include <map>
using std::map;

#include "Object.h"
#include "Sample.h"
#include "LocalFileMng.h"
#include "fx/LadspaFX.h"


class Note;
class Instrument;
class Sequence;
class Pattern;
class Song;


class PatternGroupSequence : public Object
{
	public:
		PatternGroupSequence();
		~PatternGroupSequence();

	private:

};


/// Sequence List
class SequenceList : public Object {
	public:
		SequenceList();
		~SequenceList();

		void add(Sequence* newSequence);
		Sequence* get( int nPos );
		unsigned int getSize();

		SequenceList* copy();

	private:
		vector<Sequence*> list;
};




/// Pattern List
class PatternList : public Object {
	private:
		vector<Pattern*> list;

	public:
		PatternList();
		~PatternList();

		void add( Pattern* newPattern );
		Pattern* get( int nPos );
		unsigned int getSize();
		void clear();

		void replace( Pattern* newPattern, unsigned int pos );

		void del(Pattern *pattern);
		void del(uint index);
};




/// Instrument List
class InstrumentList : public Object {
	private:
		vector<Instrument*> list;
		map<Instrument*, unsigned int> posmap;
	public:
		InstrumentList();
		~InstrumentList();

		void add(Instrument* newPattern);
		Instrument* get(unsigned int pos);
		int getPos(Instrument* inst);
		unsigned int getSize();
};





/// A note...
class Note : public Object {
	public:
		float m_fSamplePosition;

		Note(uint nPosition,
			float fVelocity,
			float fPan_L,
			float fPan_R,
			int nLength,
			float fPitch = 0.0
		);

		~Note();
		
		Note* copy();

		unsigned getPosition(){	return m_nPosition;	}
		void setPosition(unsigned nPos) { 	m_nPosition = nPos;	}

		float getPan_L(){	return m_fPan_L;	}
		void setPan_L( float val );

		float getPan_R(){	return m_fPan_R;	}
		void setPan_R( float val );

		void setInstrument(Instrument* pInstrument);
		Instrument* getInstrument();

		float getVelocity(){	return m_fVelocity;	}
		void setVelocity( float val );

		uint getHumanizeDelay() {	return m_nHumanizeDelay;	}
		void setHumanizeDelay( unsigned nVal ) {	m_nHumanizeDelay = nVal;	}

		float getPitch() {	return m_fPitch;	}
		void setPitch( float nPitch ) {	m_fPitch = nPitch;	}

		void setLength(int nLength) {	m_nLength = nLength;	}
		int getLength() {	return m_nLength;	}

	private:
		
		unsigned int m_nPosition;	///< Note position inside the pattern


		float m_fVelocity;		///< Velocity (intensity) of the note
		float m_fPan_L;			///< Pan of the note (left volume)
		float m_fPan_R;			///< Pan of the note (right volume)
		int m_nLength;
		float m_fPitch;
		Instrument* m_pInstrument;

		uint m_nHumanizeDelay;	///< Used in "humanize" function

};



/// A layer...
class InstrumentLayer : public Object 
{
	public:
		InstrumentLayer( Sample *pSample );
		~InstrumentLayer();
	
		Sample* getSample() {	return m_pSample;	}
		void setSample( Sample *pSample );
		
		float getStartVelocity() {	return m_fStartVelocity;	}
		void setStartVelocity( float fVel );
		
		float getEndVelocity() {	return m_fEndVelocity;	}
		void setEndVelocity( float fVel );
		
		float getGain() {	return m_fGain;	}
		void setGain( float fGain );
		
		float getPitch() {	return m_fPitch;	}
		void setPitch( float fPitch );
		
	private:
		Sample *m_pSample;
		
		float m_fStartVelocity;		///< Start velocity
		float m_fEndVelocity;		///< End velocity
		
		float m_fPitch;
		float m_fGain;
};



/// Instrument class
class Instrument : public Object
{
	public:
		Instrument( string sId = "", string sName = "", string sAuthor = "", float fVolume = 0, bool bMuted = false, float fPan_L = 1.0, float fPan_R = 1.0, string sDrumkitName = "" );
		~Instrument();

		float getVolume(){	return m_fVolume;	}
		void setVolume(float volume);

		void setId(string id) {	m_sId = id;	}
		string getId(){	return m_sId;	}

		string getName(){	return m_sName;	}
		void setName(string name) {	m_sName = name;	}

		string getAuthor(){	return m_sAuthor;	}

		bool isMuted() {	return m_bMuted;	}
		void setMuted(bool muted) {	m_bMuted = muted;	}

		void setPeak_L( float fPeak );
		float getPeak_L() {	return m_fPeak_L;	}

		void setPeak_R( float fPeak );
		float getPeak_R() {	return m_fPeak_R;	}

		float getPan_L(){	return m_fPan_L;	}
		void setPan_L( float val );

		float getPan_R(){	return m_fPan_R;	}
		void setPan_R( float val );

		// LADSPA FX
		float getFXLevel(uint nFX){	return m_fFXLevel[nFX];	}
		void setFXLevel( uint nFX, float value ) {	m_fFXLevel[nFX] = value;	}

		void setDrumkitName( string drumkitName ) {	m_sDrumkitName = drumkitName;	}
		string getDrumkitName() {	return m_sDrumkitName;	}

		// FIXME: passare il riferimento
//		vector<Instrument*> getExcludeVect() {	return excludeVect;	}

		/** exclude this instruments */
		vector<Instrument*> m_excludeVect;

		vector<int> m_excludeVectId;
		
		InstrumentLayer* getLayer( int nLayer );
		void setLayer( InstrumentLayer* pLayer, uint nLayer );

		bool isActive() {	return m_bActive;	}
		void setActive( bool bActive ) {	m_bActive = bActive;	}
		
	private:
		string m_sId;		///< ID of the instrument
		float m_fVolume;	///< Volume of the instrument
		string m_sName;		///< Instrument name
		string m_sAuthor;	///< author of the Instrument
		bool m_bMuted;
		float m_fPeak_L;		///< current peak value (left)
		float m_fPeak_R;	///< current peak value (right)
		float m_fPan_L;		///< Pan of the instrument (left)
		float m_fPan_R;		///< Pan of the instrument (right)
		string m_sDrumkitName;	///< Drumkit name
		float m_fFXLevel[MAX_FX];	///< Ladspa FX level
		InstrumentLayer* m_layers[MAX_LAYERS];
		bool m_bActive;		///< is the instrument active?
};





/// Sequence of notes
class Sequence : public Object{
	public:
		map <int, Note*> m_noteList; 
		
		Sequence();
		~Sequence();

		Sequence* copy();
};





/// Pattern (a container of sequences)
class Pattern : public Object
{
	public:
		Pattern( string name, uint nPatternSize = MAX_NOTES );
		~Pattern();

		void setName(string sName);
		string getName();

		SequenceList* getSequenceList(){	return m_pSequenceList;	}
		void setSequenceList( SequenceList *pList ){	m_pSequenceList = pList;	}

		static Pattern* getEmptyPattern();
		Pattern* copy();

		void setSize(uint nSize) {	m_nSize = nSize;	}
		uint getSize() {	return m_nSize;	}
	
	private:
		string m_sName;
		uint m_nSize;
		SequenceList *m_pSequenceList;

};






/// Song class.
class Song : public Object{
	public:
		enum SongMode {
			PATTERN_MODE,
			SONG_MODE
		};
	
		Song(string id, string name, string author, float bpm, float volume);
		~Song();

		string getId(){	return m_sId;	}

		float getBpm(){	return m_fBpm;	}
		void setBpm(float fBpm) {	m_fBpm = fBpm;	}

		void setVolume(float fVolume) {	m_fVolume = fVolume;	}
		float getVolume() {	return m_fVolume;	}

		void setMetronomeVolume(float fVolume) {	m_fMetronomeVolume = fVolume;	}
		float getMetronomeVolume() {	return m_fMetronomeVolume;	}

		unsigned int getResolution(){	return m_nResolution;	}

		string getName(){	return m_sName;	}
		void setName(string sName) {	m_sName = sName;	}

		void setAuthor(string sAuthor) {	m_sAuthor = sAuthor;	}
		string getAuthor(){	return m_sAuthor;	}

		PatternList* getPatternList(){	return m_pPatternList;	}
		void setPatternList( PatternList *pList ){	m_pPatternList = pList;	}

		vector<PatternList*>* getPatternGroupVector(){	return m_pPatternSequence;	}
		void setPatternGroupVector( vector<PatternList*>* pVect ){	m_pPatternSequence = pVect;	}

		static Song* load(string sFilename);
		void save(string sFilename);

		InstrumentList* getInstrumentList(){	return m_pInstrumentList;	}
		void setInstrumentList( InstrumentList *pList ){	m_pInstrumentList = pList;	}

		static Song* getEmptySong();

		void setNotes(string sNotes) {	m_sNotes = sNotes;	}
		string getNotes() {	return m_sNotes;	}

		string getFilename() {	return m_sFilename;	}
		void setFilename(string sFilename) {	m_sFilename = sFilename;	}

		bool isModified() {	return m_bIsModified;	}
		void setModified(bool bIsModified) {	m_bIsModified = bIsModified;	}

		bool isLoopEnabled() {	return m_bIsLoopEnabled;	}
		void setLoopEnabled( bool bIsLoopEnabled ) {	m_bIsLoopEnabled = bIsLoopEnabled;	}

		float getHumanizeTimeValue() {	return m_fHumanizeTimeValue;	}
		void setHumanizeTimeValue(float fValue) {	m_fHumanizeTimeValue = fValue;	}

		float getHumanizeVelocityValue() {	return m_fHumanizeVelocityValue;	 }
		void setHumanizeVelocityValue(float fValue) { m_fHumanizeVelocityValue = fValue;	}

		float getSwingFactor() {	return m_fSwingFactor;	}
		void setSwingFactor(float fFactor);

		SongMode getMode() {	return m_songMode;	}
		void setMode( SongMode newMode ) {	m_songMode = newMode;	}

		bool isHumanizeTimeEnabled() {	return m_bIsHumanizeTimeEnabled;	}
		void setHumanizeTimeEnabled( bool bEnabled ) {	m_bIsHumanizeTimeEnabled = bEnabled;	}

		bool isHumanizeVelocityEnabled() {	return m_bIsHumanizeVelocityEnabled;	}
		void setHumanizeVelocityEnabled( bool bEnabled ) {	m_bIsHumanizeVelocityEnabled = bEnabled;	}

		bool isSwingEnabled() {	return m_bIsSwingEnabled;	}
		void setSwingEnabled( bool bEnabled ) {	m_bIsSwingEnabled = bEnabled;	}

		LadspaFX* getLadspaFX(uint nFX) {	return m_pLadspaFX[nFX];	}
		void setLadspaFX( uint nFX, LadspaFX* pLadspaFX ) {	m_pLadspaFX[nFX] = pLadspaFX;	}

	private:
		string m_sId;				///< ID of the song
		float m_fBpm;				///< Beats per minute
		unsigned m_nResolution;		///< Resolution of the song (number of ticks per quarter)
		float m_fVolume;			///< volume of the song (0.0..1.0)
		float m_fMetronomeVolume;	///< Metronome volume
		string m_sName;			///< song name
		string m_sAuthor;			///< author of the song
		string m_sNotes;
		PatternList *m_pPatternList;	///< Pattern list
		vector<PatternList*>* m_pPatternSequence;	///< Sequence of pattern groups
		InstrumentList *m_pInstrumentList;	///< Instrument list
		string m_sFilename;
		bool m_bIsModified;
		bool m_bIsLoopEnabled;
		bool m_bIsHumanizeTimeEnabled;
		float m_fHumanizeTimeValue;
		bool m_bIsHumanizeVelocityEnabled;
		float m_fHumanizeVelocityValue;
		bool m_bIsSwingEnabled;
		float m_fSwingFactor;

		SongMode m_songMode;

		LadspaFX* m_pLadspaFX[MAX_FX];
};





/// Drumkit info
class DrumkitInfo : public Object {
	public:
		DrumkitInfo();
		~DrumkitInfo();

		InstrumentList *getInstrumentList() {	return instrumentList;	}
		void setInstrumentList( InstrumentList* instr ) {	this->instrumentList = instr;	}

		void setName( string name ) {	this->name = name;	}
		string getName() {	return name;	}

		void setAuthor( string author ) {	this->author = author;	}
		string getAuthor() {	return author;	}

		void setInfo( string info ) {	this->info = info;	}
		string getInfo() {	return info;	}

		void dump();

	private:
		InstrumentList *instrumentList;
		string name;
		string author;
		string info;
};


#endif


