/* ptal-connect -- PTAL command-line stream connection client */

/* Copyright (C) 2000-2003 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * In addition, as a special exception, Hewlett-Packard Company
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included LICENSE.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */

/* Original author: David Paschal */

#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include "ptal.h"

#define LEN_BUFFER 4096
static char uel1[]="\033%-12345X";
static char uel2[]="\033E\033%-12345X";

#define STDIN	0
#define STDOUT	1

int main(int argc,char **argv) {
	char *argv0=*argv,*serviceName=0;
	int retcode=1,dump=0,serviceType=PTAL_STYPE_NONE,socketID=0;
	char buffer[LEN_BUFFER];
	int wsize=LEN_BUFFER,rsize=LEN_BUFFER;
	int maxForwardDatalen=wsize,maxReverseDatalen=rsize;
	ptalDevice_t dev=0;
	ptalChannel_t chan=0;
	int r,w,n,fd,ateof=0,quiet=0,sendUel=0;
	int retryCount=PTAL_DEFAULT_RETRY_COUNT;
	int retryDelay=PTAL_DEFAULT_RETRY_DELAY;
	fd_set rset;
	struct timeval timeout,*pTimeout=0;

	timeout.tv_sec=0;
	timeout.tv_usec=0;

	ptalInit();

	while (42) {
		argc--; argv++; if (argc<=0) break;

		if (**argv!='-') {
			/* if (dev) goto syntaxError; */

			dev=ptalDeviceOpen(*argv);
			if (!dev) {
				PTAL_LOG_ERROR("%s: Unable to open "
					"device \"%s\"!\n\n",argv0,*argv);
				goto syntaxError;
			}

		} else if (!strcmp(*argv,"-eoftimeout")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			timeout.tv_sec=atoi(*argv);

		} else if (!strcmp(*argv,"-fwdlen")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			maxForwardDatalen=atoi(*argv);

		} else if (!strcmp(*argv,"-revlen")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			maxReverseDatalen=atoi(*argv);

		} else if (!strcmp(*argv,"-print")) {
			serviceType=PTAL_STYPE_PRINT;

		} else if (!strcmp(*argv,"-scan")) {
			serviceType=PTAL_STYPE_SCAN;

		} else if (!strcmp(*argv,"-socket")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			serviceType=PTAL_STYPE_GENERIC;
			socketID=atoi(*argv);

		} else if (!strcmp(*argv,"-service")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			serviceType=PTAL_STYPE_GENERIC;
			serviceName=*argv;

		} else if (!strcmp(*argv,"-retrycount")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			retryCount=atoi(*argv);

		} else if (!strcmp(*argv,"-noretry")) {
			retryCount=0;

		} else if (!strcmp(*argv,"-infretry")) {
			retryCount=-1;

		} else if (!strcmp(*argv,"-retrydelay")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			retryDelay=atoi(*argv);

		} else if (!strcmp(*argv,"-quiet")) {
			quiet=1;

		} else if (!strcmp(*argv,"-uel")) {
			sendUel=1;

		} else if (!strcmp(*argv,"-nouel")) {
			sendUel=0;

		} else if (!strcmp(*argv,"-dump")) {
			dump=1;

		} else {
syntaxError:
			PTAL_LOG_ERROR("Syntax: ptal-connect "
				"[<devname>] <action> [<options>]\n"
				"Where <devname> may be one of:\n");
			ptalDeviceEnumerate(0,
				ptalDeviceEnumeratePrintCallback,0);
			PTAL_LOG_ERROR("<action> may be one of:\n"
				"\t-print   (default for ptal-print)\n"
				"\t-scan\n"
				"\t-socket <socketID>\n"
				"\t-service <serviceName>\n"
				"<options> is zero or more of:\n"
				"\t-eoftimeout <seconds>\n"
				"\t-fwdlen <maxForwardDatalen>\n"
				"\t-revlen <maxReverseDatalen>\n"
				"\t-retrycount <seconds>\n"
				"\t-noretry\n"
				"\t-infretry\n"
				"\t-retrydelay <seconds>\n"
				"\t-quiet\n"
				"\t-uel, -nouel\n"
				"\t-dump\n"
				);
			goto abort;
		}
	}

	if (!dev) {
		dev=ptalDeviceOpen(0);
		if (!dev) {
			PTAL_LOG_ERROR("%s: No default device defined!\n\n",
				argv0);
			goto syntaxError;
		}
	}
	chan=ptalChannelAllocate(dev);
	if (!chan) {
		goto abort;
	}

	if (serviceType==PTAL_STYPE_NONE) {
		if (!strstr(argv0,"ptal-print")) goto syntaxError;
		serviceType=PTAL_STYPE_PRINT;
	}
	ptalChannelSetRemoteService(chan,serviceType,socketID,serviceName);

	ptalChannelSetPacketSizes(chan,maxForwardDatalen,maxReverseDatalen);
	ptalChannelSetErrorHandling(chan,retryCount,retryDelay);

	if (!quiet) PTAL_LOG_ERROR("%s: connecting to device \"%s\"...  ",
		argv0,ptalDeviceGetName(dev));
	while (ptalChannelOpen(chan)==PTAL_ERROR) {
		if (retryCount<0) {
			sleep(retryDelay);
			continue;
		}
		if (!quiet) {
			PTAL_LOG_ERROR("failed!\n");
		} else {
			PTAL_LOG_ERROR("%s: failed connecting to "
				"device \"%s\"!\n",
				argv0,ptalDeviceGetName(dev));
		}
		goto abort;
	}
	if (!quiet) PTAL_LOG_ERROR("successful.\n");

	ptalChannelAdjustPacketSizes(chan,&wsize,&rsize);

	if (dump) {
		ptalDump(0);
	}

	if (sendUel) {
		r=strlen(uel1);
		w=ptalChannelWrite(chan,uel1,r);
		if (w!=r) {
			PTAL_LOG_WARN("%s: ptalChannelWrite(uel1) "
				"returns %d, expected=%d!\n",
				argv0,w,r);
		}
	}

	while (42) {
		n=0;
		FD_ZERO(&rset);

		if (!ateof) ptalFdPrepareForSelect(STDIN,&n,&rset,0,0);
		ptalChannelPrepareForSelect(chan,&fd,&n,&rset,0,0);

		r=select(n,&rset,0,0,pTimeout);
		if (r<=0) {
			PTAL_LOG_DEBUG("%s: select returns %d.\n",argv0,r);
			break;
		}

		if (FD_ISSET(STDIN,&rset)) {
			r=read(STDIN,buffer,wsize);
			if (r<=0) {
				PTAL_LOG_DEBUG("%s: read returns %d.\n",
					argv0,r);
				if (!timeout.tv_sec) break;
				ateof=1;
				FD_CLR(STDIN,&rset);
				pTimeout=&timeout;
			} else {
				w=ptalChannelWrite(chan,buffer,r);
				if (w!=r) {
					PTAL_LOG_WARN("%s: ptalChannelWrite "
						"returns %d, expected=%d!\n",
						argv0,w,r);
					break;
				}
			}
		}

		if (FD_ISSET(fd,&rset)) {
			r=ptalChannelRead(chan,buffer,rsize);
			if (r<=0) {
				PTAL_LOG_DEBUG("%s: ptalChannelRead "
					"returns %d.\n",argv0,r);
				break;
			}
			w=write(STDOUT,buffer,r);
			if (w!=r) {
				PTAL_LOG_WARN("%s: write returns %d, "
					"expected=%d!\n",argv0,w,r);
				break;
			}
		}
	}

	if (sendUel) {
		r=strlen(uel2);
		w=ptalChannelWrite(chan,uel2,r);
		if (w!=r) {
			PTAL_LOG_WARN("%s: ptalChannelWrite(uel2) "
				"returns %d, expected=%d!\n",
				argv0,w,r);
		}
	}

	retcode=0;
abort:
	ptalDone();
	return retcode;
}
