/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

/** Arbol simple recursivo.
 * <pre>
 * El rbol se organiza as:
 *
 *(parent de cur y root de todo el rbol)
 * |  (cur, iterador aqu)
 * |   |  
 * v   v  
 * O===O===O <-- (child)
 *     |
 *     O <-- (next)
 *     |
 *     O===O
 *     |   |
 *     O   O
 *     |
 *     O===O===O
 *     ^
 *     |
 *  (branch)
 * </pre>
 * 
 * @file    JTree.h.
 * @author  Juan Carlos Seijo Prez
 * @date    26/10/2003
 * @version 0.0.1 - 26/10/2003 - Primera versin.
 */

#ifndef _JTREE_INCLUDED
#define _JTREE_INCLUDED

#include <JLib/Util/JTypes.h>
#include <JLib/Util/JObject.h>
#include <list>

/** Arbol simple recursivo. El tipo del rbol debe tener un constructor por defecto.
 */
template <typename T>
class JTree : public JObject
{
 public:
	/** Nodo bsico del rbol.
	 */
	class Node
	{
	public:
		T data;                   /**< Datos del nodo */
		Node *prev;               /**< Nodo anterior del arbol */
		Node *next;               /**< Nodo siguiente del arbol */
		Node *child;              /**< Nodo hijo del arbol */
		Node *parent;             /**< Nodo padre del arbol */

		/** Crea un nuevo nodo de arbol. El tipo T requiere constructor por defecto.
		 */
		Node() : prev(0), next(0), parent(0), child(0)
		{}

		/** Crea un nuevo nodo de arbol.
		 * @param  _data Datos de este nodo.
		 * @param  _prev Nodo anterior.
		 * @param  _next Nodo posterior.
		 * @param  _parent Nodo padre.
		 */
		Node(T _data, Node *_prev = 0, Node *_next = 0, Node *_parent = 0, Node *_gPrev = 0, Node *_gNext = 0)
		: data(_data), prev(_prev), next(_next), parent(_parent), child(0)
		{}
	};
	
	Node* root;                           /**< Nodo raz del rbol. Es el padre del primer nodo. */
	Node* end;                            /**< Nodo terminador del rbol. Es el siguiente del ltimo de cada rama. */
	std::list<Node *> nodes;              /**< Todos los nodos en forma de lista. */
	typename std::list<Node *>::iterator itG; /**< Iterador global de nodos. */
		
	/** Crea el rbol vaco.
	 */
	JTree<T>()
	{
		root = new Node;
		root->parent = root;
		nodes.push_back(root);
	}

	/** Destruye el rbol. Borra los nodos creados (PERO NO su contenido, en caso de ser punteros)
	 */
	~JTree<T>()
	{
		nodes.clear();
	}

  /** Iterador del rbol.
   */
  class Iterator
  {
  public:
		JTree *tree;                                /**< rbol al que pertenece este iterador. */
		Node *node;                                 /**< Nodo actual. */
    
    /** Crea un iterador a partir de un rbol.
     */
    Iterator(JTree &t)
    {
			tree = &t;
			node = *(tree->nodes.begin());
			if (node->child)
			{
				// Salta root para dejarlo en el primer elemento, si no la nica operacin
				// vlida ser AddBranchGo()
				node = node->child;
			}
    }

    /** Asigna los datos del nodo actual.
     * @param data Nuevos datos.
     */
    void Data(T data)
    {
      node->data = data;
    }

    /** Devuelve los datos del nodo actual.
     * @return Datos del nodo actual.
     */
    T& Data()
    {
      return node->data;
    }

    /** Va al nodo siguiente.
     * @return <b>true</b> si el nodo exista, <b>false</b> si no.
     */
    bool Next()
    {
      if (node->next)
      {
        node = node->next;
        return true;
      }

      return false;
    }
    
    /** Va al nodo anterior.
     * @return <b>true</b> si el nodo exista, <b>false</b> si no.
     */
    bool Prev()
    {
      if (node->prev)
      {
        node = node->prev;
        return true;
      }

      return false;
    }
    
    /** Va al nodo padre.
     * @return <b>true</b> si el nodo exista, <b>false</b> si no.
     */
    bool Parent()
    {
      if (node->parent != tree->root)
      {
        node = node->parent;
        return true;
      }

      return false;
    }
    
    /** Va al nodo hijo.
     * @return <b>true</b> si el nodo exista, <b>false</b> si no.
     */
    bool Child()
    {
      if (node->child)
      {
        node = node->child;
        return true;
      }

      return false;
    }

    /** Aade un nodo anterior o posterior.
     * @param  nodeData Datos asociados al nuevo nodo.
     * @param  after <b>true</b> para ponerlo despus del actual,
     * <b>false</b> para ponerlo antes.
     */
    void AddNode(T nodeData, bool after = true)
    {
      if (after)
			{
        node->next = new Node(nodeData, node, node->next, node->parent);
				tree->nodes.push_back(node->next);
			}
      else
			{
        node->prev = new Node(nodeData, node->prev, node, node->parent);
				tree->nodes.push_back(node->prev);
			}
    }

    /** Aade un nodo anterior o posterior y se sita en l.
     * @param  nodeData Datos asociados al nuevo nodo.
     * @param  after <b>true</b> para ponerlo despus del actual,
     * <b>false</b> para ponerlo antes.
     */
    void AddNodeGo(T nodeData, bool after = true)
    {
      if (after)
      {
        node->next = new Node(nodeData, node, node->next, node->parent);
        node = node->next;
      }
      else
      {
        node->prev = new Node(nodeData, node->prev, node, node->parent);
        node = node->prev;
      }

			tree->nodes.push_back(node);
    }
    
    /** Aade un nodo hijo, si no existe ya.
     * @param  nodeData Dtos del nuevo nodo hijo.
     */
    void AddBranch(T nodeData)
    {
      if (!node->child)
			{
        node->child = new Node(nodeData, 0, 0, node);
				tree->nodes.push_back(node->child);
			}
    }

    /** Aade un nodo hijo, si no existe ya, y se sita en l.
     * @param  nodeData Dtos del nuevo nodo hijo.
     */
    void AddBranchGo(T nodeData)
    {
      if (!node->child)
      {
        node->child = new Node(nodeData, 0, 0, node);
        node = node->child;
				tree->nodes.push_back(node);
      }
    }

    /** Borra un nodo y sus hijos
     */
    void RemoveNode()
    {
      if (node->next)
        node->next->prev = node->prev;

      if (node->prev)
        node->prev->next = node->next;

			// HMMM... igual no es tan lista :L
			tree->nodes.remove(node);

      JDELETE(node);
    }
    
    /** Va al primer nodo de la rama.
     */
    void FirstInBranch()
    {
      while (Prev());
    }

    /** Va al timo nodo de la rama.
     */
    void LastInBranch()
    {
      while (Next());
    }

    /** Va al primer nodo del rbol.
     */
    void FirstInTree()
    {
      while (Parent())
			{}

      FirstInBranch();
    }

    /** Va al primer nodo del rbol es lo mismo que FirstInTree().
     */
    void Root()
    {
      FirstInTree();
    }

    /** Determina si tiene hijos.
		 * @return <b>true</b> En caso de que los tenga, <b>false</b> si no.
     */
    bool HasChilds()
    {
      return (0 != node->child);
    }

    /** Determina si tiene anterior.
		 * @return <b>true</b> En caso de que lo tenga, <b>false</b> si no.
     */
    bool HasPrev()
    {
      return (0 != node->prev);
    }

    /** Determina si tiene siguiente.
		 * @return <b>true</b> En caso de que lo tenga, <b>false</b> si no.
     */
    bool HasNext()
    {
      return (0 != node->next);
    }
  };

	/** Devuelve un nuevo iterador. El llamador debe borrar el objeto
	 * al terminar de usarlo.
	 * @return Iterador posicionado en el elemento raz.
	 */
	Iterator * NewIterator() {return new Iterator(*this);}
	
	/** Se situa al comienzo de la lista global de nodos.
	 */
	void Begin() {itG = nodes.begin();}
	
	/** Determina si est al final de la lista global de nodos.
	 */
	bool End() {return itG == nodes.end();}
	
	/** Devuelve el objeto en la posicin actual de la lista global de nodos.
	 */
	T& Cur() {return (*itG)->data;}

	/** Va al nodo anterior en la lista global de nodos.
	 */
	void Prev() {--itG;}
	
	/** Va al siguiente nodo en la lista global de nodos.
	 */
	void Next() {++itG;}

	/** Devuelve el nmero de elementos del rbol.
	 * @return Nmero total de elementos en el rbol.
	 */
	void Size() {return nodes.size();}

	/** Borra los nodos del rbol (PERO NO SU CONTENIDO SI SON PUNTEROS).
	 */
	void Clear() {nodes.clear();}

  friend class Iterator;
};

#endif  // _JTREE_INCLUDED
