/*  GTKtalog.
 *  Copyright (C) 1999  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>

#include "folder.h"
#include "report.h"
#include "categories.h"
#include "exit.h"
#include "config_common.h"

static gchar DEFAULT_REPORT[] = "report";
static gchar DEFAULT_EXTENSION[] = "txt";
static gchar ALL_DISKS[] = N_("All disks");
static gchar ALL_CATEGORIES[] = N_("(All)");
static gchar CATEGORY_NONE[] = N_("(None)");
static GtkWidget *report_dialog_window;
static GtkWidget *disk_combo;
static GtkWidget *cat_combobox;
static GtkWidget *check_disks;
static GtkWidget *check_folders;
static GtkWidget *check_files;
static GtkWidget *check_sizes;
static GtkWidget *check_dates;
static GtkWidget *check_category;
static GtkWidget *exclude_vfs;
static GtkWidget *include_dir_names;
static GtkWidget *check_location;
static GtkWidget *check_description;
static GtkWidget *check_information;
static GtkWidget *check_only_desc;
static GtkWidget *check_only_cat;
static GtkWidget *check_quote;
static GtkWidget *no_newline;
static GtkWidget *add_legend;
static GtkWidget *button_sortbycat;
static GtkWidget *button_sortbyfile;
static GtkWidget *button_tree;
static GtkWidget *button_html;
static GtkWidget *button_xml;
static GtkWidget *button_txt;
static GtkWidget *button_manydirs;
static GSList *group_sort;
static GSList *group_html_xml;
static GtkWidget *list_separator;
static GtkWidget *output_entry;
static GtkWidget *gnome_output_entry;
static gboolean gb_check_disks;
static gboolean gb_check_disks_only;
static gboolean gb_check_folders;
static gboolean gb_check_files;
static gboolean gb_check_sizes;
static gboolean gb_check_dates;
static gboolean gb_check_category;
static gboolean gb_include_vfs;
static gboolean gb_include_dir_names;
static gboolean gb_check_location;
static gboolean gb_check_description;
static gboolean gb_check_information;
static gboolean gb_check_only_desc;
static gboolean gb_check_only_cat;
static gboolean gb_spec_report;
static gboolean gb_html_report;
static gboolean gb_xml_report;
static gboolean gb_manydirs_report;
static gint cat_to_report;

struct _FILE_and_int
{
  FILE *s;
  gint i;
};

void
_which_disk (GNode * gn, gpointer data)
{
  if (!strcmp
      (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (disk_combo)->entry)),
       folder_get_name (gn)))
    {
      ((GNode **) data)[0] = gn;
    }
}

char *tmp;

void
strswap (char *s, char src, char dst)
{
  int i;
  for (i = 0; s[i]; i++)
    if (s[i] == src)
      s[i] = dst;
}

int s_len;

int
counttags (char *s)
{
  int len;
  char *p;
  len = 0;
  s_len = 0;
  for (p = s; *p; p++)
    {
      s_len += 1;
      if ((*p) == '<' || (*p) == '>')
	len += 3;
      else if ((*p) == '&')
	len += 4;
      else if ((*p) == '\"')
	len += 5;
    }
  return (s_len - len);
}

/* #define MAX_ENTITY_LENGTH 6 */

char *
htmlize (char *s)
{
  int i, k;
  char *p;

  if (!gb_html_report)
    return (s);
  i = counttags (s);
  if (i == 0)
    return (s);
  g_free (tmp);
  tmp = g_new0 (char, s_len + i + 1);
  /* tmp = g_new0 (char, strlen (s) * MAX_ENTITY_LENGTH + 1); */
  k = 0;
  for (p = s; *p; p++)
    {
      if ((*p) == '<' || (*p) == '>' || (*p) == '&' || (*p) == '\"')
	{
	  tmp[k++] = '&';
	  if ((*p) == '<')
	    {
	      tmp[k++] = 'l';
	      tmp[k++] = 't';
	    }
	  if ((*p) == '>')
	    {
	      tmp[k++] = 'g';
	      tmp[k++] = 't';
	    }
	  if ((*p) == '&')
	    {
	      tmp[k++] = 'a';
	      tmp[k++] = 'm';
	      tmp[k++] = 'p';
	    }
	  if ((*p) == '\"')
	    {
	      tmp[k++] = 'q';
	      tmp[k++] = 'u';
	      tmp[k++] = 'o';
	      tmp[k++] = 't';
	    }
	  tmp[k++] = ';';
	}
      else
	tmp[k++] = p[0];
    }
  tmp[k] = 0;
  return (tmp);
}

char *
cstring (char *separator)
{
  int emode;
  char *s, *t;

  emode = 0;
  t = s = separator;
  while (s[0])
    {
      if (emode == 0)
	{
	  if (s[0] == '\\')
	    emode = 1;
	  else
	    {
	      t[0] = s[0];
	      t++;
	    }
	}
      else
	{
	  emode = 0;
	  if (s[0] == '\\')
	    t[0] = '\\';
	  else if (s[0] == 'r')
	    t[0] = '\r';
	  else if (s[0] == 't')
	    t[0] = '\t';
	  else if (s[0] == 'b')
	    t[0] = '\b';
	  else if (s[0] == 'a')
	    t[0] = '\a';
	  else if (s[0] == 'n')
	    t[0] = '\n';
	  else
	    t[0] = s[0];
	  t++;
	}
      s++;
    }
  t[0] = 0;
  return (separator);
}

/* The following function is recursive.
 * last_disk_name is initialized inside when g_node_depth(gn) == 2.
 * Then that value is used!
 * Same for quot, nothing, double_quotes, separator
 *
 * However, it would be better to use something more obvious if possible.
 * (Yves 2001-05-22)
 */
char *last_disk_name;
char *separator;
char *double_quotes = "\"";
char *nothing = "";
char *quot;
char *html_td_start = "<TD>";
char *html_td_end = "</TD>";
char *xml_report_st = "<report>";
char *xml_report_ed = "</report>";
char *xml_disc_name_st = "\t<disc_name><![CDATA[";
char *xml_disc_name_ed = "]]></disc_name>\n";
char *xml_legend_st = "<legend><![CDATA[";
char *xml_legend_ed = "]]></legend>";
char *xml_directory_st = "<directory>\n";
char *xml_directory_ed = "</directory>";
char *xml_file_st = "\t<file_name><![CDATA[";
char *xml_file_ed = "]]></file_name>\n";
char *xml_size_st = "\t<file_size><![CDATA[";
char *xml_size_ed = "]]></file_size>\n";
char *xml_date_st = "\t<file_date><![CDATA[";
char *xml_date_ed = "]]></file_date>\n";
char *xml_description_st = "\t<description><![CDATA[";
char *xml_description_ed = "]]></description>\n";
char *xml_information_st = "\t<information><![CDATA[";
char *xml_information_ed = "]]></information>\n";
char *xml_category_st = "\t<category><![CDATA[";
char *xml_category_ed = "]]></category>\n";
char *xml_location_st = "\t<directory_name><![CDATA[";
char *xml_location_ed = "]]></directory_name>\n";
char *td_st;
char *td_ed;

void
set_quotes_and_separator ()
{
  separator =
    cstring ((gchar *) gtk_entry_get_text (GTK_ENTRY (list_separator)));

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_quote)) &&
      !gb_html_report && !gb_xml_report)
    quot = double_quotes;
  else
    quot = nothing;

  if (gb_html_report)
    {
      td_st = html_td_start;
      td_ed = html_td_end;
      separator = nothing;
    }
  else if (gb_xml_report)
    {
      separator = nothing;
    }
  else
    {
      td_st = nothing;
      td_ed = nothing;
    }
}

void
print_file (FILE_DATA * fd, FILE * file)
{
  GNode *gn = fd->node;
  GString *gs;
  gchar *gdescr;

  if (is_link_dest (gn))
    return;

  if (is_dir (gn) && (!gb_include_dir_names) && gb_check_files)
    return;

  if (!is_dir (gn) && (!is_vfs_any (gn)) && (!gb_check_files))
    return;

  if ((!gb_include_vfs)
      && ((fd->a_parent_node_is_vfs == IS_VFS)
	  || (fd->a_parent_node_is_vfs == IS_VFS_EXTENDED)))
    return;

  if ((cat_to_report != -1) && (fd->categorie != cat_to_report))
    return;

  if ((gb_check_only_desc) && (fd->description == 0))
    return;
  if ((gb_check_only_cat) && (fd->categorie == 0))
    return;

  if (gb_html_report)
    fprintf (file, "<TR>");
  if (gb_xml_report)
    fprintf (file, xml_directory_st);
  if (gb_check_disks)
    {
      if (gb_xml_report)
	{
	  td_st = xml_disc_name_st;
	  td_ed = xml_disc_name_ed;
	}
      fprintf (file, "%s%s%s%s%s%s", td_st, quot,
	       (last_disk_name ? htmlize (last_disk_name) : ""), quot,
	       separator, td_ed);
    }

  if (gb_check_folders)
    {
      if (gb_xml_report)
	{
	  td_st = xml_location_st;
	  td_ed = xml_location_ed;
	}
      gs = get_path_from_node (gn);
      gdescr = g_strdup (gs->str);
      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (no_newline)))
	strswap (gdescr, '\n', ' ');
      fprintf (file, "%s", td_st);
      if (gb_manydirs_report)
	{
	  gchar *p1, *p2, c;
	  p1 = gdescr;
	  c = 1;
	  while (c)
	    {
	      while (p1[0] == '/')
		p1++;
	      p2 = p1;
	      while (p2[0] && (p2[0] != '/'))
		p2++;
	      c = p2[0];
	      p2[0] = '\0';
	      if (p1 != p2)
		fprintf (file, "%s%s%s%s", quot, p1, quot, separator);
	      p1 = p2 + 1;
	    }
	  fprintf (file, "%s", td_ed);

	}
      else
	{
	  fprintf (file, "%s%s%s%s%s", quot, htmlize (gdescr), quot,
		   separator, td_ed);
	}
      g_free (gdescr);
      g_string_free (gs, TRUE);
    }

  if ((gb_check_files)
      || (((is_dir (gn) || (is_vfs_any (gn))) && (!gb_check_files))))
    {
      if (is_link (gn))
	{
	  FILE_DATA *fdl;
	  fdl = get_file_data_from_gnode (g_node_first_child (gn));
	  gdescr = g_strdup (fdl->name->str);
	}
      else
	{

	  gdescr = g_strdup (fd->name->str);
	}
      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (no_newline)))
	strswap (gdescr, '\n', ' ');
      if (gb_xml_report)
	{
	  td_st = xml_file_st;
	  td_ed = xml_file_ed;
	}
      if (is_dir (gn) && gb_check_files)
	{
	  fprintf (file, "%s%s%s/%s%s", td_st, quot, htmlize (gdescr), quot,
		   td_ed);
	}
      else
	{
	  fprintf (file, "%s%s%s%s%s", td_st, quot, htmlize (gdescr), quot,
		   td_ed);
	}
      g_free (gdescr);
    }
  if (gb_check_sizes)
    {
      if (gb_xml_report)
	{
	  td_st = xml_size_st;
	  td_ed = xml_size_ed;
	}
      if (!((fd->a_parent_node_is_vfs == IS_VFS) || (is_unreadable (gn))))
	{
	  fprintf (file, "%s%s%s%d%s%s", td_st, separator, quot, fd->taille,
		   quot, td_ed);
	}
      else
	{
	  fprintf (file, "%s%s%s-1%s%s", td_st, separator, quot, quot, td_ed);
	}
    }
  if (gb_check_dates)
    {
      if (gb_xml_report)
	{
	  td_st = xml_date_st;
	  td_ed = xml_date_ed;
	}
      if (!((fd->a_parent_node_is_vfs == IS_VFS) || (is_unreadable (gn))))
	{
	  char humandate[] = "DD/MM/YYYY HH:MM:SS ";
	  strftime (humandate, sizeof (humandate), "%d/%m/%Y %H:%M:%S",
		    gmtime (&fd->date));
	  fprintf (file, "%s%s%s%s%s%s", td_st, separator, quot, humandate,
		   quot, td_ed);
	}
      else
	{
	  fprintf (file, "%s%s%s-1%s%s", td_st, separator, quot, quot, td_ed);
	}
    }
  if (gb_check_category)
    {
      if (gb_xml_report)
	{
	  td_st = xml_category_st;
	  td_ed = xml_category_ed;
	}
      if (fd->categorie != 0)
	{
	  gdescr = g_strdup (folder_get_category_from_file_data (fd));
	  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (no_newline)))
	    strswap (gdescr, '\n', ' ');
	  fprintf (file, "%s%s%s%s%s%s", td_st, separator, quot,
		   htmlize (gdescr), quot, td_ed);
	  g_free (gdescr);
	}
      else
	fprintf (file, "%s%s%s%s%s", td_st, separator, quot, quot, td_ed);
    }


  if (gb_check_description)
    {
      if (gb_xml_report)
	{
	  td_st = xml_description_st;
	  td_ed = xml_description_ed;
	}
      if (fd->description != 0)
	{
	  gdescr = g_strdup (folder_get_description_from_file_data (fd));
	  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (no_newline)))
	    strswap (gdescr, '\n', ' ');
	  fprintf (file, "%s%s%s%s%s%s", td_st, separator, quot,
		   htmlize (gdescr), quot, td_ed);
	  g_free (gdescr);
	}
      else
	fprintf (file, "%s%s%s%s%s", td_st, separator, quot, quot, td_ed);
    }

  if (gb_check_information)
    {
      if (gb_xml_report)
	{
	  td_st = xml_information_st;
	  td_ed = xml_information_ed;
	}
      if (fd->information != 0)
	{
	  gdescr = g_strdup (fd->information->str);
	  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (no_newline)))
	    strswap (gdescr, '\n', ' ');
	  fprintf (file, "%s%s%s%s%s%s", td_st, separator, quot,
		   htmlize (gdescr), quot, td_ed);
	  g_free (gdescr);
	}
      else
	fprintf (file, "%s%s%s%s%s", td_st, separator, quot, quot, td_ed);
    }
  if (gb_html_report)
    fprintf (file, "</TR>");
  if (gb_xml_report)
    fprintf (file, xml_directory_ed);
  fprintf (file, "\n");
}


/* by tamas gervai <crown1@operamail.com> */
void
_report_importable_files (GNode * gn, gpointer data)
{
  FILE_DATA *fd = get_file_data_from_gnode (gn);
  FILE *file = data;

  if (g_node_depth (gn) == 2)
    {
      last_disk_name = fd->name->str;
      g_node_children_foreach (gn, G_TRAVERSE_ALL, _report_importable_files,
			       file);
    }
  else
    {
      print_file (fd, file);
      g_node_children_foreach (gn, G_TRAVERSE_ALL, _report_importable_files,
			       file);
    }
}

void
_report_files (GNode * gn, gpointer data)
{
  FILE *file = data;
  FILE_DATA *fd = get_file_data_from_gnode (gn);
  int i;
  GString *gs;
  gchar *gdescr;

  if (is_link_dest (gn))
    return;

  if (g_node_depth (gn) == 2)
    {
      if ((gb_check_disks)
	  && ((cat_to_report == -1)
	      || (fd->categorie == cat_to_report))
	  && ((gb_check_only_desc == FALSE)
	      || ((gb_check_only_desc) && (fd->description != 0))))
	{
	  fprintf (file, "\n%s\n", fd->name->str);
	  for (i = 0; i < fd->name->len; i++)
	    fprintf (file, "-");
	  fprintf (file, "\n");
	}
    }
  else
    {
      if (((gb_include_dir_names)
	   && ((is_dir (gn)) || (is_vfs_any (gn))))
	  || ((gb_check_files)
	      && ((is_file (gn)) || (is_unreadable (gn)) || (is_link (gn)))))
	{
	  if (((fd->categorie == cat_to_report)
	       || (cat_to_report == -1))
	      && ((gb_check_only_desc == FALSE)
		  || ((gb_check_only_desc)
		      && (fd->description != 0)))
	      && ((((gb_check_only_cat) && (fd->categorie != 0))
		   || (gb_check_only_cat == FALSE))))
	    {
	      for (i = 0; i < g_node_depth (gn) - 1; i++)
		fprintf (file, "  ");
	      if (is_link (gn))
		{
		  FILE_DATA *fdl;
		  fdl = get_file_data_from_gnode (g_node_first_child (gn));
		  fprintf (file, "%s", fdl->name->str);
		}
	      else
		{
		  fprintf (file, "%s", fd->name->str);
		}


	      if (((gb_check_category) && (fd->categorie != 0))
		  ||
		  ((gb_check_description)
		   && (fd->description != 0))
		  || ((gb_check_information) && (fd->information != 0)))
		fprintf (file, "  (");



	      if ((gb_check_category) && (fd->categorie != 0))
		{
		  if (gtk_toggle_button_get_active
		      (GTK_TOGGLE_BUTTON (no_newline)))
		    {
		      gdescr =
			g_strdup (folder_get_category_from_file_data (fd));
		      strswap (gdescr, '\n', ' ');
		      fprintf (file, " %s '%s' ", _("category"), gdescr);
		      g_free (gdescr);
		    }
		  else
		    {
		      fprintf (file, " %s '%s' ", _("category"),
			       folder_get_category_from_file_data (fd));
		    }
		}
	      if ((gb_check_description) && (fd->description != 0))
		{
		  if (gtk_toggle_button_get_active
		      (GTK_TOGGLE_BUTTON (no_newline)))
		    {
		      gdescr =
			g_strdup (folder_get_description_from_file_data (fd));
		      strswap (gdescr, '\n', ' ');
		      fprintf (file, " %s '%s' ", _("description"), gdescr);
		      g_free (gdescr);
		    }
		  else
		    {
		      fprintf (file, " %s '%s' ", _("description"),
			       folder_get_description_from_file_data (fd));
		    }
		}
	      if ((gb_check_information) && (fd->information != 0))
		{
		  if (gtk_toggle_button_get_active
		      (GTK_TOGGLE_BUTTON (no_newline)))
		    {
		      gdescr = g_strdup (fd->information->str);
		      strswap (gdescr, '\n', ' ');
		      fprintf (file, " %s '%s' ", _("information"), gdescr);
		      g_free (gdescr);
		    }
		  else
		    {
		      fprintf (file, " %s '%s' ", _("information"),
			       fd->information->str);
		    }
		}
	      if (((gb_check_category) && (fd->categorie != 0))
		  ||
		  ((gb_check_description)
		   && (fd->description != 0))
		  || ((gb_check_information) && (fd->information != 0)))
		fprintf (file, ")");



	      if ((gb_check_folders) && ((is_dir (gn)) || (is_vfs_any (gn))))
		{
		  if (gb_check_location)
		    {
		      gs = get_path_from_node (gn);
		      fprintf (file, " (%s)", gs->str);
		      g_string_free (gs, TRUE);
		    }
		}
	      fprintf (file, "\n");
	    }
	}
    }
  g_node_children_foreach (gn, G_TRAVERSE_ALL, _report_files, file);
}

gboolean
generate_file_report (FOLDER * racine, FILE * file)
{
  int i;
  char *the_cat;
  GNode *disk_to_report = NULL;
  FILE_DATA *fd;

  the_cat =
    g_strdup (gtk_entry_get_text
	      (GTK_ENTRY (GTK_COMBO (cat_combobox)->entry)));
  cat_to_report = -2;
  if (strlen (the_cat) == 0)
    {
      cat_to_report = 0;
    }
  else if (strcmp (the_cat, _(CATEGORY_NONE)) == 0)
    {
      cat_to_report = 0;
    }
  else if (strcmp (the_cat, _(ALL_CATEGORIES)) == 0)
    {
      cat_to_report = -1;
    }
  else
    {
      for (i = 1; i <= racine->categories->len; i++)
	{
	  if (strcmp (the_cat, get_categorie (racine, i)) == 0)
	    cat_to_report = i;
	}
    }
  g_free (the_cat);
  if (cat_to_report == -2)
    {
      ERROR_DIALOG (_("Wrong category"), report_dialog_window);
      return (FALSE);
    }

  if (strcmp (gtk_entry_get_text
	      (GTK_ENTRY (GTK_COMBO (disk_combo)->entry)), _(ALL_DISKS)) == 0)
    disk_to_report = racine->tree;
  else
    {
      disk_to_report = NULL;
      g_node_children_foreach (racine->tree, G_TRAVERSE_ALL,
			       _which_disk, &disk_to_report);
    }
  if (disk_to_report == NULL)
    {
      ERROR_DIALOG (_("Wrong disk name"), report_dialog_window);
      return (FALSE);
    }
  else if (disk_to_report != racine->tree)
    {
      fd = get_file_data_from_gnode (disk_to_report);

      last_disk_name = fd->name->str;
    }

  if (gb_spec_report)
    {
      set_quotes_and_separator ();
      tmp = g_new0 (char, 1);
      g_node_children_foreach (disk_to_report, G_TRAVERSE_ALL,
			       _report_importable_files, file);
      g_free (tmp);
    }
  else
    {
      g_node_children_foreach (disk_to_report, G_TRAVERSE_ALL,
			       _report_files, file);
    }

  return (TRUE);
}


void
_print_disk (GNode * gn, gpointer data)
{
  FILE *file = data;
  gchar *name;

  set_quotes_and_separator ();
  if (!strcmp
      (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (disk_combo)->entry)),
       folder_get_name (gn)))
    {
      ((GNode **) data)[0] = gn;
    }
  if (gb_html_report)
    fprintf (file, "<TR>");

  if (gb_html_report)
    {
      name = g_strdup (folder_get_name (gn));
      fprintf (file, "%s%s%s%s%s%s", td_st, quot, htmlize (name),
	       quot, separator, td_ed);
      g_free (name);
    }
  else
    fprintf (file, "%s", folder_get_name (gn));

  if (gb_html_report)
    fprintf (file, "</TR>");
  fprintf (file, "\n");
}


gboolean
generate_disk_report (FOLDER * racine, FILE * file)
{
  int i;
  char *the_cat;

  fprintf (stderr, "generate_disk_report()\n");

  the_cat =
    g_strdup (gtk_entry_get_text
	      (GTK_ENTRY (GTK_COMBO (cat_combobox)->entry)));
  cat_to_report = -2;
  if (strlen (the_cat) == 0)
    {
      cat_to_report = 0;
    }
  else if (strcmp (the_cat, _(CATEGORY_NONE)) == 0)
    {
      cat_to_report = 0;
    }
  else if (strcmp (the_cat, _(ALL_CATEGORIES)) == 0)
    {
      cat_to_report = -1;
    }
  else
    {
      for (i = 1; i <= racine->categories->len; i++)
	{
	  if (strcmp (the_cat, get_categorie (racine, i)) == 0)
	    cat_to_report = i;
	}
    }
  g_free (the_cat);
  if (cat_to_report == -2)
    {
      ERROR_DIALOG (_("Wrong category"), report_dialog_window);
      return (FALSE);
    }

  g_node_children_foreach (racine->tree, G_TRAVERSE_ALL, _print_disk, file);

  return (TRUE);
}


gboolean
_report_cats (GNode * _gn, gpointer data)
{
  struct _FILE_and_int *si = data;
  FILE *file = si->s;
  gint cat = si->i;
  FILE_DATA *fd;
  GNode *gn;
  if (!_gn->data)
    return (FALSE);
  if (G_NODE_IS_ROOT (_gn))
    return (FALSE);

  fd = _gn->data;
  gn = fd->node;

  if (((gb_check_folders)
       && ((is_dir (gn)) || (is_vfs_any (gn))))
      || ((gb_check_files)
	  && ((is_file (gn)) || (is_unreadable (gn)) || (is_link (gn)))))
    {

      if (((fd->categorie == cat) || (cat == -1))
	  && ((gb_check_only_desc == FALSE)
	      || ((gb_check_only_desc) && (fd->description != 0))))
	{
	  last_disk_name = g_strdup (get_disk_from_node (gn));
	  print_file (fd, file);
	  g_free (last_disk_name);
	}
    }
  return (FALSE);
}

gboolean
_sort_disk_by_cats (GNode * gn, gpointer data)
{
  GNode *sort_by_categories = data;
  FILE_DATA *fd = get_file_data_from_gnode (gn);
  gint i = fd->categorie;
  GNode *cat_node;

  if (fd->categorie)
    {
      cat_node =
	g_node_find_child (sort_by_categories, G_TRAVERSE_ALL,
			   GINT_TO_POINTER (i));
      if (!cat_node)
	{
	  cat_node =
	    g_node_append_data (sort_by_categories, GINT_TO_POINTER (i));
	}
      g_node_append_data (cat_node, fd);
    }
  return (FALSE);
}

gboolean
generate_cat_report (FOLDER * racine, FILE * file)
{
  gint i;
  gint j;
  gchar *the_cat;
  const gchar *tmpcat;
  static struct _FILE_and_int _file_i;
  GNode *disk_to_report = NULL;
  GNode *sort_by_categories;

  the_cat =
    g_strdup (gtk_entry_get_text
	      (GTK_ENTRY (GTK_COMBO (cat_combobox)->entry)));
  set_quotes_and_separator ();
  cat_to_report = -2;
  if (strlen (the_cat) == 0)
    {
      cat_to_report = 0;
    }
  else if (strcmp (the_cat, _(CATEGORY_NONE)) == 0)
    {
      cat_to_report = 0;
    }
  else if (strcmp (the_cat, _(ALL_CATEGORIES)) == 0)
    {
      cat_to_report = -1;
    }
  else
    {
      for (i = 1; i <= racine->categories->len; i++)
	{
	  if (strcmp (the_cat, get_categorie (racine, i)) == 0)
	    cat_to_report = i;
	}
    }
  g_free (the_cat);

  if (cat_to_report == -2)
    {
      ERROR_DIALOG (_("Wrong category"), report_dialog_window);
      return (FALSE);
    }

  if (strcmp (gtk_entry_get_text
	      (GTK_ENTRY (GTK_COMBO (disk_combo)->entry)), _(ALL_DISKS)) == 0)
    disk_to_report = racine->tree;
  else
    {
      disk_to_report = NULL;
      g_node_children_foreach (racine->tree, G_TRAVERSE_ALL,
			       _which_disk, &disk_to_report);
    }
  if (disk_to_report == NULL)
    {
      ERROR_DIALOG (_("Wrong disk name"), report_dialog_window);
      return (FALSE);
    }

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (add_legend)))
    {

      fprintf (file,
	       _
	       ("Categories present in archive\n=============================\n"));
      for (i = 0; i < racine->categories->len; i++)
	fprintf (file, "    %s\n", get_categorie (racine, i + 1));
      fprintf (file, "\n");
    }
  sort_by_categories = g_node_new (NULL);
  g_node_traverse (disk_to_report, G_IN_ORDER, G_TRAVERSE_ALL, -1,
		   _sort_disk_by_cats, sort_by_categories);
  if (cat_to_report == -1)
    {
      for (i = 1; i < racine->categories->len + 1; i++)
	{
	  GNode *cat;
	  tmpcat = get_categorie (racine, i);
	  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (add_legend)))
	    {
	      fprintf (file, _("\nCategory: %s\n=========="), tmpcat);
	      for (j = 0; j < strlen (tmpcat); j++)
		fprintf (file, "=");
	      fprintf (file, "\n");
	    }
	  _file_i.i = i;
	  _file_i.s = file;
	  cat =
	    g_node_find_child (sort_by_categories, G_TRAVERSE_ALL,
			       GINT_TO_POINTER (i));
	  if (cat)
	    {
	      g_node_traverse (cat, G_IN_ORDER, G_TRAVERSE_LEAFS, -1,
			       _report_cats, &_file_i);

	    }
	}
      if (gb_check_only_cat == FALSE)
	{
	  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (add_legend)))
	    {
	      fprintf (file, _("\nCategory: None\n==============\n"));
	    }
	  _file_i.i = 0;
	  _file_i.s = file;
	  g_node_traverse (disk_to_report, G_IN_ORDER, G_TRAVERSE_ALL, -1,
			   _report_cats, &_file_i);
	}
    }
  else
    {
      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (add_legend)))
	{
	  if (cat_to_report != 0)
	    {
	      tmpcat = get_categorie (racine, cat_to_report);
	      fprintf (file, _("\nCategory: %s\n=========="), tmpcat);
	      for (j = 0; j < strlen (tmpcat); j++)
		fprintf (file, "=");
	    }
	  else
	    {
	      fprintf (file, _("\nCategory: None\n=============="));
	    }
	  fprintf (file, "\n");
	}
      _file_i.i = cat_to_report;
      _file_i.s = file;
      g_node_traverse (disk_to_report, G_IN_ORDER, G_TRAVERSE_ALL, -1,
		       _report_cats, &_file_i);
    }
  g_node_destroy (sort_by_categories);
  return TRUE;
}

void
begin_report_orig (GtkWidget * w, gpointer data)
{
  FOLDER *racine = data;

  G_CONST_RETURN char *path;
  FILE *file;
  gboolean result;
  path = gtk_entry_get_text (GTK_ENTRY (output_entry));

  if (g_file_test (path, G_FILE_TEST_IS_REGULAR))
    {
      if (yes_no_dialog
	  (_("remove file?"),
	   _("The report file already exists. Do you want to overwrite it?")))
	{
	  return;
	}
    }


  if ((file = fopen (path, "w+")) == NULL)
    {
      ERROR_DIALOG (_
		    ("Can't create file. Permission denied, disk full or wrong path."),
		    report_dialog_window);
      return;
    }
  gb_check_disks =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_disks));
  gb_check_folders =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_folders));
  gb_check_files =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_files));
  if (gb_check_files)
    {
      gb_check_sizes =
	gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_sizes));
      gb_check_dates =
	gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_dates));
    }
  else
    {
      gb_check_sizes = FALSE;
      gb_check_dates = FALSE;
    }
  gb_check_location =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_location));
  gb_check_description =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_description));
  gb_check_information =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_information));
  gb_check_category =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_category));
  gb_include_vfs =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (exclude_vfs));
  gb_include_dir_names =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (include_dir_names));
  gb_check_only_cat =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_only_cat));
  gb_check_only_desc =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_only_desc));
  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_sortbyfile)))
    {
      gb_html_report = FALSE;
      gb_xml_report = FALSE;
    }
  else
    {
      gb_html_report =
	gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_html));
      gb_xml_report = !gb_html_report;
    }

  if (gb_check_disks && !gb_check_folders && !gb_check_files &&
      !gb_check_category && !gb_check_description && !gb_check_information)
    {
      fprintf (stderr, "gb_check_disks_only = TRUE\n");
      gb_check_disks_only = TRUE;
    }
  else
    {
      gb_check_disks_only = FALSE;
    }
  gb_manydirs_report =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_manydirs));

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_tree)))
    {
      gb_include_dir_names = TRUE;
    }
  if (gb_html_report)
    {
      fprintf (file,
	       "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \"http://www.w3.org/TR/html4/loose.dtd\">");
      fprintf (file, "<HTML><HEAD><TITLE>%s</TITLE></HEAD><BODY>",
	       _("GTKtalog's Catalog"));
      fprintf (file, "<STYLE TYPE=\"text/css\"><!--\n");
      fprintf (file,
	       "body { background: white; color: black: margin: 0.5em; font-size: 9pt;}\n");
      fprintf (file, "TT { font-family: courier; }\n");
      fprintf (file,
	       "TD { font-family: helvetica, sans-serif; font-size: 9pt; border-bottom: thin solid #000000; border-left: thin solid #000000;}\n");
      fprintf (file, "CAPTION {\n");
      fprintf (file, "	font-family: helvetica, sans-serif;\n");
      fprintf (file, "	font-size: 12pt;\n");
      fprintf (file, "	text-align: left;\n");
      fprintf (file, "}\n--></STYLE>\n</HEAD><BODY>");
    }
  if (gb_xml_report)
    {
      fprintf (file, "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n");
      fprintf (file,
	       "<?xml-stylesheet type=\"text/css\" href=\"gtktalog.css\"?>\n");
      fprintf (file, "<report>\n");
      fprintf (file, "<name>%s</name>\n", _("GTKtalog's Catalog"));
    }

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (add_legend)))
    {
      if (gb_html_report)
	fprintf (file, "<PRE>");
      if (gb_xml_report)
	fprintf (file, "%s\n", xml_legend_st);
      fprintf (file, _("This report was generated by GTKtalog.\n"));
      fprintf (file, GTKTALOGWEB "\n\n");

      fprintf (file, _("The report type is: "));
      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_sortbycat)))
	{
	  fprintf (file, _("Sort by category."));
	}
      else
	if (gtk_toggle_button_get_active
	    (GTK_TOGGLE_BUTTON (button_sortbyfile)))
	{
	  fprintf (file, _("Sort by Disk->Dir->File."));
	}
      else
	{
	  fprintf (file, _("Sort as a tree."));
	}
      fprintf (file, "\n===================\n");
      fprintf (file, _("  It contains:\n"));
      if (gb_check_disks)
	fprintf (file, _("           - disk name.\n"));
      if (gb_check_folders)
	fprintf (file, _("           - directory name.\n"));
      if (gb_check_files)
	fprintf (file, _("           - file name.\n"));
      if (gb_check_sizes)
	fprintf (file, _("           - file size.\n"));
      if (gb_check_dates)
	fprintf (file, _("           - file date.\n"));
      if ((gb_check_location)
	  && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_tree)))
	fprintf (file, _("           - directory location.\n"));
      if (gb_check_description)
	fprintf (file, _("           - description.\n"));
      if (gb_check_information)
	fprintf (file, _("           - information.\n"));
      if (gb_check_category)
	fprintf (file, _("           - category.\n"));
      if (gb_check_only_cat)
	fprintf (file, _("           - Just items with category.\n"));
      if (gb_check_only_desc)
	fprintf (file, _("           - Just items with description.\n"));
      if ((gb_include_dir_names)
	  &&
	  (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_tree))))
	fprintf (file, _("           - Include directories.\n"));
      fprintf (file, "\n");
      if (gb_html_report)
	fprintf (file, "</PRE>");
      if (gb_xml_report)
	fprintf (file, "%s\n", xml_legend_ed);
    }

  if (gb_html_report)
    fprintf (file, "<TABLE border=0><CAPTION>Catalog</CAPTION>");

  if (gb_check_disks_only)
    {
      result = generate_disk_report (racine, file);
    }
  else
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_sortbycat)))
    {
      result = generate_cat_report (racine, file);
    }
  else
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_sortbyfile)))
    {
      result = generate_file_report (racine, file);
    }
  else
    {
      result = generate_file_report (racine, file);
    }

  if (gb_html_report)
    fprintf (file, "</TABLE>\n</BODY></HTML>");
  if (gb_xml_report)
    fprintf (file, "</report>");
  fclose (file);
  if (result)
    WARNING_DIALOG (_("Report successful."), report_dialog_window);

  return;
}

void
begin_report (GtkWidget * w, gpointer data)
{
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button_tree)))
    gb_spec_report = FALSE;
  else
    gb_spec_report = TRUE;

  begin_report_orig (w, data);
}

static void
cancel_clicked (GtkWidget * w, gpointer data)
{
  gtk_widget_destroy (report_dialog_window);
  return;
}

static void
destroy_window (GtkWidget * w, GdkEventKey * event, gpointer data)
{
  cancel_clicked (w, data);
}

static void
switch_size (GtkWidget * w, gpointer data)
{
  GtkWidget *w2 = data;
  gtk_widget_set_sensitive (GTK_WIDGET (w2),
			    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
							  (w)));
}

static void
switch_report_name (GtkWidget * w, gpointer data)
{
  // this should set the default output to
  // * DEFAULT_REPORT.html
  // or
  // * DEFAULT_REPORT.xml
  // or
  // * DEFAULT_REPORT.DEFAULT_EXTENSION
}

static void
switch_dirs (GtkWidget * w, gpointer data)
{
  GtkWidget *w2 = data;
  gtk_widget_set_sensitive (GTK_WIDGET (w2),
			    !gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
							   (w)));
}

void
open_report_dialog (FOLDER * racine)
{
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *vbox2;
  GtkWidget *label;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *my_button;

  GList *disk_list;
  GList *cat_list;

  gchar *report_out;
  GtkAccelGroup *accel_group;

  accel_group = gtk_accel_group_new ();

  report_dialog_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  gtk_window_set_modal (GTK_WINDOW (report_dialog_window), TRUE);
  gtk_window_set_transient_for (GTK_WINDOW (report_dialog_window),
				GTK_WINDOW (main_window));

  gtk_container_border_width (GTK_CONTAINER (report_dialog_window), 10);
  gtk_window_set_policy (GTK_WINDOW (report_dialog_window), TRUE, TRUE, TRUE);
  gtk_window_set_title (GTK_WINDOW (report_dialog_window),
			_("Report generator"));
  gtk_window_set_wmclass (GTK_WINDOW (report_dialog_window),
			  _("Report generator"), "GTKtalog");
  gtk_signal_connect (GTK_OBJECT (report_dialog_window), "delete_event",
		      G_CALLBACK (destroy_window), NULL);
  gtk_widget_show (report_dialog_window);


  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_add (GTK_CONTAINER (report_dialog_window), vbox);
  gtk_widget_show (vbox);

  /**************/
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show (hbox);

  label = gtk_label_new (_("Disk:"));
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  disk_combo = gtk_combo_new ();
  gtk_combo_set_value_in_list (GTK_COMBO (disk_combo), FALSE, TRUE);
  gtk_combo_set_use_arrows_always (GTK_COMBO (disk_combo), TRUE);
  gtk_combo_disable_activate (GTK_COMBO (disk_combo));

  disk_list = make_disk_list (racine);
  disk_list = g_list_prepend (disk_list, _(ALL_DISKS));
  gtk_combo_set_popdown_strings (GTK_COMBO (disk_combo), disk_list);
  gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO (disk_combo)->entry),
			     FALSE);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (disk_combo)->entry),
		      _("All disks"));
  gtk_box_pack_start (GTK_BOX (hbox), disk_combo, FALSE, FALSE, 0);
  gtk_widget_show (disk_combo);

  /********/


  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show (hbox);

  label = gtk_label_new (_("Category:  "));
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  cat_combobox = gtk_combo_new ();
  gtk_combo_set_value_in_list (GTK_COMBO (cat_combobox), FALSE, TRUE);
  gtk_combo_set_use_arrows_always (GTK_COMBO (cat_combobox), TRUE);
  gtk_combo_disable_activate (GTK_COMBO (cat_combobox));

  cat_list = make_categories_list (racine);
  cat_list = g_list_append (cat_list, _(ALL_CATEGORIES));
  cat_list = g_list_append (cat_list, _(CATEGORY_NONE));

  gtk_combo_set_popdown_strings (GTK_COMBO (cat_combobox), cat_list);
  gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO (cat_combobox)->entry),
			     FALSE);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (cat_combobox)->entry),
		      _(ALL_CATEGORIES));
  gtk_box_pack_start (GTK_BOX (hbox), cat_combobox, FALSE, FALSE, 0);
  gtk_widget_show (cat_combobox);

  /*********/

  frame = gtk_frame_new (NULL);
  gtk_frame_set_label (GTK_FRAME (frame), _("Report fields"));
  gtk_frame_set_label_align (GTK_FRAME (frame), 0.0, 0.0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  table = gtk_table_new (5, 2, FALSE);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_widget_show (table);

  check_disks = gtk_check_button_new_with_label (_("Disks"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_disks), TRUE);
  gtk_table_attach_defaults (GTK_TABLE (table), check_disks, 0, 1, 0, 1);
  gtk_widget_show (check_disks);

  check_folders = gtk_check_button_new_with_label (_("Directories"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_folders), TRUE);
  gtk_table_attach_defaults (GTK_TABLE (table), check_folders, 0, 1, 1, 2);
  gtk_widget_show (check_folders);

  check_files = gtk_check_button_new_with_label (_("Files"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_files), TRUE);
  gtk_table_attach_defaults (GTK_TABLE (table), check_files, 0, 1, 2, 3);
  gtk_widget_show (check_files);

  check_sizes = gtk_check_button_new_with_label (_("Sizes"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_sizes), TRUE);
  gtk_table_attach_defaults (GTK_TABLE (table), check_sizes, 0, 1, 3, 4);
  gtk_widget_show (check_sizes);

  check_dates = gtk_check_button_new_with_label (_("Dates"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_dates), TRUE);
  gtk_table_attach_defaults (GTK_TABLE (table), check_dates, 0, 1, 4, 5);
  gtk_widget_show (check_dates);
  /* Warning: if storing the states of those buttons, this check_sizes button should be set or not
   * sensitive, depending on the check_files button.
   * Currently, the check_files button is enabled so the check_sizes button is set sensitive.
   */
  gtk_signal_connect (GTK_OBJECT (check_files), "clicked",
		      GTK_SIGNAL_FUNC (switch_size), check_sizes);

  gtk_signal_connect (GTK_OBJECT (check_files), "clicked",
		      GTK_SIGNAL_FUNC (switch_size), check_dates);


  check_category = gtk_check_button_new_with_label (_("Category"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_category), TRUE);
  gtk_table_attach_defaults (GTK_TABLE (table), check_category, 1, 2, 0, 1);
  gtk_widget_show (check_category);

  check_description = gtk_check_button_new_with_label (_("Descriptions"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_description), FALSE);
  gtk_table_attach_defaults (GTK_TABLE (table), check_description, 1, 2,
			     1, 2);
  gtk_widget_show (check_description);

  check_information = gtk_check_button_new_with_label (_("Information"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_information), TRUE);
  gtk_table_attach_defaults (GTK_TABLE (table), check_information, 1, 2,
			     2, 3);
  gtk_widget_show (check_information);

  /*********/

  frame = gtk_frame_new (NULL);
  gtk_frame_set_label (GTK_FRAME (frame), _("Report options"));
  gtk_frame_set_label_align (GTK_FRAME (frame), 0.0, 0.0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  table = gtk_table_new (2, 2, FALSE);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_widget_show (table);

  exclude_vfs = gtk_check_button_new_with_label (_("Include VFS content"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (exclude_vfs), FALSE);
  gtk_table_attach_defaults (GTK_TABLE (table), exclude_vfs, 0, 1, 0, 1);
  gtk_widget_show (exclude_vfs);

  include_dir_names =
    gtk_check_button_new_with_label (_("Include directories"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (include_dir_names), FALSE);
  gtk_table_attach_defaults (GTK_TABLE (table), include_dir_names, 0, 1, 1,
			     2);
  gtk_widget_show (include_dir_names);

  check_location = gtk_check_button_new_with_label (_("Location"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_location), FALSE);
  gtk_table_attach_defaults (GTK_TABLE (table), check_location, 0, 1, 2, 3);
  gtk_widget_show (check_location);

  check_only_desc =
    gtk_check_button_new_with_label (_("Only items with description"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_only_desc), FALSE);
  gtk_table_attach_defaults (GTK_TABLE (table), check_only_desc, 1, 2, 0, 1);
  gtk_widget_show (check_only_desc);

  check_only_cat =
    gtk_check_button_new_with_label (_("Only items with category"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_only_cat), FALSE);
  gtk_table_attach_defaults (GTK_TABLE (table), check_only_cat, 1, 2, 1, 2);
  gtk_widget_show (check_only_cat);

  /************/

  frame = gtk_frame_new (NULL);
  gtk_frame_set_label (GTK_FRAME (frame), _("Sort by"));
  gtk_frame_set_label_align (GTK_FRAME (frame), 0.0, 0.0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox);
  gtk_container_border_width (GTK_CONTAINER (hbox), 3);
  gtk_widget_show (hbox);

  button_sortbycat = gtk_radio_button_new_with_label (NULL, _("Category"));
  gtk_box_pack_start (GTK_BOX (hbox), button_sortbycat, TRUE, TRUE, 0);
  gtk_widget_show (button_sortbycat);

  group_sort = gtk_radio_button_group (GTK_RADIO_BUTTON (button_sortbycat));
  button_sortbyfile =
    gtk_radio_button_new_with_label (group_sort, _("Disk->Dir->File"));
  gtk_box_pack_start (GTK_BOX (hbox), button_sortbyfile, TRUE, TRUE, 0);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_sortbyfile), TRUE);
  gtk_widget_show (button_sortbyfile);

  group_sort = gtk_radio_button_group (GTK_RADIO_BUTTON (button_sortbycat));
  button_tree = gtk_radio_button_new_with_label (group_sort, _("Tree"));
  gtk_box_pack_start (GTK_BOX (hbox), button_tree, TRUE, TRUE, 0);
  gtk_widget_show (button_tree);

  group_sort = gtk_radio_button_group (GTK_RADIO_BUTTON (button_sortbycat));

  gtk_signal_connect (GTK_OBJECT (button_tree), "clicked",
		      GTK_SIGNAL_FUNC (switch_dirs), include_dir_names);
  gtk_signal_connect (GTK_OBJECT (button_tree), "clicked",
		      GTK_SIGNAL_FUNC (switch_size), check_location);
  switch_size (button_tree, check_location);
  switch_dirs (button_tree, include_dir_names);
  /************/

  frame = gtk_frame_new (NULL);
  gtk_frame_set_label (GTK_FRAME (frame), _("CSV"));
  gtk_frame_set_label_align (GTK_FRAME (frame), 0.0, 0.0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  vbox2 = gtk_vbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  gtk_container_border_width (GTK_CONTAINER (vbox2), 3);
  gtk_widget_show (vbox2);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (vbox2), hbox);
  gtk_container_border_width (GTK_CONTAINER (hbox), 3);
  gtk_widget_show (hbox);

  label = gtk_label_new (_("Separator:"));
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  list_separator = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (hbox), list_separator, FALSE, FALSE, 0);
  gtk_entry_set_text (GTK_ENTRY (list_separator), ",");
  gtk_widget_set_usize (list_separator, 30, -2);
  gtk_widget_show (list_separator);

  check_quote =
    gtk_check_button_new_with_label (_("Put quotes around the fields"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_quote), TRUE);
  gtk_box_pack_start (GTK_BOX (vbox2), check_quote, FALSE, FALSE, 0);
  gtk_widget_show (check_quote);

  gtk_signal_connect (GTK_OBJECT (button_tree), "clicked",
		      GTK_SIGNAL_FUNC (switch_dirs), check_quote);
  switch_dirs (button_tree, check_quote);

  no_newline =
    gtk_check_button_new_with_label (_("Remove the End Of Line characters"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (no_newline), TRUE);
  gtk_box_pack_start (GTK_BOX (vbox2), no_newline, FALSE, FALSE, 0);
  gtk_widget_show (no_newline);

  add_legend = gtk_check_button_new_with_label (_("Add the legend"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (add_legend), TRUE);
  gtk_box_pack_start (GTK_BOX (vbox2), add_legend, FALSE, FALSE, 0);
  gtk_widget_show (add_legend);

  // The TEXT - HTML - XML choice radio buttons
  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (vbox2), hbox);
  gtk_container_border_width (GTK_CONTAINER (hbox), 3);
  gtk_widget_show (hbox);

  button_html = gtk_radio_button_new_with_label (NULL, _("HTML"));
  gtk_box_pack_start (GTK_BOX (hbox), button_html, TRUE, TRUE, 0);
  gtk_widget_show (button_html);

  group_html_xml = gtk_radio_button_group (GTK_RADIO_BUTTON (button_html));
  button_xml = gtk_radio_button_new_with_label (group_html_xml, _("XML"));
  gtk_box_pack_start (GTK_BOX (hbox), button_xml, TRUE, TRUE, 0);
  gtk_widget_show (button_xml);

  group_html_xml = gtk_radio_button_group (GTK_RADIO_BUTTON (button_html));
  button_txt = gtk_radio_button_new_with_label (group_html_xml, _("txt"));
  gtk_box_pack_start (GTK_BOX (hbox), button_txt, TRUE, TRUE, 0);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button_txt), TRUE);
  gtk_widget_show (button_txt);

  gtk_signal_connect (GTK_OBJECT (button_sortbyfile), "clicked",
		      GTK_SIGNAL_FUNC (switch_size), button_html);
  switch_size (button_sortbyfile, button_html);

  gtk_signal_connect (GTK_OBJECT (button_sortbyfile), "clicked",
		      GTK_SIGNAL_FUNC (switch_size), button_xml);

  gtk_signal_connect (GTK_OBJECT (button_sortbyfile), "clicked",
		      GTK_SIGNAL_FUNC (switch_size), button_txt);

  // Don't know the exact implementation
  // Should connect the HTML / XML / txt radio buttons to switch_report_name-function
  // to set the default report-name.
  // gtk_signal_connect (GTK_OBJECT (button_html), "clicked",
  //                  GTK_SIGNAL_FUNC (switch_report_name), "html");

  button_manydirs =
    gtk_check_button_new_with_label (_("dir1/dir2 becomes dir1,dir2"));
  gtk_box_pack_start (GTK_BOX (vbox2), button_manydirs, FALSE, TRUE, 0);
  gtk_widget_show (button_manydirs);

  gtk_signal_connect (GTK_OBJECT (button_sortbyfile), "clicked",
		      GTK_SIGNAL_FUNC (switch_size), button_manydirs);
  //switch_size (button_sortbyfile, button_manydirs);
  /************/
  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox);
  gtk_container_border_width (GTK_CONTAINER (hbox), 3);
  gtk_widget_show (hbox);

  label = gtk_label_new (_("Output File:"));
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);


  gnome_output_entry =
    gnome_file_entry_new (NULL, _("Output file for report"));
  gnome_file_entry_set_modal (GNOME_FILE_ENTRY (gnome_output_entry), TRUE);
  output_entry =
    gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (gnome_output_entry));
  gtk_entry_set_text (GTK_ENTRY (output_entry), report_out =
		      g_strconcat (getenv ("HOME"), "/", DEFAULT_REPORT,
				   ".", DEFAULT_EXTENSION, NULL));
  g_free (report_out);
  gtk_box_pack_start (GTK_BOX (hbox), gnome_output_entry, FALSE, FALSE, 0);
  gtk_widget_show (gnome_output_entry);

  /*******************/
  /* The small button box at the bottom */
  hbox = gtk_hbutton_box_new ();
  gtk_button_box_set_layout (GTK_BUTTON_BOX (hbox), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbox), 0);
  gtk_button_box_set_child_size (GTK_BUTTON_BOX (hbox), 0, 0);

  gtk_container_border_width (GTK_CONTAINER (hbox), 1);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show (hbox);

  my_button = gtk_button_new_with_label (_("Create report"));
  GTK_WIDGET_SET_FLAGS (my_button, GTK_CAN_DEFAULT);
  gtk_window_set_default (GTK_WINDOW (report_dialog_window), my_button);
  gtk_box_pack_start (GTK_BOX (hbox), my_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (my_button), "clicked",
		      GTK_SIGNAL_FUNC (begin_report), racine);
  gtk_widget_show (my_button);

#ifdef GNOME1
  my_button = gnome_stock_button (GNOME_STOCK_BUTTON_CLOSE);
#else
  my_button = gtk_button_new_from_stock (GNOME_STOCK_BUTTON_CLOSE);
#endif
  GTK_WIDGET_SET_FLAGS (my_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (hbox), my_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (my_button), "clicked",
		      G_CALLBACK (cancel_clicked), NULL);
  gtk_widget_show (my_button);
  gtk_widget_add_accelerator (my_button, "clicked", accel_group,
			      GDK_Escape, 0, GTK_ACCEL_VISIBLE);

  gtk_window_add_accel_group (GTK_WINDOW (report_dialog_window), accel_group);

}
