/* modmap.c
 * Written by David Allen <mda@idatar.com>
 * 
 * This file is distributed under the terms of the GNU General Public
 * License.  See COPYING or http://www.gnu.org/ for more details.
 *
 * Routines for the manipulation of XModifierMapping structures.
 * Also contains routines for manipulating Modmap_Table structures,
 * and converting between the two.
 */
/* GTKeyboard - A Graphical Keyboard For X
 * Copyright (C) 1999, 2000 David Allen  
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

#define MODMAP_C

#include "master.h"

/* Sets the modifier mapping server wide to modmap.  Returns false if it
 * fails.
 */
int set_modmap(XModifierKeymap *modmap)
{
     int tries   = 5;
     int timeout = 1;
     int res     = 0;

     while(tries > 0)
     {
          res = XSetModifierMapping(GDK_DISPLAY(), modmap);
          
          if(res == MappingSuccess)
               return 1;
          if(res == MappingFailed)
               break;     /* D'oh!  Fail at the bottom of the function */
          if(res == MappingBusy)
          {
               fprintf(Q,"MappingBusy when trying to map modifiers.\n");
               fprintf(Q,"Sleeping for %d seconds.  PLEASE RELEASE ALL\n",
                       timeout);
               fprintf(Q,"KEYS ON THE KEYBOARD");
               fflush(Q);
               sleep(timeout);
          } /* End if */

          /* Decrement and keep trying */
          tries--;
     } /* End while */

     fprintf(Q,"ERROR:  Failed to set modifier map.\n");
     fflush(Q);
     return(0);
} /* End set_modmap */

/* Returns true if table contains any non-zero modifier keycodes */
int ModmapTable_is_empty(ModmapTable *table)
{
     int x=0;
     int z=0;

     for(z=0; z<8; z++)
     {
          for(x=0; x<table->max_keypermod; x++)
          {
               if(table->modifiers[z].codes[x] != 0)
                    /* Occupied slot found.  It's not empty. */
                    return 0;
          } /* End for */
     } /* End for */

     /* No occupied slots found.  It's empty */
     return 1;
} /* End ModmapTable_is_empty() */

/* Returns true if table contains code, false otherwise. */
int ModmapTable_contains_keycode(ModmapTable *table, KeyCode code)
{
     int x=0, y=0;

     if(code <= 0)
          return 0; /* Illegal value */

     for(x=0; x<8; x++)
     {
          for(y=0; y<table->max_keypermod; y++)
          {
               if(table->modifiers[x].codes[y] == code)
                    return 1;
          } /* End for */
     } /* End for */

     return 0;
} /* End ModmapTable_contains_keycode() */

/* Inserts code into table in the first available slot.  Returns != 0 if 
 * success, 0 on failure.
 */
int ModmapTable_insert_first_empty_slot(ModmapTable *table, KeyCode code)
{
     int slot=0;
     
     if(code <= 0)
     {
          fprintf(Q,"Illegal keycode %d to insert_first_empty_slot\n", code);
          return(0);
     } /* End if */

     if(ModmapTable_contains_keycode(table, code))
     {
          /* It's already in there, so we'll say we inserted it. */
          return 1;   
     }

     for(slot=0; slot<8; slot++)
     {
          if(table->modifiers[slot].codes[0] == 0)
          {
               return ModmapTable_insert(table, code, slot);
          } /* End if */
     } /* End for */

     return 0;
} /* End ModmapTable_insert_first_empty_slot() */

/* Insert KeyCode code into slot slot in the table structure.
 * Returns != 0 on success, 0 on failure.  Failure means that 
 * the table is full, or that the code you're trying to insert is 0
 */
int ModmapTable_insert(ModmapTable *table, KeyCode code, int slot)
{
     int x=0;

     if((code == (KeyCode)0) || (slot < 0) || (slot > 8))
          /* This operation makes no sense.  Return failure. */
          return 0;

     for(x=0; x<table->max_keypermod; x++)
     {
          /* Insert in the first available open slot 
           * but not in taken slots.  That would be a bad idea to 
           * silently overwrite some of the caller's data.  :)
           */
          if(table->modifiers[slot].codes[x] == 0)
          {
               table->modifiers[slot].codes[x] = code;
               return 1;
          } /* End if */
     } /* End for */

     /* Fail - can't find empty slot */
     return(0);
} /* End ModmapTable_insert() */

ModmapTable *ModmapTable_new(void)
{
     XModifierKeymap *map = XGetModifierMapping(GDK_DISPLAY());
     ModmapTable *table;
     int mkpm = map->max_keypermod;
     int x=0;
     int y=0;

     XFreeModifiermap(map);
     table = g_new0_(ModmapTable, 1);
     table->max_keypermod = mkpm;

     for(x=0; x<8; x++)
     {
          for(y=0; y<4; y++)
          {
               table->modifiers[x].codes[y] = (KeyCode)0;
          } /* End for */
     } /* End for */

     return table;
} /* End ModmapTable_new() */

void ModmapTable_destroy(ModmapTable *table)
{
     g_free_(table);
} /* End ModmapTable_destroy() */

/* Translates a string mask name into a slot number for access to numerous
 * modmap data structures.
 */
int mask_name_to_slot_number(char *maskname)
{
     char *masks[] = { "ShiftMask", "LockMask",
                       "ControlMask", "Mod1Mask",
                       "Mod2Mask", "Mod3Mask",
                       "Mod4Mask", "Mod5Mask" };
     int maskcount = 8;
     int y = 0;

     for(y=0; y<maskcount; y++)
     {
          if(g_strcasecmp(maskname, masks[y]) == 0)
               return y;
     } /* End for */
     
     return(-1);
} /* End mask_name_to_slot_number() */

XModifierKeymap *ModmapTable_to_XModifierKeymap(ModmapTable *table)
{
     XModifierKeymap *current_map = XGetModifierMapping(GDK_DISPLAY());
     XModifierKeymap *map = (XModifierKeymap *)NULL;
     int y    = 0;
     int mkpm = 0;
     int x    = 0;

     mkpm = current_map->max_keypermod;
     map = XNewModifiermap(0);

     for(x=0; x<8; x++)
     {
          for(y=0; y<mkpm; y++)
          {
               fprintf(Q,"XInsertModifiermapEntry: code %d sym %s slot %d\n",
                       table->modifiers[x].codes[y],
                       XKeysymToString(
                            XKeycodeToKeysym(GDK_DISPLAY(), 
                                             table->modifiers[x].codes[y],0)),
                       x);
               map = XInsertModifiermapEntry(map, 
                                             table->modifiers[x].codes[y],
                                             x);
          } /* End for */
     } /* End for */

     XFreeModifiermap(current_map);

     return map;
} /* End ModmapTable_to_XModifierKeymap() */

unsigned long find_modifier_mask(KeyCode code) 
{
     XModifierKeymap *map = XGetModifierMapping(GDK_DISPLAY());
     int x=0, y=0;
     KeyCode c=(KeyCode)NULL;
     unsigned long mask = 0;

     if(code == (KeyCode)0) 
     {
          XFreeModifiermap(map);
          fprintf(Q,"Error finding modifier mask for 0 keycode:  Have you\n");
          fprintf(Q,"actually remapped your keyboard to this layout?\n");
          return 0;
     } /* End if */

     for(x=0; x<8; x++) {
          for(y=0; y<map->max_keypermod; y++) {
               c = map->modifiermap[x*map->max_keypermod+y];
               if(c == code) {
                    XFreeModifiermap(map);
                    mask = slot_number_to_mask(x);
                    fprintf(Q,"Found modifier %d in slot (%d,%d) mask %ld 0x%lx\n",
                            code, x, y, mask, mask);
                    return mask;
               } /* End if */
          } /* End for */
     } /* End for */

     XFreeModifiermap(map);

     /* Return nothing.  This is bad, but better than doing the wrong thing. */
     fprintf(Q,"***** WARNING:  find_modifier_mask failed to locate code %d\n",
             code);
     fflush(Q);
     return 0;
} /* End find_modifier_mask() */

/* Returns true if 'other' is the mirror of compare_to, false otherwise.  
 * For example, XK_Shift_L and XK_Shift_R return true.
 */
int is_twin(KeySym compare_to, KeySym other)
{
     char *txt1 = XKeysymToString(compare_to);
     char *txt2 = XKeysymToString(other);
     int txtlen1 = strlen(txt1);
     int txtlen2 = strlen(txt2);

     /* In X often KeySyms have names ending in _R and _L for left and right
      * keys that are essentially the same.  Don't bother telling me this is
      * ugly, I know it is.
      */
     if((txt1[txtlen1-1] == 'R' &&
         txt2[txtlen2-1] == 'L') ||
        (txt1[txtlen1-1] == 'L' &&
         txt2[txtlen2-1] == 'R'))
     {
          return 1;
     } /* End if */
     else 
     { 
          return 0;
     } /* End else */

     return 0; /* Never executed */
} /* End is_twin() */

/* Technically any key can be a modifier in X, but this is covering the
 * usual suspects used in keyboard mappings from different countries.
 */
int is_modifier_key(KeySym s) 
{
#define SYMS_SIZE  20
     KeySym syms[] = { XK_Scroll_Lock, XK_Mode_switch, 
                       XK_Shift_L,     XK_Shift_R,
                       XK_Meta_L,      XK_Meta_R,
                       XK_Alt_L,       XK_Alt_R,
                       XK_Super_L,     XK_Super_R,
                       XK_Hyper_L,     XK_Hyper_R,
                       XK_Shift_L,     XK_Shift_R,
                       XK_Control_L,   XK_Control_R,
                       XK_Shift_Lock,  XK_Caps_Lock,
                       XK_Num_Lock,    XK_Scroll_Lock, 0 };
     int x=0;

     for(x=0; x<20; x++) {
          if(syms[x] == s)
               return 1;
     } /* End while */
                  
     return 0;
} /* End is_modifier_key() */

void x_print_mapping(void) 
{
     XModifierKeymap *map = XGetModifierMapping(GDK_DISPLAY());
     int x=0, y=0;
     KeySym sym=(KeySym)NULL;
     KeyCode c;
     char *txt=(char*)NULL;
     char *strs[8] = { "ShiftMask",
                       "LockMask",
                       "ControlMask",
                       "Mod1Mask",
                       "Mod2Mask",
                       "Mod3Mask",
                       "Mod4Mask",
                       "Mod5Mask" };

     for(x=0; x<8; x++) {
          fprintf(Q,"====== SET #%d Name: %s Mask (Hex/Dec): 0x%lx %ld\n", 
                  x, strs[x], (unsigned long)(1<<x),
                  (unsigned long)(1<<x));
          
          for(y=0; y<map->max_keypermod; y++) { 
               c = map->modifiermap[x*map->max_keypermod+y];
               sym = XKeycodeToKeysym(GDK_DISPLAY(), c, 0);
               txt = XKeysymToString(sym);

               if(c != 0) /* Only print valid entries */
               {
                    fprintf(Q, "Modifier: code %d sym %ld string \"%s\"\n",
                            c, (long)sym, 
                            txt ? txt : "*null*");
               } /* End if */
          } /* End for */
     } /* End while */

     fflush(Q);
     XFreeModifiermap(map);
} /* End x_print_mapping() */

int insert_key_into_modifier_mapping(KeyCode code) 
{
     XModifierKeymap *map=NULL;
     int x=0, y=0;
     KeyCode c = (KeyCode)NULL;
     KeyCode firstinrow = (KeyCode)NULL;

     map = XGetModifierMapping(GDK_DISPLAY());
     
     /* Scan the entire list of items looking for its location. */
     for(x=0; x<8; x++) {
          firstinrow = map->modifiermap[x*map->max_keypermod];
          for(y=0; y<map->max_keypermod; y++) {
               c = map->modifiermap[x*map->max_keypermod+y];
               
               if(c == code) {
                    if(y == 0) {
                         XFreeModifiermap(map);
                         return MODMAP_ALREADY_PRESENT;
                    } /* End if */
                    else if(is_twin(XKeycodeToKeysym(GDK_DISPLAY(), c, 0),
                                    XKeycodeToKeysym(GDK_DISPLAY(), 
                                                     firstinrow, 0))) 
                    {
                         /* If the keycode we're looking for is a twin of 
                          * the one in the first slot, there's no need to 
                          * replace it.  They can occupy the same slot.
                          */
                         printf("Twin can coexist in this slot: %s %s\n",
                                XKeysymToString(XKeycodeToKeysym(GDK_DISPLAY(),
                                                                 firstinrow,
                                                                 0)),
                                XKeysymToString(XKeycodeToKeysym(GDK_DISPLAY(),
                                                                 c, 0)));
                         fflush(stdout);
                         return MODMAP_ALREADY_PRESENT;
                    } /* End else if */
                    else 
                    {
                         fprintf(Q,"Deleting mapping at (%d,%d).\n",
                                 x, y);
                         map = XDeleteModifiermapEntry(map, code, x);
                    } /* End else */
               } /* End if */
               else 
               { 
                    continue;
               } /* End else */
          } /* End for */
     } /* End for */

     /* Now insert it in the first empty slot */
     for(x=0; x<8; x++) {
          /* Only look at the zero slots */
          c = map->modifiermap[x*map->max_keypermod];
          
          if(c == 0) /* Empty */ { 
               map = XInsertModifiermapEntry(map, code, x);
               if(set_modmap(map)) {
                    fprintf(Q,"Successfully mapped modifier at %d\n", x);
                    fflush(Q);
                    XFreeModifiermap(map);
                    return x;
               } else { 
                    XFreeModifiermap(map);
                    return MODMAP_ERROR_INSERTING;
               } /* End else */
          } /* End if */
     } /* End for */

     fprintf(Q,"ERROR:  No empty slots in modmap\n");
     XFreeModifiermap(map);
     return MODMAP_FULL;
} /* End insert_key_into_modifier_mapping() */
