/*
 * Copyright (C) 1994, 1995, 1996 Eric M. Ludlam
 * Copyright (C) 1997 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.org.
 *
 * Purpose:
 *   This file contains the functions used while running a stream socket.
 * The basic two are REMOTESTREAM and LOCALSTREAM functions.  A remote
 * stream is connected to a remote person, and is associated with a
 * USER struct to a local stream, which is directly attached to an
 * emacs process via a TCP socket
 *
 * $Log: gt_strm.c,v $
 * Revision 1.15  1997/12/14 19:17:15  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.14  1997/03/23 14:49:53  zappo
 * uses USER_hangup when streams close which is more robust than old method.
 *
 * Revision 1.13  1997/03/12 00:25:08  zappo
 * Checks auto-clean flag to clean up the user struct.  Moved connection closed
 * message into else block which makes more sense.
 *
 * Revision 1.12  1997/02/23 03:25:53  zappo
 * API change on GT_clean_dev changes, and now pays attention to Ctxt->cleanup
 * feature.
 *
 * Revision 1.11  1997/02/01 14:23:14  zappo
 * Changed USER_send and USER_read to also take Ctxt
 *
 * Revision 1.10  1996/07/28  18:00:46  zappo
 * Replaced send routines with USER_send so encryption is maintained
 *
 * Revision 1.9  1996/02/26  00:16:17  zappo
 * Added ability to grabkeys for kbd proxied subprocesses from emacs
 * input stream
 *
 * Revision 1.8  1996/02/01  02:31:51  zappo
 * Updated to handle SUBPROCESS type user struct
 *
 * Revision 1.7  1995/11/21  03:44:38  zappo
 * Modified to handle datalinked file transfers to end nicely.
 *
 * Revision 1.6  1995/09/29  08:38:24  zappo
 * Upgraded to handle new user state USER_CALLING, and added NO_IFACE
 * ifdefs
 *
 * Revision 1.5  1995/09/20  23:28:23  zappo
 * Made updates needed for use with new structure names
 *
 * Revision 1.4  1995/09/14  10:41:54  zappo
 * Added the hooks in to update the CURSES display when a connection is
 * closed
 *
 * Revision 1.3  1995/03/25  03:56:57  zappo
 * Updateded copyright, and fixed ancient slow-write to non-etalk clients
 *
 * Revision 1.2  1995/01/29  14:45:23  zappo
 * Fixed emacs-hangup printing of status with user id.
 *
 * Revision 1.1  1994/08/29  23:30:17  zappo
 * Initial revision
 *
 * ::Header:: gtalkc.h
 */
#include "gtalklib.h"
#include "gtalkc.h"
#include "gtproc.h"


/*
 * Function: STREAM_local_read
 *
 * When talk is connected, use this to forward characters from emacs
 * process
 * 
 * Parameters: Ctxt - context
 *             io - device to read from
 * History:
 * eml 4/6/94
 */
void STREAM_local_read(Ctxt, io)
     struct TalkContext *Ctxt;
     struct InputDevice *io;
{
  struct UserObject *u, *d;
  char   buffer[100];
  int    rsize;

  /*
   * Always do one read.
   */
  rsize = GT_recv(io, buffer, sizeof(buffer));

  u = USER_iofind(io);

  if((rsize == 1) && (d = USER_finddata(u))) /* only scan for non-blocks */
    {
      /* Check for a grabbed key (this auto-handles it as well) */
      if(FORK_grabkey(Ctxt, buffer[0], d))
	return;
    }

  if(rsize > 0)
    {
      if(!u)
	{
	  printf("Stream is bound to local_read but is not in user struct!\n");
	  return;
	}
      if(!u->remote)
	{
	  /*
	   * If there is no remote, then check to see timeout on remote_connect
	   * and put hyper speed on it so we can reannounce.
	   */
	  if(Ctxt->remote_connect->timeout)
	    {
	      Ctxt->remote_connect->timeout = 1;
	    }
	  return;
	}
      
      USER_send(Ctxt, u, buffer, rsize);
    }
  else
    {
      printf("Connection to %s closed by emacs!\n",
	     u->name);
      
      GT_clean_dev(io, Ctxt->cleanup);

      if((u = USER_iofind(io)) != NULL)
	{
	  printf("\03%c%d\n", TTY_DELETED, u->id);
	  /* delete if userstruct is is currently dialing out, cancel that! */
	  if(u->state == USER_CALLING)
	    PROTOCOL_abort(Ctxt);
	  else
	    /* If attached to someone, just clean up the message */
	    GT_clean_dev(u->remote, Ctxt->cleanup);

	  u->state = USER_CLOSED;

	  if(Ctxt->cleanup == AutoClean) USER_clean();
	}
    }
}

/*
 * Function: STREAM_remote_read
 *
 * When talk is connected, use this to forward characters to emacs process,
 * or to push the characters onto the locally controlled display.
 * 
 * Parameters: Ctxt - context
 *             io - input device to read from
 * History:
 * eml 4/6/94
 */
void STREAM_remote_read(Ctxt, io)
     struct TalkContext *Ctxt;
     struct InputDevice *io;
{
  struct UserObject *u;
  char  *errmsg = "etalk error:  user struct binding error!\n";
  char   buffer[100];
  int    rsize;

  u = USER_iofind(io);

  if(!u)
    {
      DISP_message(Ctxt, "Stream is bound to local_read but is not in user struct!");
      rsize = GT_recv(io, buffer, sizeof(buffer));
      return;
    }

  /* Check, just in case, but this should never happen */
  if(u->remote != io)
    {
      DISP_message(Ctxt, "Something really goofy is going on.");
      rsize = GT_recv(io, buffer, sizeof(buffer));
    }

  rsize = USER_read(Ctxt, u, buffer, sizeof(buffer));
  /* rsize = GT_recv(io, buffer, sizeof(buffer)); */

  if(rsize > 0)
    {
      if((Ctxt->runstate == Socket) && !u->local)
	{
	  GT_send(io, errmsg, strlen(errmsg));
	  printf("\03%s", errmsg);
	  return;
	}
      
      if((u->type != DATALINK) && (u->type != SUBPROCESS))
	/* This routine will scan out any output not meant for display,
	   returning a modified version of buffer */
	SCAN_remote_io(Ctxt, u, buffer, &rsize);

#ifndef NO_IFACE
      /* If in interface mode, datalinks DO get sent to associated
	 io device --> the file */
      if((Ctxt->runstate != Socket) && (u->type != DATALINK) &&
	 (u->type != SUBPROCESS))
	{
	  DISP_remote_string(Ctxt, buffer, rsize, u);
	}
      else
#endif
	GT_send(u->local, buffer, rsize);
    }
  else
    {
      if(u) {
	if((Ctxt->runstate == Socket) && (u->type != DATALINK)
	   && (u->type != SUBPROCESS))
	  {
	    printf("\03%c%d\n", TTY_DELETED, u->id);
	    GT_clean_dev(u->local, Ctxt->cleanup);
	    u->local = NULL;
	  }

	if((u->type == DATALINK) && (u->type != SUBPROCESS))
	  {
	    /* If this is associated with a file, then
	     * fork it, otherwise, don't!
	     */
	    if(u->local->type == IO_FILE)
	      {
		/* Fork it: name field in io dev is the file name.  The io
		   object (cleaned above) still exists until a garbage
		   collection is done */
		if(ASSOC_fork_assoc(Ctxt, u->local->name) == Fail)
		  {
		    DISP_message(Ctxt, "Could not launch viewer!");
		  }
	      }

	    /* Close the file afterwards in case cleanup type was auto.
	     * The hangup is handy because it cleans up all IO and frees
	     * our structures automatically
	     */
	    USER_hangup(Ctxt, u->id);

	    if(verbose)
	      printf("Just closed datalink to %s\n", u->name);

	  } else {
	    sprintf(buffer, "\03Connection to %s closed by remote!",
		    u->name);
	    DISP_message(Ctxt, buffer);
	  }

	u->state = USER_CLOSED;
	u->remote = NULL;

	if(Ctxt->cleanup == AutoClean) USER_clean();

	DISP_update_user(Ctxt, u);
      }

      GT_clean_dev(io, Ctxt->cleanup);
    }  
}

