/*
 * Copyright (C) 2002-2004 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#define GTK_DISABLE_DEPRECATED 1
#define GNOME_DISABLE_DEPRECATED 1

#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dlfcn.h>
#include <math.h>
#include <gnome.h>
#include <glade/glade.h>
#include <config.h>
#include "lib/benchmark.h"
#include "pref.h"
#include "grid.h"
#include "module.h"
#include "action.h"
#include "draw.h"
#include "undo.h"
#include "gui.h"
#include "play.h"
#include "modutils.h"
#include "shell.h"

struct marker_list *cuepoints_clipboard[MAX_TRACKS] = { 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
};
struct marker_list *envelopes_clipboard[MAX_TRACKS] = { 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
};

extern shell *clipboard_shell;
extern int module_count;
extern module modules[];
extern int mem_fail_allocation_on_zero;
extern int quit_requested;
extern int draw_prio;
GList *shells;

void
print_adjust(const char *string,
             GtkAdjustment *adjust) {
    //    DEBUG("%s: value: %f, lower: %f, upper: %f, page_size: %f, page_increment: %f\n",
    //          string, adjust->value, adjust->lower, adjust->upper, adjust->page_size, adjust->page_increment);
}

int
shell_canvas_width_get(shell *shl) {
    return GTK_WIDGET(shl->canvas)->allocation.width;
}

int
shell_height_get(shell *shl) {
    return GTK_WIDGET(shl->canvas)->allocation.height;
}

void 
shell_vadjust_set(shell *shl, 
                  double value) {
    
    if(!shl->sr)
        return;
    shl->vadjust->lower = 0;
    shl->vadjust->upper = shl->sr->channels;
    shl->vadjust->page_size = 1;
    shl->vadjust->step_increment = shl->vadjust->page_increment = 1;
    gtk_adjustment_changed(shl->vadjust);
    gtk_adjustment_value_changed(shl->vadjust);

    /* Show/hide the vertical scrollbar. */

    if(shl->vadjust->value != 0 ||
       shl->sr->channels * shl->vres > shell_height_get(shl)) 
        gtk_widget_show(GTK_WIDGET(shl->vscrollbar));
    else 
        gtk_widget_hide(GTK_WIDGET(shl->vscrollbar));
}

void
shell_hadjust_set(shell *shl, 
                  double value) {
    if(!shl->sr)
        return;

    print_adjust("hadjust_set enter", shl->hadjust);
    shl->hadjust->upper = snd_frame_count(shl->sr);
    shl->hadjust->lower = 0;
    shl->hadjust->step_increment = shl->hres < 1 ? 1 : shl->hres * 10;
    shl->hadjust->page_size = 
        MIN(shl->hadjust->upper,
            floor(shell_canvas_width_get(shl) * shl->hres));
    shl->hadjust->page_increment = 
        MIN(shl->hadjust->upper, 
            MAX(0, shl->hadjust->page_size - shl->hres));
    shl->hadjust->value = 
        CLAMP(rint(value) + shl->hadjust->page_size > shl->hadjust->upper ? 
              shl->hadjust->upper - shl->hadjust->page_increment : rint(value),
              shl->hadjust->lower,
              shl->hadjust->upper);
    gtk_adjustment_changed(shl->hadjust);
    gtk_adjustment_value_changed(shl->hadjust);
    print_adjust("hadjust_set exit", shl->hadjust);
}

void
shell_viewport_center(shell *shl, 
                      AFframecount start,
                      AFframecount end) {
    AFframecount delta = end - start;
    print_adjust("viewport_center enter", shl->hadjust);
    shl->hadjust->value = 
        CLAMP(start, shl->hadjust->lower, shl->hadjust->upper);
    
    if(delta < shell_canvas_width_get(shl) * shl->hres)
        shl->hadjust->value -= 
            MAX(0, ((shell_canvas_width_get(shl) * shl->hres) - delta) / 2);
    
    shell_hadjust_set(shl, shl->hadjust->value);
    print_adjust("viewport_center exit", shl->hadjust);
}

void 
shell_hruler_set(shell *shl) {
    if(shl->sr) 
        gtk_ruler_set_range(GTK_RULER(shl->hruler),
                            snd_frames_to_time(shl->sr, shl->hadjust->value),
                            snd_frames_to_time(shl->sr, shl->hadjust->value + 
                                               (GTK_WIDGET(shl->hruler)->allocation.width) * 
                                               shl->hres),
                            snd_frames_to_time(shl->sr, shl->hadjust->value),
                            shl->sr->time * (shl->hres < 1 ? 
                                             1 / shl->hres : shl->hres));
}

void
shell_hres_set(shell *shl,
               float zoom) {
    AFframecount oldpos = PIXEL_TO_FRAME(shell_canvas_width_get(shl) / 2);
    if(zoom < HRES_MIN)
        zoom = HRES_MIN;
    if(zoom > HRES_MAX)
        zoom = HRES_MAX;
    shl->hres = zoom;
    shell_viewport_center(shl, oldpos, oldpos);
    shell_redraw(shl);
}

void
shell_vres_set(shell *shl,
               int zoom) {
    shl->vres = zoom;
    gtk_widget_queue_draw(GTK_WIDGET(shl->mixercanvas));
    shell_redraw(shl);
}

void 
shell_cursor_set(shell *shl, 
                 GdkCursorType type) {
    gui_window_set_cursor(GTK_WIDGET(shl->canvas)->window, type);
}

void
shell_redraw(shell *shl) {
    if(!shl->sr) 
        return;

    shell_hadjust_set(shl, shl->hadjust->value);
    shell_vadjust_set(shl, shl->vadjust->value);
    gtk_window_set_title(GTK_WINDOW(shl->appwindow), shell_path_strip(shl->sr->name));
    shell_hruler_set(shl);
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
}

char *
shell_path_strip(const char *const_p) {
    char *p = (char *)const_p;
    char *name_ptr;

    if(!p) 
        return NULL;

    for(name_ptr = p + (strlen(p) - 1); 
        name_ptr > p && *name_ptr != '/'; 
        name_ptr--);
    
    if(*name_ptr == '/')
        name_ptr++;
    else
        name_ptr = p;

    return name_ptr;
}

/* Signal/event handlers. */

void
shell_size_allocate_signal(GtkWidget *widget,
                           GtkAllocation *allocation,
                           shell *shl) {

    /* This function attaches to the window's "size_allocate"
       signal. The assumption is that we get here after the drawing
       area's "configure" handler has run (see below). We simply force
       another resize of the drawing area here after processing any
       outstanding events. Otherwise it may happen in some cases that
       the drawing area gets the wrong size; in particular, when I use
       my window manager's maximize function, and if this causes the
       vertical scrollbar to be hidden (because there is now enough
       vertical screen space to show all the tracks), then without
       this code, the drawing area will be sized as if the scrollbar
       was not there. The strange thing is that this problem does not
       happen when resizing a window by its corner. 

       This problem does not occur under GNOME 2 and Sawfish 2. */

    if(quit_requested)
        return;
#ifndef HAVE_GNOME2
    gtk_widget_queue_resize(GTK_WIDGET(shl->canvas));
#endif
}

gboolean 
shell_focus_in_event(GtkWidget *widget,
                     GdkEventFocus *event,
                     shell *shl) {
    return FALSE;
}

gboolean 
shell_focus_out_event(GtkWidget *widget,
                      GdkEventFocus *event,
                      shell *shl) {
    return FALSE;
}

void 
shell_vscrollbar_value_changed(GtkAdjustment *adj, 
                               shell *shl) {
    adj->value = (int) adj->value;
    gtk_widget_queue_draw(GTK_WIDGET(shl->mixercanvas));
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
}

void 
shell_hscrollbar_value_changed(GtkAdjustment *adj, 
                               shell *shl) {
    adj->value = floor(adj->value);
    shell_hruler_set(shl);
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
}

void
shell_show_grid_toggle_activate(GtkMenuItem *menuitem,
                                shell *shl) {
    pref_set_int("show_grid", shl->show_grid ? 0 : 1);
    shl->show_grid = pref_get_as_int("show_grid");
    shell_redraw(shl);
}

void
shell_snap_to_grid_toggle_activate(GtkMenuItem *menuitem,
                                   shell *shl) {
    pref_set_int("snap_to_grid", shl->snap_to_grid ? 0 : 1);
    shl->snap_to_grid = pref_get_as_int("snap_to_grid");
    shell_redraw(shl);
}

void
shell_snap_to_cuepoints_toggle_activate(GtkMenuItem *menuitem,
                                       shell *shl) {
    pref_set_int("snap_to_cuepoints", shl->snap_to_cuepoints ? 0 : 1);
    shl->snap_to_cuepoints = pref_get_as_int("snap_to_cuepoints");
    shell_redraw(shl);
}

void
shell_grid_configure(shell *shl) {

    /* Reset invalid unit entry. */

    if(shl->grid.units != gtk_spin_button_get_value(shl->gridunits))
        gtk_spin_button_set_value(shl->gridunits,
                                  shl->grid.units);
    shell_redraw(shl);
}

void
shell_grid_bpm_set(shell *shl,
                   float bpm) {
    grid_bpm_set(&shl->grid, bpm);
    gtk_spin_button_set_value(shl->gridbpm, shl->grid.bpm);
    shell_grid_configure(shl);
}

void
shell_grid_measurement_set(shell *shl,
                           enum grid_measurement m) {
    grid_measurement_set(&shl->grid, m);
    gtk_option_menu_set_history(shl->gridmeasurement, shl->grid.measurement);
    if(m == GRID_BEATS) 
        gtk_widget_show(GTK_WIDGET(shl->gridbpm));
    else
        gtk_widget_hide(GTK_WIDGET(shl->gridbpm));
    shell_grid_configure(shl);
}

void
shell_grid_units_set(shell *shl,
                     float units) {
    grid_units_set(&shl->grid, units);
    gtk_spin_button_set_value(shl->gridunits, shl->grid.units);
    shell_grid_configure(shl);
}

void
shell_grid_bpm_changed(GtkSpinButton *spinbutton,
                       shell *shl) {
    grid_bpm_set(&shl->grid,
                 gtk_spin_button_get_value(spinbutton));
    shell_grid_configure(shl);
}

void
shell_grid_units_changed(GtkSpinButton *spinbutton,
                         shell *shl) {
    grid_units_set(&shl->grid, 
                   gtk_spin_button_get_value(spinbutton));
    shell_grid_configure(shl);
}

void
shell_grid_measurement_changed(GtkOptionMenu *optionmenu,
                               shell *shl) {
    enum grid_measurement m = gui_option_menu_get_active(shl->gridmeasurement);
    shell_grid_measurement_set(shl, m);
}

gint 
shell_mixercanvas_configure_event(GtkWidget *widget, 
                                  GdkEventConfigure *event,
                                  shell *shl) {
    if(shl->mixerpixmap)
        gdk_pixmap_unref(shl->mixerpixmap);

    shl->mixerpixmap = gdk_pixmap_new(widget->window,
                                      widget->allocation.width,
                                      widget->allocation.height,
                                      -1);
    return TRUE;

}

gint 
shell_canvas_configure_event(GtkWidget *widget, 
                             GdkEventConfigure *event,
                             shell *shl) {
    size_t gb_sz;

    if(shl->pixmap)
        gdk_pixmap_unref(shl->pixmap);

    if(shl->graph_bits_buffer_low)
        mem_free(shl->graph_bits_buffer_low);
    //    if(shl->graph_bits_buffer_high)
    //        mem_free(shl->graph_bits_buffer_high);
    shl->graph_bits_buffer_low = NULL;
    shl->graph_bits_buffer_high = NULL;

    shl->pixmap = gdk_pixmap_new(widget->window,
                                 widget->allocation.width,
                                 widget->allocation.height,
                                 -1);

    gb_sz = (size_t)widget->allocation.width * sizeof(graph_bits_unit_t) * 2;
    DEBUG("gb_sz: %d\n", gb_sz);
    shl->graph_bits_buffer_low = mem_calloc(1, gb_sz);

    if(!shl->graph_bits_buffer_low) {
        FAIL("not enough memory to allocate peak buffer (%d bytes)\n",
             gb_sz);
        return TRUE;
    }
    
    shl->graph_bits_buffer_high = (graph_bits_unit_t *)((int8_t *)shl->graph_bits_buffer_low + (gb_sz / 2));

    /* for caching waveform bitmap?
       if(shl->graph_pixmap)
       gdk_pixmap_unref(shl->graph_pixmap);

       shl->graph_pixmap = gdk_pixmap_new(widget->window,
       widget->allocation.width,
       snd_frame_count(shl->sr) * shl->vres,
       -1);
    */

    shell_vadjust_set(shl, shl->vadjust->value);
    shell_hadjust_set(shl, shl->hadjust->value);
    shell_hruler_set(shl);

    return TRUE;

}

gboolean 
shell_infocanvas_expose_event(GtkWidget *widget, 
                              GdkEventExpose *event, 
                              shell *shl) {
    draw_info_canvas(widget, &event->area, shl);
    return TRUE;
}

gboolean 
shell_mixercanvas_expose_event(GtkWidget *widget, 
                                   GdkEventExpose *event, 
                                   shell *shl) {
    draw_mixer_canvas(widget, &event->area, shl);
    return TRUE;
}

gboolean 
shell_mixercanvas_button_press_event_toggles(GtkWidget *widget,
                                             GdkEventButton *event,
                                             shell *shl) {
    int index_rel = event->y / shl->vres;
    int index_abs = index_rel + shl->vadjust->value;
    int rel_y = (event->y - index_rel * shl->vres) - 1;
    if(rel_y > 
       MIXER_TOGGLES_MUTE_HEIGHT + MIXER_TOGGLES_SOLO_HEIGHT + 4)
        return FALSE;
    
    if(rel_y > MIXER_TOGGLES_MUTE_HEIGHT + 2)
        mixer_toggle_source_solo(shl->mixer, index_abs);
    else {
        if(shl->mixer->num_solos)
            mixer_toggle_source_solo(shl->mixer, index_abs);
        else
            mixer_toggle_source_mute(shl->mixer, index_abs);
    }
    gtk_widget_queue_draw(GTK_WIDGET(shl->mixercanvas));
    return FALSE;
}

void
shell_mixercanvas_adjust_level(shell *shl,
                               double x) {
    double mixervalue, rel_x;
    if(x > (MIXER_LEVEL_OFFSET_X + 8) - 1 && 
       x < (MIXER_LEVEL_OFFSET_X + 8 + MIXER_LEVEL_WIDTH)) {
        rel_x = x - (MIXER_LEVEL_OFFSET_X + 8);
        mixervalue = (double)rel_x / MIXER_LEVEL_WIDTH;
    } else if(x < MIXER_LEVEL_OFFSET_X + 8) {
        mixervalue = 0;
    } else {
        mixervalue = 1;
    }
    DEBUG("think target mix value %f\n", mixervalue);
    shl->mixer->mixtable[shl->target_channel_being_dragged]
        [shl->source_channel_being_dragged] = mixervalue;
}

gboolean 
shell_mixercanvas_button_press_event(GtkWidget *widget,
                                     GdkEventButton *event,
                                     shell *shl) {
    int index_rel = event->y / shl->vres;
    int index_abs = index_rel + shl->vadjust->value;
    int dst_track;
    int y = event->y;
    int mix_lvl_height = MIXER_LEVEL_HEIGHT;

    if(event->x < MIXER_LEVEL_OFFSET_X) 
        return shell_mixercanvas_button_press_event_toggles(widget, event, shl);

    gtk_widget_grab_focus(widget);

    y -= (index_rel * shl->vres) + 1;

    DEBUG("press on track %d, remainder: %d\n", index_abs, y);
    shl->source_channel_being_dragged = -1;
    shl->target_channel_being_dragged = -1;

   
    if(index_abs < 0 || index_abs >= shl->sr->channels)
        return TRUE;

    if(shl->mixer->target_channels * mix_lvl_height > shl->vres) 
        mix_lvl_height /= 2;

    dst_track = y / (mix_lvl_height + 1);
    
    if(dst_track < 0 || dst_track >= shl->mixer->target_channels)
        return TRUE;

    shl->source_channel_being_dragged = index_abs;
    shl->target_channel_being_dragged = dst_track;

    shell_mixercanvas_adjust_level(shl, event->x);
    gtk_widget_queue_draw(GTK_WIDGET(shl->mixercanvas));
    return TRUE;
}

gboolean
shell_mixercanvas_motion_notify_event(GtkWidget *widget,
                                      GdkEventMotion *event,
                                      shell *shl) {
    if(shl->source_channel_being_dragged == -1 ||
       shl->target_channel_being_dragged == -1)
        return TRUE;

    shell_mixercanvas_adjust_level(shl, event->x);
    
    gtk_widget_queue_draw(GTK_WIDGET(shl->mixercanvas));
    return TRUE;
}

gboolean 
shell_mixercanvas_button_release_event(GtkWidget *widget,
                                       GdkEventButton *event,
                                       shell *shl) {
    //    DEBUG("release\n");
    shl->source_channel_being_dragged = -1;
    shl->target_channel_being_dragged = -1;
    shl->scroll_drag = 0;
    return TRUE;
}

gboolean 
shell_canvas_expose_event(GtkWidget *widget, 
                          GdkEventExpose *event, 
                          shell *shl) {
    draw_canvas(widget, &event->area, shl);
    return TRUE;
}

struct marker *
shell_find_nearest_marker(shell *shl,
                          int track,
                          AFframecount frame_offset,
                          enum marker_type type) {
    struct marker *mn, *mp, *m = NULL;
    DEBUG("finding marker in neighbourhood %ld\n",
          frame_offset);
    mn = marker_list_next(shl->sr->markers[track],
                          MAX(0, frame_offset - 1),
                          type);
    mp = marker_list_previous(shl->sr->markers[track],
                              MAX(0, frame_offset - 1),
                              type);
    if(mn && !mp)
        m = mn;
    else if(mp && !mn)
        m = mp;
    else if(mp && mn)
        m = DISTANCE(frame_offset, mp->frame_offset) > 
            DISTANCE(frame_offset, mn->frame_offset) ?
            mn : mp;
    return m;
}

gboolean
shell_process_envelope_click(shell *shl,
                             enum marker_type type,
                             int x,
                             int y) {
    double multiplier, mytol, mxtol;
    struct marker *m = NULL;
    int track_bit = y < 0 ? -1 : (y / shl->vres) + shl->vadjust->value;

    /* Find nearest envelope marker. */
    
    m = shell_find_nearest_marker(shl,
                                  track_bit,
                                  PIXEL_TO_FRAME(x),
                                  type);
    multiplier = (-((y - ((track_bit - shl->vadjust->value) * shl->vres)) -
                    (shl->vres / 2)) / (shl->vres / 2));
    if(m) {
        DEBUG("nearest marker:\n");
        marker_dump(m);
        mxtol = shl->hres * (MARKER_HANDLE_SIZE / 2);
        if(PIXEL_TO_FRAME(x) > m->frame_offset - mxtol &&
           PIXEL_TO_FRAME(x) < m->frame_offset + mxtol) {
            DEBUG("is in marker handle x range\n");
            mytol = ((double)1 / (shl->vres / 2)) * (MARKER_HANDLE_SIZE / 2);
            if(multiplier > m->multiplier - mytol &&
               multiplier < m->multiplier + mytol) {
                DEBUG("is in marker handle y range, dragging marker on track %d\n",
                      track_bit);
                shl->marker_being_dragged = m;
                shl->marker_dragged_on_track = track_bit;
                return TRUE;
            }
        }
    }
    
    DEBUG("no marker in the neighbourhood or not in range: x: %d, y: %d\n",
          x, y);
    shl->marker_being_dragged = marker_list_insert(shl->sr->markers[track_bit],
                                                   PIXEL_TO_FRAME(x),
                                                   type,
                                                   multiplier,
                                                   NULL);
    DEBUG("new marker: %p, x: %ld, y: %f, track: %d\n", 
          shl->marker_being_dragged, (AFframecount)PIXEL_TO_FRAME(x), multiplier, track_bit);
    if(shl->marker_being_dragged)
        shl->marker_dragged_on_track = track_bit;
    marker_list_dump(shl->sr->markers[track_bit]);
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
    return TRUE;
}

gboolean
shell_process_cuepoint_click(shell *shl,
                             struct marker *m,
                             GdkEventButton *event) {
    double mxtol;
    int track_bit = event->y < 0 ? -1 : (event->y / shl->vres) + shl->vadjust->value;

    marker_dump(m);
    mxtol = shl->hres * MARKER_CUEPOINT_HANDLE_SIZE;
    if(PIXEL_TO_FRAME(event->x) > m->frame_offset &&
       PIXEL_TO_FRAME(event->x) < m->frame_offset + mxtol) {
        DEBUG("is in marker handle x range\n");
        if(((event->y / (float)shl->vres) + shl->vadjust->value) - track_bit > 0.9) {
            DEBUG("is in marker handle y range\n");
            shl->marker_being_dragged = m;
            shl->marker_dragged_on_track = track_bit;
            if(event->type == GDK_2BUTTON_PRESS) {
                gtk_entry_set_text(shl->marker_label, m->label);
                gtk_widget_show(GTK_WIDGET(shl->marker_dialog));
            }
            return TRUE;
        }
    }
    return FALSE;
}

void
shell_zoom_in_activate(GtkWidget *w,
                       shell *shl);

void
shell_zoom_out_activate(GtkWidget *w,
                        shell *shl);

void
shell_smoothscroll(shell *shl,
                   AFframecount from,
                   AFframecount to) {
    double ratios[] = { .05, .15, .4, .25, .125 };
    int i, sign = (from - to) > 0 ? 1 : -1;
    struct timespec ts = { 0, 50000 };
    AFframecount delta = MAX(from, to) - MIN(from, to);
    AFframecount pos = from;
    AFframecount amount;
    DEBUG("from: %ld, to: %ld\n", from, to);
    for(i = 0; i < sizeof ratios / sizeof ratios[0]; i++) {
        amount = ratios[i] * delta;
        pos -= sign * amount;
        DEBUG("amount: %ld, new pos: %ld\n", amount, pos);
        shell_viewport_center(shl, pos, pos);
        gui_yield();
        nanosleep(&ts, NULL);
    }
    shell_viewport_center(shl, to, to);
}

void
shell_mouse_wheel_event(shell *shl,
                        double x,
                        int direction) {
    struct timeval now;
    AFframecount pos = PIXEL_TO_FRAME(x);
    int x_treshold = pref_get_as_int("wheel_x_treshold");
    float time_treshold = pref_get_as_float("wheel_time_treshold");

    gettimeofday(&now, NULL);
    DEBUG("timeval difference: %f, x difference: %d, time treshold: %f, x treshold: %d\n",
          TIMEVAL_DIFF(shl->last_wheel_movement, now), 
          (int)(shl->last_wheel_x - x), time_treshold, x_treshold);

    /*
     * Only recenter when time and/or pixel treshold is crossed.  This
     * way you can zoom in/out the same spot but still use the pointer
     * position to select another spot for zoom.
     */

    if(shl->last_wheel_x == -1 ||
       abs(shl->last_wheel_x - x) > x_treshold) {
        DEBUG("recentering on %ld\n", pos);
        gettimeofday(&shl->last_wheel_movement, NULL);
        shl->last_wheel_pos = pos;
        shl->last_wheel_x = x;
        shell_smoothscroll(shl, 
                           (shell_canvas_width_get(shl) * shl->hres / 2) + 
                           shl->hadjust->value, pos);
        return;
    }
    if(direction == WHEEL_FORWARD) {
        shell_zoom_in_activate(NULL, shl);
    } else {
        shell_zoom_out_activate(NULL, shl);
    }
}

gboolean 
shell_canvas_button_press_event(GtkWidget *widget,
                                GdkEventButton *event,
                                shell *shl) {
    AFframecount ss, se;
    int track_bit, sc = shl->select_channel_map;
    enum marker_type type = MARKER_SLOPE;
    struct marker *m;

    if(!shl->sr)
        return FALSE;

    /* Mouse wheel? */

    if(event->button == 4 || event->button == 5) {
        shell_mouse_wheel_event(shl, event->x,
                                event->button == 4 ?
                                WHEEL_FORWARD : WHEEL_BACKWARD);
        return FALSE;
    }

    track_bit = event->y < 0 ? -1 : (event->y / shl->vres) + 
        shl->vadjust->value;

    if(track_bit > shl->sr->channels - 1)
        return FALSE;

    /* Check for pending marker drag on previous button
       press/motion. */

    if(shl->marker_being_dragged) {
        DEBUG("releasing marker\n");
        if((shl->marker_being_dragged->flags & MARKER_IS_DISABLED) == 
           MARKER_IS_DISABLED) {
            DEBUG("marker disabled, deleting marker\n");
            marker_list_marker_delete(shl->sr->markers[shl->marker_dragged_on_track],
                                      shl->marker_being_dragged);
        }
        shl->marker_being_dragged = NULL;
        shl->marker_dragged_on_track = -1;
    }

    shl->source_channel_being_dragged = -1;
    shl->target_channel_being_dragged = -1;

    /* Second mouse button selects MARKER_SLOPE_AUX. */

    if(event->button == 2)
        type = MARKER_SLOPE_AUX;

    if(shl->envelope_enabled) 
        return shell_process_envelope_click(shl,
                                            type,
                                            event->x,
                                            event->y);

    m = shell_find_nearest_marker(shl,
                                  track_bit,
                                  PIXEL_TO_FRAME(event->x),
                                  MARKER_TEXT);
    if(m && shell_process_cuepoint_click(shl, m, event))
        return TRUE;
    
    /* Waveform selection */
    /* Click + control key modifies which channels are selected. */

    if(event->state & GDK_CONTROL_MASK) {
        shl->select_channel_map = (shl->select_channel_map & (1 << track_bit)) ? 
            shl->select_channel_map & ~(1 << track_bit) : 
            shl->select_channel_map | (1 << track_bit);
        gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
        return TRUE;
    }
    
    shl->select_flex = PIXEL_TO_FRAME(event->x);
    
    if(shl->select_flex < 0 || shl->select_flex > snd_frame_count(shl->sr))
        return FALSE;
    
    shl->select_pivot = (DISTANCE(shl->select_flex, shl->select_start) >
                         DISTANCE(shl->select_flex, shl->select_end) ? 
                         shl->select_start : shl->select_end);
    
    if(!(event->state & GDK_SHIFT_MASK)) {
        
        sc = (1 << track_bit);
        shl->select_pivot = shl->select_flex = PIXEL_TO_FRAME(event->x);
        
    }

    /* Using an action here gives us undo. */

    ss = MIN(shl->select_pivot, shl->select_flex);
    se = MAX(shl->select_pivot, shl->select_flex);
    
    action_do(ACTION_SELECT_NEW(WITH_UNDO,
                                shl,
                                shl->sr,
                                sc,
                                ss,
                                se - ss));
    return TRUE;
}

void
shell_drag_select(shell *shl,
                  AFframecount flex) {
    
    shl->select_flex = flex;
    
    if(shl->select_flex < 0)
        shl->select_flex = 0;
    if(shl->select_flex > snd_frame_count(shl->sr))
        shl->select_flex = snd_frame_count(shl->sr);
    
    shl->select_start = MIN(shl->select_pivot, shl->select_flex);
    shl->select_end = MAX(shl->select_pivot, shl->select_flex);
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
}

void
shell_drag_marker(shell *shl,
                  int x,
                  int y) {
    int track_bit =  y < 0 ? -1 : (y / shl->vres) + shl->vadjust->value;
    double multiplier =  (-((y - ((track_bit - shl->vadjust->value) * shl->vres)) -
                            (shl->vres / 2)) / (shl->vres / 2));
    long long mpos = PIXEL_TO_FRAME(x);
    
    if(shl->snap_to_grid) 
        mpos -= mpos % shl->grid.gap;
    if(mpos < 0)
        mpos = 0;

    shl->marker_being_dragged->multiplier = multiplier;
    marker_list_marker_position_set(shl->sr->markers[shl->marker_dragged_on_track],
                                    shl->marker_being_dragged,
                                    mpos);
    if(track_bit != shl->marker_dragged_on_track) 
        shl->marker_being_dragged->flags |= MARKER_IS_DISABLED;
    else
        shl->marker_being_dragged->flags &= ~MARKER_IS_DISABLED;
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
}

gint
shell_canvas_keep_dragging(gpointer userdata);

void
shell_canvas_check_scroll(shell *shl,
                          int x_root) {
    int width, height, abs_x, abs_y, delta;

    /* 
     * See if x_root is outside our canvas. If so we set a timer to 
     * shell_canvas_keep_dragging 25 ms. from now, which in turn
     * invokes us again.
     */
    
    gdk_window_get_size(gui_get_window(GTK_WIDGET(shl->canvas)), &width, &height);
    gui_get_widget_position_absolute(GTK_WIDGET(shl->canvas), &abs_x, &abs_y);
    delta = x_root - abs_x;
    //    DEBUG("delta: %d, abs_x: %d, x_root: %d, width: %d, scroll_drag %d\n", delta, abs_x, x_root, width, shl->scroll_drag);
    if(delta < 0 || delta - width > 0) {
        if(delta > 0)
            delta -= width;
        shl->last_mouse_x_root = x_root;
        shell_hadjust_set(shl, shl->hadjust->value + ((delta / 4) * shl->hres));
        if(!shl->scroll_drag)
            gtk_timeout_add(25, shell_canvas_keep_dragging, shl);
        shl->scroll_drag = 1;
    } else
        shl->scroll_drag = 0;

}

gint
shell_canvas_keep_dragging(gpointer userdata) {
    shell *shl = (shell *)userdata;

    if(!shl->scroll_drag)
        return FALSE;

    shell_canvas_check_scroll(shl, shl->last_mouse_x_root);

    if(shl->marker_being_dragged)
        shell_drag_marker(shl, shl->last_mouse_x, shl->last_mouse_y);
    else
        shell_drag_select(shl, PIXEL_TO_FRAME(shl->last_mouse_x));

    if(!shl->scroll_drag)
        return FALSE;
    
    return TRUE;
}

gboolean
shell_canvas_motion_notify_event(GtkWidget *widget,
                                 GdkEventMotion *event,
                                 shell *shl) {

    /* User came from mixer canvas, is meaningless here. */

    if(shl->source_channel_being_dragged != -1)
        return FALSE;

    shl->last_mouse_x = event->x;
    shl->last_mouse_y = event->y;

    gtk_widget_queue_draw(GTK_WIDGET(shl->infocanvas));

    if(!shl->sr) 
        return FALSE;

    if(!((event->state & GDK_BUTTON1_MASK) ||
         (event->state & GDK_BUTTON2_MASK)))
        return FALSE;

    if(event->state & GDK_CONTROL_MASK) 
        return FALSE;
    
    shell_canvas_check_scroll(shl, event->x_root);

    /* Dragging a marker. */
    
    if(shl->marker_being_dragged) {
        shell_drag_marker(shl, event->x, event->y);
        return TRUE;
    }
    
    if(shl->envelope_enabled) 
        return FALSE;

    /* Set selection. */

    shell_drag_select(shl, PIXEL_TO_FRAME(event->x));    
    return TRUE;
}
    
gboolean 
shell_canvas_button_release_event(GtkWidget *widget,
                                  GdkEventButton *event,
                                  shell *shl) {
    shl->scroll_drag = 0;

    if(!shl->sr)
        return FALSE;

    /* Mouse wheel release. */

    if(event->button == 4 || event->button == 5) 
        return FALSE;

    /* User came from mixer canvas, is meaningless here. */

    if(shl->source_channel_being_dragged != -1)
        return FALSE;

    if(shl->envelope_enabled)
        return FALSE;

    if(event->state & GDK_CONTROL_MASK)
        return FALSE;

    if(event->y > (shl->sr->channels - shl->vadjust->value) * shl->vres)
        return FALSE;

    if(shl->marker_being_dragged)
        return FALSE;

    shell_drag_select(shl, PIXEL_TO_FRAME(event->x));

    return TRUE;
}

void
shell_show_zero_line_toggle_activate(GtkMenuItem *menuitem,
                                     shell *shl) {
    pref_set_int("show_zero", shl->show_zero ? 0 : 1);
    shl->show_zero = pref_get_as_int("show_zero");
    shell_redraw(shl);
}

void
shell_show_envelope_toggle_activate(GtkMenuItem *menuitem,
                                    shell *shl) {
    int i;
    pref_set_int("show_envelope", shl->show_envelope ? 0 : 1);
    shl->show_envelope = pref_get_as_int("show_envelope");
    if(shl->show_envelope)
        for(i = 0; i < pref_get_as_int("invariant_max_tracks"); i++)
            shl->sr->markers[i]->marker_types_enabled = 
                MARKER_SLOPE | MARKER_SLOPE_AUX;
    else
        for(i = 0; i < pref_get_as_int("invariant_max_tracks"); i++)
            shl->sr->markers[i]->marker_types_enabled = 0;
    
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));    
}

void
shell_envelope_enable_toggle_activate(GtkMenuItem *menuitem,
                                      shell *shl) {
    shl->envelope_enabled = shl->envelope_enabled ? 0 : 1;
    DEBUG("envelope_enabled: %d\n", shl->envelope_enabled);
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));    
}

void
shell_play_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    if(shl->player.player_running) 
        action_do(ACTION_PLAYER_STOP_NEW(shl));
    else { 
        shl->record_mode = 0;
        action_do(ACTION_PLAYER_PLAY_NEW(shl));
    }
}

void
shell_stop_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    action_do(ACTION_PLAYER_STOP_NEW(shl));
}

void
shell_cue_play_activate(GtkMenuItem *menuitem,
                        shell *shl) {
    AFframecount 
        s = LOOP_IS_ACTIVE(shl) ? shl->loop_start : shl->select_start;
    shl->player.player_pos = s;
    shell_viewport_center(shl, shl->player.player_pos, 
                          shl->player.player_pos);
    
    if(!shl->player.player_running)
        shell_play_activate(menuitem, shl);
}

void
shell_scrub_right(shell *shl) {
    shl->player.player_pos = MIN(snd_frame_count(shl->sr),
                                 shl->player.player_pos + SCRUB_AMOUNT);
    shell_viewport_center(shl, shl->player.player_pos, 
                          shl->player.player_pos);
}

void
shell_scrub_left(shell *shl) {
    shl->player.player_pos = 
        MAX(0, shl->player.player_pos - SCRUB_AMOUNT);
    shell_viewport_center(shl, shl->player.player_pos, shl->player.player_pos);
}

gint
shell_keep_scrubbing_left(gpointer userdata) {
    shell *shl = (shell *)userdata;
    if(shl->scrubbing) {
        shell_scrub_left(shl);
        return TRUE;
    }
    return FALSE;
}

gint
shell_keep_scrubbing_right(gpointer userdata) {
    shell *shl = (shell *)userdata;
    if(shl->scrubbing) {
        shell_scrub_right(shl);
        return TRUE;
    }
    return FALSE;
}

void
shell_scrub_left_pressed(GtkButton *button,
                         shell *shl) {
    shl->scrubbing = 1;
    gtk_timeout_add(50, shell_keep_scrubbing_left, shl);
    shell_scrub_left(shl);
}

void
shell_scrub_right_pressed(GtkButton *button,
                          shell *shl) {
    shl->scrubbing = 1;
    gtk_timeout_add(50, shell_keep_scrubbing_right, shl);
    shell_scrub_right(shl);
}

void
shell_scrub_released(GtkButton *button,
                     shell *shl) {
    shl->scrubbing = 0;
}

void
shell_scrub_right_activate(GtkMenuItem *menuitem,
                           shell *shl) {
    shell_scrub_right(shl);
}

void
shell_scrub_left_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    shell_scrub_left(shl);
}

void
shell_loop_toggle_activate(GtkMenuItem *menuitem,
                           shell *shl) {
    shl->loop = shl->loop ? 0 : 1;
    shell_redraw(shl);
}

void
shell_record_replace_toggle_activate(GtkMenuItem *menuitem,
                                     shell *shl) {
    pref_set_int("record_replace", shl->record_replace ? 0 : 1);
    shl->record_replace = pref_get_as_int("record_replace");
    shell_redraw(shl);
}

void
shell_record_activate(GtkMenuItem *menuitem,
                      shell *shl) {
    shl->record_mode = 1;
    if(shl->player.player_running) 
        action_do(ACTION_PLAYER_STOP_NEW(shl));
    else 
        action_do(ACTION_PLAYER_PLAY_NEW(shl));
}

void
shell_follow_playback_toggle_activate(GtkMenuItem *menuitem,
                                      shell *shl) {
    pref_set_int("follow_playback", shl->follow_playback ? 0 : 1);
    shl->follow_playback = pref_get_as_int("follow_playback");
    shell_redraw(shl);
}

void
shell_loop_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    shl->loop = shl->loop ? 0 : 1;
}

void 
shell_undo_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    if(!shl->sr)
        return;

    shl->undo_stack = undo_pop(shl->undo_stack);
    shell_status_default_set(shl);
    shell_redraw(shl);
}

void
shell_mixdown_activate(GtkWidget *w,
                       shell *shl) {
    action_do(ACTION_FILE_MIXDOWN_NEW(shl));
}

void
shell_saveas_activate(GtkWidget *w,
                      shell *shl) {
    action_do(ACTION_FILE_SAVE_AS_NEW(shl));
}

void
shell_save_activate(GtkWidget *w,
                    shell *shl) {
    action_do(ACTION_FILE_SAVE_NEW(shl));
}

void 
shell_copy_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    action_do(ACTION_COPY_NEW(DONT_UNDO,
                              shl,
                              shl->sr,
                              shl->select_channel_map,
                              shl->select_start,
                              shl->select_end - shl->select_start));
}

void 
shell_paste_activate(GtkMenuItem *menuitem,
                     shell *shl) {
    action_do(ACTION_PASTE_NEW(WITH_UNDO,
                               shl,
                               shl->sr,
                               shl->select_channel_map,
                               shl->select_start,
                               shl->select_end - shl->select_start));
}

void 
shell_paste_fit_activate(GtkMenuItem *menuitem,
                         shell *shl) {
    action_do(ACTION_PASTE_FIT_NEW(WITH_UNDO,
                                   shl,
                                   shl->sr,
                                   shl->select_channel_map,
                                   shl->select_start,
                                   shl->select_end - shl->select_start));
}

void 
shell_paste_mix_activate(GtkMenuItem *menuitem,
                         shell *shl) {
    action_do(ACTION_PASTE_MIX_NEW(WITH_UNDO,
                                   shl,
                                   shl->sr,
                                   shl->select_channel_map,
                                   shl->select_start,
                                   shl->select_end - shl->select_start));
}

void 
shell_paste_over_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    action_do(ACTION_PASTE_OVER_NEW(WITH_UNDO,
                                    shl,
                                    shl->sr,
                                    shl->select_channel_map,
                                    shl->select_start,
                                    shl->select_end - shl->select_start));
}

void
shell_insert_tracks_activate(GtkMenuItem *menuitem,
                             shell *shl) {
    int map = 0, i, j = shl->sr->channels;

    for(i = 0; i < shl->sr->channels; i++) 
        if((1 << i) & shl->select_channel_map)
            map |= (1 << j);

    action_group_do(action_group_new(2,
                                     ACTION_TRACKS_INSERT_NEW(WITH_UNDO,
                                                              shl,
                                                              shl->sr,
                                                              NULL,
                                                              map),
                                     ACTION_SELECT_NEW(WITH_UNDO,
                                                       shl,
                                                       shl->sr,
                                                       map,
                                                       shl->select_start,
                                                       shl->select_end - shl->select_start)));
}

void 
shell_delete_tracks_activate(GtkMenuItem *menuitem,
                             shell *shl) {
    action_do(ACTION_TRACKS_DELETE_NEW(WITH_UNDO,
                                       shl,
                                       shl->sr,
                                       shl->select_channel_map));
}

void
shell_cut_activate(GtkMenuItem *menuitem,
                   shell *shl) {
    action_do(ACTION_CUT_NEW(WITH_UNDO,
                             shl,
                             shl->sr,
                             shl->select_channel_map,
                             shl->select_start,
                             shl->select_end - shl->select_start));
}

void 
shell_clear_activate(GtkMenuItem *menuitem,
                     shell *shl) {
    action_do(ACTION_DELETE_NEW(WITH_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                shl->select_start,
                                shl->select_end - shl->select_start));
}

void 
shell_erase_activate(GtkMenuItem *menuitem,
                     shell *shl) {
    action_do(ACTION_ERASE_NEW(WITH_UNDO,
                               shl,
                               shl->sr,
                               shl->select_channel_map,
                               shl->select_start,
                               shl->select_end - shl->select_start));

}

void 
shell_crop_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    action_group *ag;
    action *a;
    ag = action_group_new(0);
    a = ACTION_DELETE_NEW(WITH_UNDO, 
                          shl,
                          shl->sr,
                          shl->select_channel_map,
                          shl->select_end,
                          snd_frame_count(shl->sr) - shl->select_end);
    ag = action_group_append(ag, a);
    a = ACTION_DELETE_NEW(WITH_UNDO,
                          shl,
                          shl->sr,
                          shl->select_channel_map,
                          0,
                          shl->select_start);
    ag = action_group_append(ag, a);
    a = ACTION_SELECT_NEW(WITH_UNDO,
                          shl,
                          shl->sr,
                          shl->select_channel_map,
                          0,
                          shl->select_end - shl->select_start);
    ag = action_group_append(ag, a);
    action_group_do(ag);
}

void
shell_copy_markers(shell *shl,
                   struct marker_list *marker_clipboard[],
                   enum marker_type type) {
    int t, i = 0;
    if(shl->select_start == shl->select_end)
        return;
    for(t = 0; t < MAX_TRACKS; t++) {
        if(marker_clipboard[t]) {
            marker_list_destroy(marker_clipboard[t]);
            marker_clipboard[t] = NULL;
        }
    }

    for(t = 0; t < shl->sr->channels; t++) 
        if((1 << t) & shl->select_channel_map) 
            marker_clipboard[i++] = 
                marker_list_clone(shl->sr->markers[t],
                                  shl->select_start,
                                  shl->select_end - shl->select_start,
                                  type);
}

void
shell_paste_markers(shell *shl,
                    struct marker_list *marker_clipboard[]) {
    if(shl->select_start == shl->select_end)
        return;
    action_do(ACTION_MARKER_LIST_INSERT_NEW(WITH_UNDO,
                                            shl,
                                            shl->sr->markers, 
                                            marker_clipboard,
                                            shl->select_channel_map,
                                            shl->select_start,
                                            shl->select_end - shl->select_start,
                                            MARKER_SLOPE | MARKER_SLOPE_AUX | MARKER_TEXT));
    
}

void 
shell_delete_markers(shell *shl,
                     enum marker_type type) {
    action_do(ACTION_MARKER_LIST_DELETE_NEW(WITH_UNDO,
                                            shl,
                                            shl->sr->markers, 
                                            shl->select_channel_map,
                                            shl->select_start,
                                            shl->select_end - shl->select_start,
                                            type));
    
}

void
shell_insert_marker(shell *shl,
                    AFframecount start,
                    AFframecount end,
                    enum marker_type type,
                    float multiplier,
                    const char *label) {
    int i;
    struct marker_list **mla;
    mla = marker_list_array_new();
    if(!mla) {
        FAIL("no memory to create new marker list\n");
        return;
    }

    /* This is a little bit outrageous: we generate a marker list for
       every track in case multiple tracks are selected. */

    for(i = 0; i < shl->sr->channels; i++) {
        mla[i] = marker_list_new();
        if(!mla[i]) {
            FAIL("failed to allocate marker list %d\n", i);
            for(i--; i; i--) {
                marker_list_destroy(mla[i]);
                return;
            }
        }
        marker_list_insert(mla[i], 0, type, multiplier, label);
        if(start != end) 
            marker_list_insert(mla[i], end - start, type, multiplier, label);
    }

    action_do(ACTION_MARKER_LIST_INSERT_NEW(WITH_UNDO,
                                            shl,
                                            shl->sr->markers, 
                                            mla,
                                            shl->select_channel_map,
                                            start,
                                            end - start,
                                            MARKER_SLOPE | MARKER_SLOPE_AUX | MARKER_TEXT));
}

void 
shell_insert_envelope_activate(GtkMenuItem *menuitem,
                               shell *shl) {
    shell_insert_marker(shl, 
                        shl->select_start,
                        shl->select_end,
                        MARKER_SLOPE,
                        0,
                        NULL);
}

void
shell_copy_envelopes_activate(GtkMenuItem *menuitem,
                              shell *shl) {
    shell_copy_markers(shl,
                       envelopes_clipboard,
                       MARKER_SLOPE | MARKER_SLOPE_AUX);
}

void
shell_paste_envelopes_activate(GtkMenuItem *menuitem,
                               shell *shl) {
    shell_paste_markers(shl,
                        envelopes_clipboard);

}

void
shell_invert_envelopes_activate(GtkMenuItem *menuitem,
                                shell *shl) {
    int t;
    for(t = 0; t < shl->sr->channels; t++) 
        if((1 << t) & shl->select_channel_map) 
            marker_list_invert(shl->sr->markers[t],
                               shl->select_start,
                               shl->select_end - shl->select_start,
                               MARKER_SLOPE | MARKER_SLOPE_AUX);
    shell_redraw(shl);
}

void
shell_delete_envelopes_activate(GtkMenuItem *menuitem,
                                shell *shl) {
    shell_delete_markers(shl, MARKER_SLOPE | MARKER_SLOPE_AUX);
}

void 
shell_insert_cuepoint_activate(GtkMenuItem *menuitem,
                               shell *shl) {
    char label[256];
    snprintf(label, 256, "label%d", shl->marker_next_label++);
    shell_insert_marker(shl, 
                        shl->player.player_running ? 
                        shl->player.player_pos : shl->select_start,
                        shl->player.player_running ? 
                        shl->player.player_pos : shl->select_end,
                        MARKER_TEXT,
                        0,
                        label);
}

void
shell_copy_cuepoints_activate(GtkMenuItem *menuitem,
                              shell *shl) {
    shell_copy_markers(shl,
                       cuepoints_clipboard,
                       MARKER_TEXT);
}

void
shell_paste_cuepoints_activate(GtkMenuItem *menuitem,
                               shell *shl) {
    shell_paste_markers(shl,
                        cuepoints_clipboard);

}

void
shell_delete_cuepoints_activate(GtkMenuItem *menuitem,
                                shell *shl) {
    shell_delete_markers(shl, MARKER_TEXT);
}

void
shell_delete_time_activate(GtkMenuItem *menuitem,
                           shell *shl) {
    action *a;
    a = ACTION_MARKER_LIST_DELETE_NEW(WITH_UNDO,
                                      shl,
                                      shl->sr->markers, 
                                      shl->select_channel_map,
                                      shl->select_start,
                                      shl->select_end - shl->select_start,
                                      MARKER_SLOPE | MARKER_SLOPE_AUX | MARKER_TEXT);
    if(!a) {
        FAIL("a == NULL\n");
        return;
    }

    a->m_time = 1;
    action_do(a);
}

void
shell_insert_time_activate(GtkMenuItem *menuitem,
                           shell *shl) {
    action *a = ACTION_MARKER_LIST_INSERT_TIME_NEW(WITH_UNDO,
                                                   shl,
                                                   shl->sr->markers, 
                                                   shl->select_channel_map,
                                                   shl->select_start,
                                                   shl->select_end - shl->select_start,
                                                   MARKER_SLOPE | MARKER_SLOPE_AUX | MARKER_TEXT);
    if(!a) { 
        FAIL("a == NULL!\n");
        return;
    }
    a->m_time = 1;
    action_do(a);
}

void 
shell_selection_to_loop_activate(GtkMenuItem *menuitem,
                                 shell *shl) {
    action_do(ACTION_SELECTION_TO_LOOP_NEW(shl));
}

void 
shell_loop_to_selection_activate(GtkMenuItem *menuitem,
                                 shell *shl) {
    action_do(ACTION_LOOP_TO_SELECTION_NEW(shl));
}

void 
shell_select_all_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    action_do(ACTION_SELECT_NEW(WITH_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                0,
                                snd_frame_count(shl->sr)));
}

void
shell_fit_selection_to_window_activate(GtkMenuItem *menuitem,
                                       shell *shl) {
    action_do(ACTION_SELECTION_FIT_NEW(shl));
}

void
shell_center_on_selection_start_activate(GtkMenuItem *menuitem,
                                         shell *shl) {
    shell_viewport_center(shl, 
                          shl->select_start,
                          shl->select_start);
}

void
shell_center_on_selection_end_activate(GtkMenuItem *menuitem,
                                       shell *shl) {
    shell_viewport_center(shl, 
                          shl->select_end,
                          shl->select_end);
}

void
shell_selection_to_4_beats_activate(GtkWidget *w,
                                    shell *shl) {

    if(shl->select_end == shl->select_start)
        return;

    gtk_option_menu_set_history(shl->gridmeasurement, GRID_BEATS);
    shell_grid_measurement_changed(shl->gridmeasurement, shl);
    grid_bpm_from_frames_set(&shl->grid,
                             (shl->select_end - shl->select_start) / 4);
    gtk_spin_button_set_value(shl->gridbpm, shl->grid.bpm);
    shell_grid_configure(shl);
    shell_redraw(shl);
}

void
shell_zoom_activate(GtkWidget *w,
                    shell *shl) {
    const char *s;
    int numer, denom;
    float factor;

#ifdef HAVE_GNOME2
    s = gtk_label_get_text(GTK_LABEL(GTK_BIN(w)->child));
#else
    gtk_label_get(GTK_LABEL(GTK_BIN(w)->child), (char **)&s);
#endif
    while(*s != ':') s++;
    while(*s != ' ') s--;
    denom = atoi(s);
    while(*s != ':') s++;
    numer = atoi(++s);
    factor = (float)numer / denom;
    shell_hres_set(shl, factor);
}

void
shell_zoom_in_activate(GtkWidget *w,
                       shell *shl) {
    float zoom = shl->hres;
    if(zoom > GRAPH_BITS_HRES) {
        if(zoom > HRES_CUTOFF)
            zoom = shl->hres - GRAPH_BITS_HRES * 
                (2 * ((int)shl->hres/HRES_CUTOFF));
        else
           zoom = shl->hres - GRAPH_BITS_HRES;
    } else
        zoom = shl->hres - (shl->hres / 2);

    shell_hres_set(shl, zoom);
}

void
shell_zoom_out_activate(GtkWidget *w,
                        shell *shl) {
    float zoom = shl->hres;
    if(zoom < GRAPH_BITS_HRES)
        zoom = shl->hres + shl->hres;
    else {
        if(zoom > HRES_CUTOFF) 
            zoom = shl->hres + GRAPH_BITS_HRES * 
                (2 * ((int)shl->hres/HRES_CUTOFF));
        else
            zoom = shl->hres + GRAPH_BITS_HRES;
    }
            
    shell_hres_set(shl, zoom);
}

void
shell_tiny_activate(GtkWidget *w,
                     shell *shl) {
    shell_vres_set(shl, 32);
}

void
shell_small_activate(GtkWidget *w,
                     shell *shl) {
    shell_vres_set(shl, 64);
}

void
shell_medium_activate(GtkWidget *w,
                      shell *shl) {
    shell_vres_set(shl, 128);
}

void
shell_large_activate(GtkWidget *w,
                     shell *shl) {
    shell_vres_set(shl, 256);
}

struct marker *
shell_find_cuepoint_previous(shell *shl,
                             int map,
                             AFframecount frame_offset) {
    int i;
    struct marker *m, *previous = NULL;
    for(i = 0; i < shl->sr->channels; i++) {
        if((1 << i) & map) {
            m = marker_list_previous(shl->sr->markers[i],
                                     frame_offset,
                                     MARKER_TEXT);
            if(previous == NULL || 
               (m && m->frame_offset > previous->frame_offset))
                previous = m;
        }
    }
    return previous;
}

struct marker *
shell_find_cuepoint_next(shell *shl,
                         int map,
                         AFframecount frame_offset) {
    int i;
    struct marker *m, *next = NULL;
    for(i = 0; i < shl->sr->channels; i++) {
        if((1 << i) & map) {
            m = marker_list_next(shl->sr->markers[i],
                                 frame_offset,
                                 MARKER_TEXT);
            if(next == NULL || 
               (m && m->frame_offset < next->frame_offset))
                next = m;
        }
    }
    return next;
}

void
shell_left_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    AFframecount 
        ss = (shl->select_start == shl->select_end ? 
              MAX(0, shl->select_start - MAX(1, shl->hres)) :
              shl->select_start);
    struct marker *m;

    if(shl->snap_to_grid) 
        ss -= ss % shl->grid.gap;
    if(shl->snap_to_cuepoints) {
        m = shell_find_cuepoint_previous(shl,
                                         shl->select_channel_map,
                                         ss);
        if(m)
            ss = m->frame_offset;
    }

    action_do(ACTION_SELECT_NEW(WITH_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                ss,
                                0));
}

void
shell_right_activate(GtkMenuItem *menuitem,
                     shell *shl) {
    AFframecount ss;
    struct marker *m;

    ss = (shl->select_start == shl->select_end ?                                
          MIN(snd_frame_count(shl->sr), shl->select_end + 
              MAX(1, shl->hres)) : 
          shl->select_end);

    if(shl->snap_to_grid) {
        ss -= ss % shl->grid.gap;
        ss += shl->grid.gap;
        ss = MIN(snd_frame_count(shl->sr), ss);
    }
    if(shl->snap_to_cuepoints) {
        m = shell_find_cuepoint_next(shl,
                                     shl->select_channel_map,
                                     ss);
        if(m)
            ss = m->frame_offset;
    }

    action_do(ACTION_SELECT_NEW(WITH_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                ss,
                                0));
}

void
shell_track_up_activate(GtkMenuItem *menuitem,
                        shell *shl) {
    int first_sel_track = 1, i;
    for(i = 0; i < shl->sr->channels; i++) {
        if((1 << i) & shl->select_channel_map) {
            first_sel_track = i;
            break;
        }
    }
    
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                1 << MAX(first_sel_track - 1, 0),
                                shl->select_start,
                                0));
}

void
shell_track_down_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    int last_sel_track = 0, i;
    for(i = 0; i < shl->sr->channels; i++)
        if((1 << i) & shl->select_channel_map)
            last_sel_track = i;
    
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                1 << MIN(last_sel_track + 1, 
                                         shl->sr->channels - 1),
                                shl->select_start,
                                0));
}

void
shell_left_select_activate(GtkMenuItem *menuitem,
                           shell *shl) {
    AFframecount ss, se;
    struct marker *m;

    ss = MAX(0, shl->select_start - MAX(1, shl->hres));
    se = shl->select_end;
    
    if(shl->snap_to_grid) 
        ss = MAX(0, ((ss + shl->grid.gap) - 
                     (ss % shl->grid.gap)) - shl->grid.gap);
    if(shl->snap_to_cuepoints) {
        m = shell_find_cuepoint_previous(shl,
                                         shl->select_channel_map,
                                         ss);
        if(m)
            ss = m->frame_offset;
    }

    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                ss,
                                se - ss));
}

void
shell_right_select_activate(GtkMenuItem *menuitem,
                            shell *shl) {
    AFframecount ss, se;
    struct marker *m;

    ss = shl->select_start;
    se = MIN(snd_frame_count(shl->sr), 
             shl->select_end + MAX(1, shl->hres));

    if(shl->snap_to_grid)
        se = MIN(snd_frame_count(shl->sr), 
                 (se - (se % shl->grid.gap)) + shl->grid.gap);
    if(shl->snap_to_cuepoints) {
        m = shell_find_cuepoint_next(shl,
                                     shl->select_channel_map,
                                     se);
        if(m)
            se = m->frame_offset;
    }

    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                ss,
                                se - ss));
}

void
shell_track_up_select_activate(GtkMenuItem *menuitem,
                               shell *shl) {
    int next_sel_track = shl->sr->channels - 1, i;
    for(i = 0; i < shl->sr->channels; i++) {
        if((1 << i) & shl->select_channel_map) {
            next_sel_track = MAX(0, i - 1);
            break;
        }
    }
    
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                (1 << next_sel_track) | shl->select_channel_map,
                                shl->select_start,
                                shl->select_end - shl->select_start));
}

void
shell_track_down_select_activate(GtkMenuItem *menuitem,
                                 shell *shl) {
    int next_sel_track = 0, i;
    for(i = 0; i < shl->sr->channels; i++)
        if((1 << i) & shl->select_channel_map)
            next_sel_track = i + 1;
    
    if(next_sel_track >= shl->sr->channels)
        return;

    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                (1 << next_sel_track) | shl->select_channel_map,
                                shl->select_start,
                                shl->select_end - shl->select_start));
}

void
shell_left_transfer_activate(GtkMenuItem *menuitem,
                             shell *shl) {
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                MAX(0, shl->select_start - MAX(1, shl->hres)),
                                shl->select_end - shl->select_start));
}

void
shell_right_transfer_activate(GtkMenuItem *menuitem,
                              shell *shl) {
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                shl->select_start + MAX(1, shl->hres),
                                shl->select_end - shl->select_start));
}

void
shell_left_jump_activate(GtkMenuItem *menuitem,
                         shell *shl) {
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                MAX(0, shl->select_start - 
                                    (shl->select_end - shl->select_start)),
                                shl->select_end - shl->select_start));
}

void
shell_right_jump_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                shl->select_start +
                                (shl->select_end - shl->select_start),
                                shl->select_end - shl->select_start));
}

void
shell_module_cancel_activate(GtkMenuItem *menuitem,
                             shell *shl) {
    shl->module_cancel_requested = 1;
}

void 
shell_module_activate(GtkMenuItem *menuitem,
                      shell *shl) {
    int module = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(menuitem), 
                                                   "user_data"));
    action_do(ACTION_MODULE_OPEN_NEW(WITH_UNDO,
                                     shl,
                                     module));
    
}

void
shell_mixer_dump_activate(GtkWidget *w,
                          shell *shl) {
    mixer_dump(shl->mixer);
}

void
shell_blocks_dump_activate(GtkWidget *w,
                           shell *shl) {
    int i;
    if(!shl->sr)
        return;

    for(i = 0; i < shl->sr->channels; i++) {
        DEBUG("blocks for track %d\n", i);
        blocklist_dump(shl->sr->tracks[i]->bl);
    }
}

void
shell_sound_info_dump_activate(GtkWidget *w,
                               shell *shl) {
    if(!shl->sr)
        return;

    snd_info_dump(shl->sr);
}

void
shell_join_blocks_activate(GtkWidget *w,
                           shell *shl) {
    //    int i;
    if(!shl->sr)
        return;
    DEBUG("not enabled\n");
    /*
    rwlock_rlock(&shl->sr->rwl);
    for(i = 0; i < shl->sr->channels; i++)
        track_compact(shl->sr->tracks[i]);
    rwlock_runlock(&shl->sr->rwl);
    */
}

void
shell_force_sigsegv_activate(GtkWidget *w,
                             shell *shl) {
    char *p = 0x0;
    *p = '0';
}

void
shell_fail_next_allocation_toggle_activate(GtkWidget *w,
                                           shell *shl) {
    mem_fail_allocation_on_zero = mem_fail_allocation_on_zero > 0 ? 0 : 1;
    DEBUG("mem_fail_allocation_on_zero: %d\n",
          mem_fail_allocation_on_zero);
}

void
shell_step_mode_toggle_activate(GtkWidget *w,
                                  shell *shl) {
    if(shl->debug_flags & DEBUG_FLAG_STEP_MODE)
        shl->debug_flags &= ~DEBUG_FLAG_STEP_MODE;
    else
        shl->debug_flags |= DEBUG_FLAG_STEP_MODE;
    shell_redraw(shl);
}

void
shell_draw_blocks_toggle_activate(GtkWidget *w,
                                  shell *shl) {
    if(shl->debug_flags & DEBUG_FLAG_DRAW_BLOCKS)
        shl->debug_flags &= ~DEBUG_FLAG_DRAW_BLOCKS;
    else
        shl->debug_flags |= DEBUG_FLAG_DRAW_BLOCKS;
    shell_redraw(shl);
}

void
shell_about_activate(GtkWidget *w,
                     shell *shl) {
    GtkWidget *about;
    const gchar *authors[] = {
        "Pascal Haakmat",
        NULL
    };
    const gchar *documenters[] = {
        NULL
    };
    const gchar *translator_credits = NULL;
    const gchar *logo_filename = NULL;
    const gchar *comments = 
        _("GNOME U sound editor.\n" 
          "http://awacs.dhs.org/software/gnusound\n" 
          "Licensed under the GNU General Public License.\n" 
          "Absolutely no warranty.");
    
#ifdef HAVE_GNOME2
    GdkPixbuf *logo_pixbuf = NULL;
    if(!logo_filename) {
        /* FIXME: do something to load logo. */
    }
#endif

    DEBUG("about_activate...\n");
    about = gnome_about_new("GNUsound",
                            VERSION,
                            "Copyright (C) 2002-2004 Pascal Haakmat.",
#ifdef HAVE_GNOME2
                            comments,
                            authors,
                            documenters,
                            translator_credits,
                            logo_pixbuf
#else
                            authors,
                            comments,
                            logo_filename
#endif
                            );
    gtk_widget_show(about);
}

void
shell_destroy_undo_activate(GtkWidget *w,
                            shell *shl) {
    undo_destroy(shl->undo_stack);
    shl->undo_stack = NULL;
}

gboolean
shell_close(shell *shl) {
    action_result *ar = action_do(ACTION_FILE_CLOSE_NEW(shl));
    return action_result_as_bool(ar);
}

gboolean
shell_delete_event(GtkWidget *w,
                   GdkEvent *ev,
                   shell *shl) {
    DEBUG("delete_event\n");
    shell_close(shl);
    return TRUE;
}

gboolean
shell_delete_signal(GtkWidget *w,
                    shell *shl) {
    DEBUG("delete_signal\n");
    shell_close(shl);
    return TRUE;
}

void
shell_bindings_free(shell *shl) {
    if(!shl->bindings)
        return;

    mem_free(shl->bindings);
}

void
shell_modules_close(shell *shl) {
    int i;
    void (*module_close_func)(struct _module_state *module_state);
    char *error;
    for(i = 0; i < module_count; i++) {
        DEBUG("module %d: is_open: %d\n", i, shl->module_state[i].is_open);
        if(shl->module_state[i].is_open) {
            module_close_func = dlsym(modules[i].handle, "module_close");
            if((error = dlerror()) != NULL) {
                FAIL("module %s is open, " \
                     "but an error occurred trying to close it: %s\n",
                     modules[i].name, error);
            } else {
                if(module_close_func) {
                    DEBUG("closing module %s\n", modules[i].name);
                    (*module_close_func)(&shl->module_state[i]);
                }
            }
        }
    }
}

void
shell_destroy(shell *shl) {
    player_stop(shl);
    pthread_mutex_destroy(&shl->player.player_display_lock);
    pthread_cond_destroy(&shl->player.player_display_cond);
    pthread_mutex_destroy(&shl->player.player_running_lock);
    pthread_cond_destroy(&shl->player.player_running_cond);
    shell_modules_close(shl);
    undo_destroy(shl->undo_stack);
    shl->undo_stack = NULL;
    mixer_destroy(shl->mixer);
    DEBUG("destroying sr for shell\n");
    if(shl == clipboard_shell)
        clipboard_shell = NULL;
    else 
        snd_destroy(shl->sr);
    if(shl->pixmap)
        gdk_pixmap_unref(shl->pixmap);
    
    if(shl->graph_bits_buffer_low)
        mem_free(shl->graph_bits_buffer_low);
    /* shl->graph_bits_buffer_high lives in the same allocated memory
       so we don't need to free it separately. */
    shell_bindings_free(shl);
    shells = g_list_remove(shells, shl);
    if(!g_list_length(shells)) {
        quit_requested = 1;
        gtk_main_quit();
    }
}

void
shell_destroy_signal(void *obj,
                     shell *shl) {
    DEBUG("destroy_signal\n");
    DEBUG("shl->use: %d, shl->close_requested: %d\n", shl->use, shl->close_requested);
    if(shl->use)
        FAIL("warning: destroy while still busy?!?\n");
    shell_destroy(shl);
}

void
shell_status_pop(shell *shl) {
    gnome_appbar_pop(shl->appbar);
}

void
shell_status_push(shell *shl, 
                  const char *format, 
                  ...) {
    char status[4096];
    va_list ap;

    va_start(ap, format);
    vsnprintf(status, 4096, format, ap);
    va_end(ap);

    gnome_appbar_push(shl->appbar, status);
}

/*
 * Marker dialog.
 */

void
shell_marker_ok_clicked(GtkWidget *w,
                        shell *shl) {
    char *newlabel = strdup(gtk_entry_get_text(shl->marker_label));
    if(newlabel && strlen(newlabel)) {
        if(shl->marker_being_dragged->label)
            free(shl->marker_being_dragged->label);
        shl->marker_being_dragged->label = newlabel;
        gtk_widget_hide(GTK_WIDGET(shl->marker_dialog));
        shell_redraw(shl);
    }
    if(newlabel == NULL)
        FAIL("could not get memory to copy label string (very low memory)\n");
}


/*
 * Properties dialog.
 */

void
shell_properties_cancel_clicked(GtkWidget *w,
                                shell *shl) {
    DEBUG("cancel\n");
    gtk_widget_hide(shl->props.dialog);
}

void
shell_properties_ok_clicked(GtkWidget *w,
                            shell *shl) {
    struct properties *props = &shl->props;
    int i;
    float old_sample_rate = shl->sr->rate;
    AFframecount new_frame_count, end_offset;

    gtk_widget_hide(shl->props.dialog);
    if(shl->action_state) {
        gui_alert("Please wait for operation in progress to finish.");
        return;
    }

    action_do(ACTION_PLAYER_STOP_NEW(shl));

    shell_cursor_set(shl, GDK_WATCH);
    gui_window_set_cursor(GTK_WIDGET(shl->props.dialog)->window, 
                      GDK_WATCH);
    gui_yield();
    snd_convert(shl->sr,
                props->sample_width,
                props->sample_rate);
    if(RESULT_IS_ERROR(shl->sr)) {
        gui_alert("Could not convert %s: %s", 
                  shl->sr->name, snd_error_get(shl->sr));
        snd_error_free(shl->sr);
        gui_window_set_cursor(GTK_WIDGET(shl->props.dialog)->window, 
                          GDK_LEFT_PTR);
        shell_cursor_set(shl, GDK_LEFT_PTR);
        return;
    }
    undo_destroy(shl->undo_stack);
    shl->undo_stack = NULL;
    grid_rate_set(&shl->grid, shl->sr->rate);

    shell_redraw(shl);
    gui_window_set_cursor(GTK_WIDGET(shl->props.dialog)->window, 
                      GDK_LEFT_PTR);
    shell_cursor_set(shl, GDK_LEFT_PTR);

    if(!gtk_toggle_button_get_active(props->resample))
        return;

    if(props->sample_rate == old_sample_rate)
        return;

    shell_cursor_set(shl, GDK_WATCH);
    gui_window_set_cursor(GTK_WIDGET(shl->props.dialog)->window, 
                      GDK_WATCH);
    shell_status_push(shl, "Please wait, resampling to %.2fhz...",
                      props->sample_rate);
    end_offset = snd_frame_count(shl->sr);
    new_frame_count = snd_frame_count(shl->sr) * 
        ((double)shl->sr->rate / old_sample_rate);
    for(i = 0; i < shl->sr->channels; i++) {
        resample(shl,
                 i,
                 0,
                 end_offset,
                 new_frame_count,
                 RESAMPLE_BEST_ALGORITHM,
                 !HONOR_ENVELOPES,
                 GUARANTEE_NEW_FRAME_COUNT);
    }
    shell_status_pop(shl);
    gtk_toggle_button_set_active(props->resample, FALSE);    
    gui_window_set_cursor(GTK_WIDGET(shl->props.dialog)->window, 
                      GDK_LEFT_PTR);
    shell_cursor_set(shl, GDK_LEFT_PTR);
    shell_redraw(shl);
    return;
}

void
shell_properties_size_labels_set(shell *shl) {
    char s[50];
    size_t current_size, new_size;
    struct properties *props = &shl->props;
    int resample_toggle = gtk_toggle_button_get_active(props->resample);
    current_size = shl->sr->channels * 
        snd_frame_count(shl->sr) *
        shl->sr->frame_width;
    new_size = ceil(shl->sr->channels * 
                    snd_frame_count(shl->sr) *
                    props->sample_width *
                    (resample_toggle ?
                     (props->sample_rate / shl->sr->rate) :
                     1));
    snprintf(s, 50, "%d", current_size);
    gtk_label_set_text(props->cur_label, s);
    snprintf(s, 50, "%d", new_size);
    gtk_label_set_text(props->new_label, s);
}

void
shell_properties_sample_rate_changed(GtkSpinButton *w,
                                     shell *shl) {
    DEBUG("sample rate\n");
    shl->props.sample_rate = 
        gtk_spin_button_get_value(w);
    shell_properties_size_labels_set(shl);
}

void
shell_properties_sample_width_8_clicked(GtkToggleButton *w,
                                        shell *shl) {
    DEBUG("sample width 8\n");
    if(gtk_toggle_button_get_active(w)) 
        shl->props.sample_width = 1;        
    shell_properties_size_labels_set(shl);
}

void
shell_properties_sample_width_16_clicked(GtkToggleButton *w,
                                         shell *shl) {
    DEBUG("sample width 16\n");
    if(gtk_toggle_button_get_active(w)) 
        shl->props.sample_width = 2;
    shell_properties_size_labels_set(shl);
}

void
shell_properties_sample_width_32_clicked(GtkToggleButton *w,
                                         shell *shl) {
    DEBUG("sample width 32\n");
    if(gtk_toggle_button_get_active(w)) 
        shl->props.sample_width = 4;
    shell_properties_size_labels_set(shl);
}

void
shell_properties_resample_toggled(GtkWidget *w,
                                  shell *shl) {
    DEBUG("resample\n");
    shell_properties_size_labels_set(shl);
}

void
shell_properties_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    shl->props.sample_rate = shl->sr->rate;
    shl->props.sample_width = shl->sr->frame_width;
    shell_properties_size_labels_set(shl);
    if(!(GTK_OBJECT_FLAGS(GTK_OBJECT(shl->props.dialog)) & GTK_VISIBLE)) {
        gtk_widget_show(GTK_WIDGET(shl->props.dialog));
        gtk_spin_button_set_value(shl->props.rate, shl->sr->rate);
    } else 
        gdk_window_raise(GTK_WIDGET(shl->props.dialog)->window);
}

/*
 * Core...
 */

void
shell_status_default_set(shell *shl) {
    if(!shl->sr)
        return;

    shell_status_pop(shl);
    shell_status_push(shl,
                      "%s  %.2f sec.  %ld frames", 
                      shell_path_strip(shl->sr->name),
                      snd_frames_to_time(shl->sr, snd_frame_count(shl->sr)), 
                      snd_frame_count(shl->sr));
    shell_properties_size_labels_set(shl);
}

static GnomeUIInfo help_menu[] = {
    GNOMEUIINFO_HELP("gnusound"),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_ABOUT_ITEM(shell_about_activate, "About GNUsound"),
    GNOMEUIINFO_END
};

signal_bindings bindings[] = {

    /* Basic window and canvas events. */

    { "canvas", "expose_event", (GtkSignalFunc)shell_canvas_expose_event },
    { "canvas", "configure_event", (GtkSignalFunc)shell_canvas_configure_event },
    { "canvas", "button-press-event", (GtkSignalFunc)shell_canvas_button_press_event },
    { "canvas", "button_release_event", (GtkSignalFunc)shell_canvas_button_release_event },
    { "canvas", "motion-notify-event", (GtkSignalFunc)shell_canvas_motion_notify_event },
    { "infocanvas", "expose_event", (GtkSignalFunc)shell_infocanvas_expose_event },
    { "mixercanvas", "expose_event", (GtkSignalFunc)shell_mixercanvas_expose_event },
    { "mixercanvas", "configure_event", (GtkSignalFunc)shell_mixercanvas_configure_event },
    { "mixercanvas", "button_press_event", (GtkSignalFunc)shell_mixercanvas_button_press_event },
    { "mixercanvas", "button_release_event", (GtkSignalFunc)shell_mixercanvas_button_release_event },
    { "mixercanvas", "motion_notify_event", (GtkSignalFunc)shell_mixercanvas_motion_notify_event },
    { "appwindow", "size_allocate", (GtkSignalFunc)shell_size_allocate_signal },
    { "appwindow", "delete_event", (GtkSignalFunc)shell_delete_event },
    { "appwindow", "focus_in_event", (GtkSignalFunc)shell_focus_in_event },
    { "appwindow", "focus_out_event", (GtkSignalFunc)shell_focus_out_event },
    { "appwindow", "destroy", (GtkSignalFunc)shell_destroy_signal },

    /* Toolbar buttons. */
    
    { "b_newfile", "clicked", (GtkSignalFunc)gui_new_activate },
    { "b_open", "clicked", (GtkSignalFunc)gui_open_activate },
    { "b_save", "clicked", (GtkSignalFunc)shell_save_activate },
    { "b_cut", "clicked", (GtkSignalFunc)shell_cut_activate },
    { "b_copy", "clicked", (GtkSignalFunc)shell_copy_activate },
    { "b_paste", "clicked", (GtkSignalFunc)shell_paste_activate },
    { "b_clear", "clicked", (GtkSignalFunc)shell_clear_activate },
    { "b_play", "clicked", (GtkSignalFunc)shell_play_activate },
    { "b_cue_play", "clicked", (GtkSignalFunc)shell_cue_play_activate },
    { "b_stop", "clicked", (GtkSignalFunc)shell_stop_activate },
    { "b_scrub_left", "pressed", (GtkSignalFunc)shell_scrub_left_pressed },
    { "b_scrub_right", "pressed", (GtkSignalFunc)shell_scrub_right_pressed },
    { "b_scrub_left", "released", (GtkSignalFunc)shell_scrub_released },
    { "b_scrub_right", "released", (GtkSignalFunc)shell_scrub_released },
    { "b_record", "clicked", (GtkSignalFunc)shell_record_activate },
    
    /* Grid toolbar. */

    { "grid_units", "changed", (GtkSignalFunc)shell_grid_units_changed },
    { "grid_bpm", "changed", (GtkSignalFunc)shell_grid_bpm_changed },

    /* File menu. */

    { "newfile", "activate", (GtkSignalFunc)gui_new_activate },
    { "open", "activate", (GtkSignalFunc)gui_open_activate },
    { "save", "activate", (GtkSignalFunc)shell_save_activate },
    { "saveas", "activate", (GtkSignalFunc)shell_saveas_activate },
    { "mixdown", "activate", (GtkSignalFunc)shell_mixdown_activate },
    { "close", "activate", (GtkSignalFunc)shell_delete_signal },
    { "exit", "activate", (GtkSignalFunc)gui_exit_activate },

    /* Edit menu. */

    { "undo", "activate", (GtkSignalFunc)shell_undo_activate },
    { "cut", "activate", (GtkSignalFunc)shell_cut_activate },
    { "paste", "activate", (GtkSignalFunc)shell_paste_activate },
    { "paste_fit", "activate", (GtkSignalFunc)shell_paste_fit_activate },
    { "paste_over", "activate", (GtkSignalFunc)shell_paste_over_activate },
    { "paste_mix", "activate", (GtkSignalFunc)shell_paste_mix_activate },
    { "copy", "activate", (GtkSignalFunc)shell_copy_activate },
    { "clear", "activate", (GtkSignalFunc)shell_clear_activate },
    { "erase", "activate", (GtkSignalFunc)shell_erase_activate },
    { "crop", "activate", (GtkSignalFunc)shell_crop_activate },
    { "insert_tracks", "activate", (GtkSignalFunc)shell_insert_tracks_activate },
    { "delete_tracks", "activate", (GtkSignalFunc)shell_delete_tracks_activate },
    { "show_clipboard", "activate", (GtkSignalFunc)gui_show_clipboard_activate },
    { "properties", "activate", (GtkSignalFunc)shell_properties_activate },

    /* Markers menu. */
    
    { "insert_cuepoint", "activate", (GtkSignalFunc)shell_insert_cuepoint_activate },
    { "delete_cuepoints", "activate", (GtkSignalFunc)shell_delete_cuepoints_activate },
    { "copy_cuepoints", "activate", (GtkSignalFunc)shell_copy_cuepoints_activate },
    { "paste_cuepoints", "activate", (GtkSignalFunc)shell_paste_cuepoints_activate },
    { "insert_envelope", "activate", (GtkSignalFunc)shell_insert_envelope_activate },
    { "delete_envelopes", "activate", (GtkSignalFunc)shell_delete_envelopes_activate },
    { "copy_envelopes", "activate", (GtkSignalFunc)shell_copy_envelopes_activate },
    { "paste_envelopes", "activate", (GtkSignalFunc)shell_paste_envelopes_activate },
    { "invert_envelopes", "activate", (GtkSignalFunc)shell_invert_envelopes_activate },
    { "delete_time", "activate", (GtkSignalFunc)shell_delete_time_activate },
    { "insert_time", "activate", (GtkSignalFunc)shell_insert_time_activate },
    { "envelope_enable_toggle", "activate", (GtkSignalFunc)shell_envelope_enable_toggle_activate },

    /* Marker edit dialog. */
    
    { "marker_ok", "clicked", (GtkSignalFunc)shell_marker_ok_clicked },
    { "marker_label", "activate", (GtkSignalFunc)shell_marker_ok_clicked },

    /* Properties dialog. */

    { "properties_resample", "toggled", (GtkSignalFunc)shell_properties_resample_toggled },
    { "properties_sample_rate", "changed", (GtkSignalFunc)shell_properties_sample_rate_changed },
    { "properties_sample_width_8", "clicked", (GtkSignalFunc)shell_properties_sample_width_8_clicked },
    { "properties_sample_width_16", "clicked", (GtkSignalFunc)shell_properties_sample_width_16_clicked },
    { "properties_sample_width_32", "clicked", (GtkSignalFunc)shell_properties_sample_width_32_clicked },
    { "properties_ok", "clicked", (GtkSignalFunc)shell_properties_ok_clicked },
    { "properties_cancel", "clicked", (GtkSignalFunc)shell_properties_cancel_clicked },

    /* View menu. */

    { "16_1", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "8_1", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "4_1", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "2_1", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "1_1", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "1_2", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "1_3", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "1_4", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "1_5", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "1_6", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "1_7", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "1_8", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "1_9", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "1_10", "activate", (GtkSignalFunc)shell_zoom_activate },
    { "zoom_in", "activate", (GtkSignalFunc)shell_zoom_in_activate },
    { "zoom_out", "activate", (GtkSignalFunc)shell_zoom_out_activate },
    { "tiny", "activate", (GtkSignalFunc)shell_tiny_activate },
    { "small", "activate", (GtkSignalFunc)shell_small_activate },
    { "medium", "activate", (GtkSignalFunc)shell_medium_activate },
    { "large", "activate", (GtkSignalFunc)shell_large_activate },
    { "show_zero_line_toggle", "activate", (GtkSignalFunc)shell_show_zero_line_toggle_activate },
    { "show_envelope_toggle", "activate", (GtkSignalFunc)shell_show_envelope_toggle_activate },
    { "show_grid_toggle", "activate", (GtkSignalFunc)shell_show_grid_toggle_activate },

    /* Select menu. */
    
    { "left", "activate", (GtkSignalFunc)shell_left_activate },
    { "right", "activate", (GtkSignalFunc)shell_right_activate },
    { "track_up", "activate", (GtkSignalFunc)shell_track_up_activate },
    { "track_down", "activate", (GtkSignalFunc)shell_track_down_activate },
    { "left_select", "activate", (GtkSignalFunc)shell_left_select_activate },
    { "right_select", "activate", (GtkSignalFunc)shell_right_select_activate },
    { "track_up_select", "activate", (GtkSignalFunc)shell_track_up_select_activate },
    { "track_down_select", "activate", (GtkSignalFunc)shell_track_down_select_activate },
    { "left_transfer", "activate", (GtkSignalFunc)shell_left_transfer_activate },
    { "right_transfer", "activate", (GtkSignalFunc)shell_right_transfer_activate },
    { "left_jump", "activate", (GtkSignalFunc)shell_left_jump_activate },
    { "right_jump", "activate", (GtkSignalFunc)shell_right_jump_activate },
    { "select_all", "activate", (GtkSignalFunc)shell_select_all_activate },
    { "selection_to_loop", "activate", (GtkSignalFunc)shell_selection_to_loop_activate },
    { "loop_to_selection", "activate", (GtkSignalFunc)shell_loop_to_selection_activate },
    { "fit_selection_to_window", "activate", (GtkSignalFunc)shell_fit_selection_to_window_activate },
    { "center_on_selection_start", "activate", (GtkSignalFunc)shell_center_on_selection_start_activate },
    { "center_on_selection_end", "activate", (GtkSignalFunc)shell_center_on_selection_end_activate },
    { "selection_to_4_beats", "activate", (GtkSignalFunc)shell_selection_to_4_beats_activate },
    { "snap_to_cuepoints_toggle", "activate", (GtkSignalFunc)shell_snap_to_cuepoints_toggle_activate },
    { "snap_to_grid_toggle", "activate", (GtkSignalFunc)shell_snap_to_grid_toggle_activate },

    /* Playback menu. */

    { "play", "activate", (GtkSignalFunc)shell_play_activate },
    { "cue_play", "activate", (GtkSignalFunc)shell_cue_play_activate },
    { "record", "activate", (GtkSignalFunc)shell_record_activate },
    { "scrub_left", "activate", (GtkSignalFunc)shell_scrub_left_activate },
    { "scrub_right", "activate", (GtkSignalFunc)shell_scrub_right_activate },
    { "loop_toggle", "activate", (GtkSignalFunc)shell_loop_toggle_activate },
    { "follow_playback_toggle", "activate", (GtkSignalFunc)shell_follow_playback_toggle_activate },
    { "record_replace_toggle", "activate", (GtkSignalFunc)shell_record_replace_toggle_activate },

    /* Settings menu. */
    
    { "preferences", "activate", (GtkSignalFunc)gui_preferences_activate },

    /* Debug menu. */

    { "dump_blocks", "activate", (GtkSignalFunc)shell_blocks_dump_activate },
    { "dump_sound_info", "activate", (GtkSignalFunc)shell_sound_info_dump_activate },
    { "dump_mixer", "activate", (GtkSignalFunc)shell_mixer_dump_activate },
    { "destroy_undo", "activate", (GtkSignalFunc)shell_destroy_undo_activate },
    { "draw_blocks_toggle", "activate", (GtkSignalFunc)shell_draw_blocks_toggle_activate },
    { "step_mode_toggle", "activate", (GtkSignalFunc)shell_step_mode_toggle_activate },
    { "join_blocks", "activate", (GtkSignalFunc)shell_join_blocks_activate },
    { "fail_next_allocation_toggle", "activate", (GtkSignalFunc)shell_fail_next_allocation_toggle_activate },
    { "force_sigsegv", "activate", (GtkSignalFunc)shell_force_sigsegv_activate },

    { NULL, NULL, NULL }
};

int
shell_bindings_alloc(shell *shl) {
    int i;
    for(i = 0; bindings[i].widget_name; i++);

    shl->bindings = mem_alloc(i * sizeof(action_bindings));

    if(!shl->bindings) {
        FAIL("could not allocate %d bindings\n", i);
        return -1;
    }

    for(i = 0; bindings[i].widget_name; i++) {
        shl->bindings[i].object = NULL;
        //        shl->bindings[i].ref = 0;
    }

    return 0;
}

GtkObject *
shell_bindings_find(shell *shl,
                    const char *name) {
    int i;
    for(i = 0; bindings[i].widget_name; i++) 
        if(!strcmp(name, bindings[i].widget_name))
            return shl->bindings[i].object;
    FAIL("cannot find binding for %s\n", name);
    abort();
}

void
shell_bindings_disable(shell *shl,
                       char *names[]) {
    int i;
    GtkWidget *w;
    
    for(i = 0; names[i]; i++) {
        w = GTK_WIDGET(shell_bindings_find(shl, names[i]));
        //        gtk_widget_hide(w);
        gtk_widget_set_sensitive(w, FALSE);
    }
}

void
shell_bindings_connect(shell *shl,
                       GladeXML *xml) {
    int i;
    GtkObject *o;

    for(i = 0; bindings[i].widget_name; i++) {
        o = GTK_OBJECT(glade_xml_get_widget(xml, bindings[i].widget_name));
        if(!o) {
            FAIL("cannot bind %s\n", bindings[i].widget_name);
            abort();
        }
        shl->bindings[i].object = o;
        g_signal_connect(o, 
                         bindings[i].signal_name,
                         bindings[i].signal_func, 
                         shl);
    }
}

shell *
shell_new() {
    GladeXML *xml;
    GtkMenuItem *tools_trigger, *debug_trigger, *help_trigger, *separator;
    GtkMenu *tools, *help;
    GtkMenuItem *item;
    GtkMenuBar *menubar;
    GtkToolbar *toolbar;
    GList *iterator;
    shell *shl;
    struct stat statbuf;
    int i, err;
#ifndef HAVE_GNOME2
    GdkEventConfigure cfg; 
#endif

    err = stat(PRIMARY_GLADE_FILE, &statbuf);
    if(!err) 
        xml = glade_xml_new(PRIMARY_GLADE_FILE, NULL, NULL);
    else
        xml = glade_xml_new(SECONDARY_GLADE_FILE, NULL, NULL);

    if(!xml) {
        FAIL("could not find interface definition, " "looked at %s:%s.", 
             PRIMARY_GLADE_FILE, SECONDARY_GLADE_FILE);
        return NULL;
    }

    shl = mem_calloc(sizeof(shell), 1);
    if(!shl) {
        FAIL("could not allocate memory for new shell object\n");
        return NULL;
    }
    if(shell_bindings_alloc(shl)) {
        FAIL("could not alloc shell bindings\n");
        shell_bindings_free(shl);
        mem_free(shl);
        return NULL;
    }

    shl->magic[0] = 'O';
    shl->magic[1] = 'k';

    shl->undo_stack = NULL;
    shl->mixer = mixer_new(0, 0);
    shl->hres = GRAPH_BITS_HRES;
    shl->vres = 128;
    shl->sr = NULL;
    shl->select_start = shl->select_end = shl->select_pivot = shl->select_flex = 0;
    shl->select_channel_map = 1;
    shl->player.player_pos = 0;
    shl->record_mode = 0;
    pthread_mutex_init(&shl->player.player_running_lock, NULL);
    pthread_cond_init(&shl->player.player_running_cond, NULL);
    pthread_mutex_init(&shl->player.player_display_lock, NULL);
    pthread_cond_init(&shl->player.player_display_cond, NULL);
    shl->scroll_drag = 0;
    shl->loop = 1;
    shl->envelope_enabled = 0;
    shl->show_envelope = pref_get_as_int("show_envelope");
    shl->show_zero = pref_get_as_int("show_zero");
    shl->record_replace = pref_get_as_int("record_replace");
    shl->follow_playback = pref_get_as_int("follow_playback");
    shl->show_grid = pref_get_as_int("show_grid");
    shl->snap_to_grid = pref_get_as_int("snap_to_grid");
    shl->snap_to_cuepoints = pref_get_as_int("snap_to_cuepoints");
    shl->pixmap = NULL;
    shl->graph_bits_buffer_low = NULL;
    shl->graph_bits_buffer_high = NULL;
    shl->active_module_id = -1;
    shl->module_cancel_requested = 0;
    shl->file_load_cancel_requested = 0;
    shl->close_requested = 0;
    shl->use = 0;
    shl->scrubbing = 0;
    shl->being_drawn = 0;
    grid_bpm_set(&shl->grid, pref_get_as_float("default_bpm"));
    grid_units_set(&shl->grid, 1);
    grid_rate_set(&shl->grid, pref_get_as_float("default_sample_rate"));
    grid_measurement_set(&shl->grid, GRID_SECONDS);

    shl->marker_being_dragged = NULL;
    shl->marker_dragged_on_track = -1;
    shl->marker_next_label = 1;
    shl->source_channel_being_dragged = -1;
    shl->target_channel_being_dragged = -1;
    shl->last_mouse_y = 0;

    gettimeofday(&shl->last_wheel_movement, NULL);
    shl->last_wheel_pos = 0;
    shl->last_wheel_x = -1;

    shl->appwindow = GNOME_APP(glade_xml_get_widget(xml, "appwindow"));
    shl->marker_dialog = glade_xml_get_widget(xml, "markerdialog");
    shl->marker_label = GTK_ENTRY(glade_xml_get_widget(xml, "marker_label"));
    shl->props.dialog = glade_xml_get_widget(xml, "propertiesdialog");
    shl->props.cur_label = GTK_LABEL(glade_xml_get_widget(xml, "properties_current_size"));
    shl->props.new_label = GTK_LABEL(glade_xml_get_widget(xml, "properties_new_size"));
    shl->props.resample = GTK_TOGGLE_BUTTON(glade_xml_get_widget(xml, "properties_resample"));
    shl->props.rate = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "properties_sample_rate"));
    shl->appbar = GNOME_APPBAR(glade_xml_get_widget(xml, "appbar"));
    shl->canvas = GTK_DRAWING_AREA(glade_xml_get_widget(xml, "canvas")); 
    shl->infocanvas = GTK_DRAWING_AREA(glade_xml_get_widget(xml, "infocanvas")); 
    shl->mixercanvas = GTK_DRAWING_AREA(glade_xml_get_widget(xml, "mixercanvas")); 
    shl->hruler = GTK_HRULER(glade_xml_get_widget(xml, "hruler"));
    shl->hscrollbar = GTK_RANGE(glade_xml_get_widget(xml, "hscrollbar"));
    shl->vscrollbar = GTK_RANGE(glade_xml_get_widget(xml, "vscrollbar"));
    shl->hadjust = gtk_range_get_adjustment(shl->hscrollbar);
    shl->hadjust->upper = 1;
    shl->hadjust->page_size = 1;
    shl->hadjust->page_increment = 1;
    gtk_adjustment_changed(shl->hadjust);
    shl->vadjust = gtk_range_get_adjustment(shl->vscrollbar);
    shl->progress = gnome_appbar_get_progress(shl->appbar);
    shl->gridunits = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "grid_units"));
    shl->gridbpm = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "grid_bpm"));
    shl->gridmeasurement = GTK_OPTION_MENU(glade_xml_get_widget(xml, "grid_measurement"));
    shl->tail_action = NULL;
    debug_trigger = GTK_MENU_ITEM(glade_xml_get_widget(xml, "debug"));
    tools_trigger = GTK_MENU_ITEM(glade_xml_get_widget(xml, "tools"));
    help_trigger = GTK_MENU_ITEM(glade_xml_get_widget(xml, "help"));
    toolbar = GTK_TOOLBAR(glade_xml_get_widget(xml, "toolbar_file"));
    gtk_toolbar_set_style(toolbar, GTK_TOOLBAR_ICONS);
    toolbar = GTK_TOOLBAR(glade_xml_get_widget(xml, "toolbar_player"));
    gtk_toolbar_set_style(toolbar, GTK_TOOLBAR_ICONS);
    g_signal_connect(GTK_OBJECT(shl->hadjust), "value_changed",
                     G_CALLBACK(shell_hscrollbar_value_changed), shl);
    g_signal_connect(GTK_OBJECT(shl->vadjust), "value_changed",
                     G_CALLBACK(shell_vscrollbar_value_changed), shl);

    for(iterator = GTK_MENU_SHELL(GTK_MENU(gtk_option_menu_get_menu(shl->gridmeasurement)))->children; 
        iterator; 
        iterator = iterator->next)         
        g_signal_connect(GTK_OBJECT(iterator->data), "activate",
                         G_CALLBACK(shell_grid_measurement_changed),
                         shl);
    
    /* Setup interface to conform to preference/recalled values. */

    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(glade_xml_get_widget(xml, "show_zero_line_toggle")), 
                                   pref_get_as_int("show_zero"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(glade_xml_get_widget(xml, "record_replace_toggle")), 
                                   pref_get_as_int("record_replace"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(glade_xml_get_widget(xml, "follow_playback_toggle")), 
                                   pref_get_as_int("follow_playback"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(glade_xml_get_widget(xml, "show_grid_toggle")), 
                                   pref_get_as_int("show_grid"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(glade_xml_get_widget(xml, "snap_to_grid_toggle")), 
                                   pref_get_as_int("snap_to_grid"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(glade_xml_get_widget(xml, "snap_to_cuepoints_toggle")),
                                   pref_get_as_int("snap_to_cuepoints"));
    gtk_spin_button_set_value(shl->gridbpm, shl->grid.bpm);

    /* Setup Help menu. */
    
    menubar = GTK_MENU_BAR(glade_xml_get_widget(xml, "menubar"));
    help = GTK_MENU(gtk_menu_new());
    gnome_app_fill_menu(GTK_MENU_SHELL(help), help_menu, 
                        GNOME_APP(shl->appwindow)->accel_group, 1, 0);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(help_trigger),
                              GTK_WIDGET(help));

    /* Connect all signal handlers 
       (glade's autoconnect is not handy for large number of signals). */

    shell_bindings_connect(shl, xml);

    /* Unref the XML object. */

    g_object_unref(G_OBJECT(xml));

    /* Setup the tools menu. The modules structure is initialized in
       module.c. */

    tools = GTK_MENU(gtk_menu_new());
    gtk_menu_set_title(tools, "Tools");

    item = GTK_MENU_ITEM(gtk_tearoff_menu_item_new());
    gtk_widget_show(GTK_WIDGET(item));
    gtk_menu_shell_append(GTK_MENU_SHELL(tools), GTK_WIDGET(item));
    shl->cancelmodule = GTK_MENU_ITEM(gtk_menu_item_new_with_label("Cancel"));
    gtk_menu_shell_append(GTK_MENU_SHELL(tools), GTK_WIDGET(shl->cancelmodule));
    gtk_widget_show(GTK_WIDGET(shl->cancelmodule));
    gtk_widget_set_sensitive(GTK_WIDGET(shl->cancelmodule), FALSE);
#ifndef HAVE_GNOME2
    gtk_widget_add_accelerator(GTK_WIDGET(shl->cancelmodule),
                               "activate",
                               gtk_accel_group_get_default(),
                               GDK_period,
                               GDK_CONTROL_MASK,
                               GTK_ACCEL_VISIBLE);
#endif

    g_signal_connect(GTK_OBJECT(shl->cancelmodule),
                     "activate",
                     G_CALLBACK(shell_module_cancel_activate),
                     shl);
    separator = GTK_MENU_ITEM(gtk_menu_item_new());
    gtk_menu_shell_append(GTK_MENU_SHELL(tools), GTK_WIDGET(separator));
    gtk_widget_show(GTK_WIDGET(separator));
    for(i = 0; i < module_count; i++) {
        item = GTK_MENU_ITEM(gtk_menu_item_new_with_label(modules[i].name));
        g_object_set_data(G_OBJECT(item), "user_data", GINT_TO_POINTER(i));
        gtk_menu_shell_append(GTK_MENU_SHELL(tools), GTK_WIDGET(item));
        gtk_widget_show(GTK_WIDGET(item));
        g_signal_connect(GTK_OBJECT(item), 
                         "activate",
                         G_CALLBACK(shell_module_activate), 
                         shl);
    }
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(tools_trigger),
                              GTK_WIDGET(tools));

#ifndef DEBUG_FLAG
    gtk_widget_hide(GTK_WIDGET(debug_trigger));
#endif
    gtk_widget_realize(GTK_WIDGET(shl->appwindow));

#ifndef HAVE_GNOME2
    /* Explicitly send another configure event because we might miss
       the first, I think. */

    cfg.type = GDK_CONFIGURE;
    cfg.window = GTK_WIDGET(shl->canvas)->window;
    cfg.send_event = 0;
    cfg.x = 0;
    cfg.y = 0;
    cfg.width = 500;
    cfg.height = 500;
    gtk_main_do_event((GdkEvent *)&cfg);
#endif

#ifdef HAVE_GNOME2
    gtk_widget_set_redraw_on_allocate(GTK_WIDGET(shl->canvas), FALSE);

    /* We perform our own buffering for the sample canvas. */
    gtk_widget_set_double_buffered(GTK_WIDGET(shl->canvas), FALSE);
    /* We don't desire any buffering on the mixer canvas and the info
       canvas. */
    gtk_widget_set_double_buffered(GTK_WIDGET(shl->infocanvas), FALSE);
    gtk_widget_set_double_buffered(GTK_WIDGET(shl->mixercanvas), FALSE);
    /* Annoyingly libglade-2 doesn't seem to pick these up from the
       config file, or maybe it's a bug elsewhere. */
    gtk_widget_add_events(GTK_WIDGET(shl->canvas), 
                          GDK_BUTTON_PRESS_MASK | 
                          GDK_BUTTON_RELEASE_MASK | 
                          GDK_POINTER_MOTION_MASK);
    gtk_widget_add_events(GTK_WIDGET(shl->mixercanvas), 
                          GDK_BUTTON_PRESS_MASK | 
                          GDK_BUTTON_RELEASE_MASK | 
                          GDK_POINTER_MOTION_MASK);
                          
#endif

    shells = g_list_append(shells, shl);

    print_adjust("init done", shl->hadjust);
    return shl;
}

