/*
**  MessageViewWindowController.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ujwal S. Sathyam <ujwal@setlurgroup.com>
**          Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "MessageViewWindowController.h"

#include "GNUMail.h"
#include "GNUMail/GNUMailBundle.h"
#include "Constants.h"
#include "ExtendedCell.h"
#include "MailHeaderCell.h"
#include "MailWindowController.h"
#include "MimeType.h"
#include "MimeTypeManager.h"
#include "Utilities.h"

#ifndef MACOSX
#include "MessageViewWindow.h"
#endif

#include <Pantomime/Container.h>
#include <Pantomime/Constants.h>
#include <Pantomime/Flags.h>
#include <Pantomime/Folder.h>
#include <Pantomime/IMAPFolder.h>
#include <Pantomime/IMAPStore.h>
#include <Pantomime/InternetAddress.h>
#include <Pantomime/LocalFolder.h>
#include <Pantomime/LocalStore.h>
#include <Pantomime/Message.h>
#include <Pantomime/MimeMultipart.h>


@implementation MessageViewWindowController

//
//
//
- (id) initWithWindowNibName: (NSString *) theWindowNibName
{
#ifdef MACOSX
  self = [super initWithWindowNibName: theWindowNibName];
#else
  MessageViewWindow *aMessageViewWindow;
  
  aMessageViewWindow = [[MessageViewWindow alloc] initWithContentRect: NSMakeRect(150,100,720,600)
						  styleMask: NSClosableWindowMask|NSTitledWindowMask|
						  NSMiniaturizableWindowMask|NSResizableWindowMask
						  backing: NSBackingStoreRetained
						  defer: NO];

  self = [super initWithWindow: aMessageViewWindow];
  
  [aMessageViewWindow layoutWindow];
  [aMessageViewWindow setDelegate: self];

  textView = aMessageViewWindow->textView;
  previousMessage = aMessageViewWindow->previousMessage;
  nextMessage = aMessageViewWindow->previousMessage;
  showOrHideAllHeaders = aMessageViewWindow->showOrHideAllHeaders;
  raw = aMessageViewWindow->raw;

  RELEASE(aMessageViewWindow);
#endif

  [[self window] setFrameAutosaveName: @"MessageViewWindow"];
  [[self window] setFrameUsingName: @"MessageViewWindow"];
  
  // We create our mail header cell
  mailHeaderCell = [[MailHeaderCell alloc] init];

  [[NSNotificationCenter defaultCenter] addObserver: mailHeaderCell
					selector: @selector (resize:) 
					name: @"NSViewFrameDidChangeNotification" 
					object: textView];

  // We load our accessory views
  [self _loadAccessoryViews];
  
  // Set out textview to non-editable
  [textView setEditable: NO];
  
  // Set ourselves up as the delegate
  [textView setDelegate: self];
  
  return self;
}


//
//
//
#ifdef MACOSX
- (void) awakeFromNib
{
  NSToolbar *toolbar;
  
  toolbar = [[NSToolbar alloc] initWithIdentifier: @"MessageViewWindowToolbar"];
  [toolbar setDelegate: self];
  [toolbar setAllowsUserCustomization: YES];
  [toolbar setAutosavesConfiguration: YES];
  [[self window] setToolbar: toolbar];
  RELEASE(toolbar);
}
#endif


//
//
//
- (void) dealloc
{
  NSDebugLog(@"MessageViewWindowController: dealloc called for message window: %@", [message subject]);
  
  [[NSNotificationCenter defaultCenter] removeObserver: mailHeaderCell
					name: @"NSViewFrameDidChangeNotification" 
					object: textView];

  RELEASE(mailHeaderCell);
  RELEASE(message);

  [super dealloc];
}



//
// action methods
//
- (IBAction) deleteMessage: (id) sender
{
  Flags *theFlags;
  int aRow;

  theFlags = [[[self message] flags] copy];
  [theFlags add: DELETED];
  [[self message] setFlags: theFlags];
  RELEASE(theFlags);

  // We refresh our row in our dataView
  if ( [mailWindowController dataViewType] == TYPE_OUTLINEVIEW )
    {
      // FIXME
      aRow = -1;
    }
  else
    {
      aRow = [[mailWindowController allMessages] indexOfObject: [self message]];
      [[mailWindowController dataView] setNeedsDisplayInRect: [[mailWindowController dataView] rectOfRow: aRow]];
    }

  // FIXME: Review all the code when "hiding" deleted messages.
  // If we are hiding deleted message, we must reload our table data
  //if ( ![[self folder] showDeleted] )
  //  {
  //    [[self folder] updateCache];
  //    [mailWindowController dataViewShouldReloadData];
  //  }

  [self nextMessage: self];
}


//
// Invoke the message Utility class method to reply to this message.
//
- (IBAction) replyMessage: (id) sender
{
  [Utilities replyToSender: [self message]
	     folder: [self folder]
	     replyToAll: (sender == [NSApp delegate] ? YES : NO)];
}


//
// Invoke the message Utility class method to forward this message.
//
- (IBAction) forwardMessage: (id) sender
{ 
  [Utilities forwardMessage: [self message]];
}


//
//
//
- (IBAction) previousMessage: (id) sender
{
  Message *aMessage;
  int row;

  indexOffset--;
  row = [[mailWindowController dataView] selectedRow] + indexOffset;
  
  NSDebugLog(@"row = %d, offset = %d", [[mailWindowController dataView] selectedRow], indexOffset);

  if ( row < 0 ) 
    {
      NSBeep();
      indexOffset++;
      return;
    }
  

  aMessage = [mailWindowController messageAtRow: row];
  
  if ( aMessage )
    {
      [self setMessage: aMessage];
      [Utilities showMessage: [self message]
		 target: [self textView]
		 showAllHeaders: [self showAllHeaders]];
      [self windowDidBecomeKey: nil];
    }
}


//
//
//
- (IBAction) nextMessage: (id) sender
{
  Message *aMessage;
  int row;

  indexOffset++;
  row = [[mailWindowController dataView] selectedRow] + indexOffset;
  
  NSDebugLog(@"row = %d, offset = %d", [[mailWindowController dataView] selectedRow], indexOffset);

  if ( row == -1 ||
       row > ([[mailWindowController dataView] numberOfRows] - 1) ) 
    {
      // We do NOT beep if sender == self
      // This happens if we clicked on the Delete button and our current
      // index in the dataView is the last one.
      if ( sender != self )
	{
	  NSBeep();
	}

      indexOffset--;
      return;
    }


  aMessage = [mailWindowController messageAtRow: row];
  
  if ( aMessage )
    {
      [self setMessage: aMessage];
      [Utilities showMessage: [self message]
		 target: [self textView]
		 showAllHeaders: [self showAllHeaders]];
      [self windowDidBecomeKey: nil];
    }
  
}


//
// 
//
- (IBAction) firstMessage: (id) sender
{  
  if ( [[mailWindowController dataView] numberOfRows] > 0 )
    {
      Message *aMessage;
      
      aMessage = [mailWindowController messageAtRow: 0];
      
      if ( aMessage )
	{
	  [self setMessage: aMessage];
	  [Utilities showMessage: [self message]
		     target: [self textView]
		     showAllHeaders: [self showAllHeaders]];
	  [self windowDidBecomeKey: nil];
	}
    }
  else
    {
      NSBeep();
    }
}


//
// 
//
- (IBAction) lastMessage: (id) sender
{
  int row;

  row = [[mailWindowController dataView] numberOfRows] - 1;
  
  if ( row >= 0 )
    {
      Message *aMessage;
      
      aMessage = [mailWindowController messageAtRow: row];
      
      if ( aMessage )
	{
	  [self setMessage: aMessage];
	  [Utilities showMessage: [self message]
		     target: [self textView]
		     showAllHeaders: [self showAllHeaders]];
	  [self windowDidBecomeKey: nil];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) showOrHideAllHeaders: (id) sender
{
  [(GNUMail *)[NSApp delegate] showAllHeaders: sender];
}


//
//
//
- (IBAction) pageDownMessage: (id) sender
{
  NSScrollView *textScrollView;
  NSRect aRect;
  double origin;

  textScrollView = [textView enclosingScrollView];
  aRect = [textScrollView documentVisibleRect];
  origin = aRect.origin.y;
  
  aRect.origin.y += aRect.size.height - [textScrollView verticalPageScroll];
  [textView scrollRectToVisible: aRect];
  
  aRect = [textScrollView documentVisibleRect];
  
  if (aRect.origin.y == origin)
    {
      [self nextMessage: nil];
    } 
}


//
//
//
- (IBAction) pageUpMessage: (id) sender
{
  NSScrollView *textScrollView;
  NSRect aRect;
  double origin;

  textScrollView = [textView enclosingScrollView];
  aRect = [textScrollView documentVisibleRect];
  origin = aRect.origin.y;

  aRect.origin.y -= aRect.size.height - [textScrollView verticalPageScroll];
  [textView scrollRectToVisible: aRect];

  aRect = [textScrollView documentVisibleRect];

  if (aRect.origin.y == origin)
    {
      [self previousMessage: nil];
    }
}



//
// accessor/mutator methods
//
- (NSTextView *) textView
{
  return textView;
}


//
//
//
- (Message *) message
{
  return message;
}


//
// This is the same as the above method. It exists just so that it matches the one in
// MailWindowController.
//
- (Message *) selectedMessage
{
  return message;
}


//
//
//
- (void) setMessage: (Message *) aMessage
{
  NSString *windowTitle = nil;
  
  if ( aMessage )
    {
      RETAIN(aMessage);
      RELEASE(message);
      message = aMessage;
      windowTitle = [message subject];
      
      if (windowTitle)
	{
	  [[self window] setTitle: windowTitle];
	}
    }
}


//
//
//
- (Folder *) folder
{
  return folder;
}


//
// 
//
- (void) setFolder: (Folder *) aFolder
{
  folder = aFolder;
}


//
//
//
- (MailWindowController *) mailWindowController
{
  return mailWindowController;
}


//
//
//
- (void) setMailWindowController: (MailWindowController *) aMailWindowController
{
  if ( aMailWindowController )
    {
      mailWindowController = aMailWindowController;
    }
}


//
//
//
- (BOOL) showAllHeaders
{
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWALLHEADERS"] )
    {
      return ([[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWALLHEADERS"] intValue] == NSOnState ? YES
	      : showAllHeaders);
    }
  
  return showAllHeaders;
}


//
//
//
- (void) setShowAllHeaders: (BOOL) aBool
{
  showAllHeaders = aBool;
}


//
//
//
- (BOOL) showRawSource
{
  return showRawSource;
}


//
//
//
- (void) setShowRawSource: (BOOL) aBool
{
  showRawSource = aBool;
}


//
//
//
- (MailHeaderCell *) mailHeaderCell
{
  return mailHeaderCell;
}


//
//
//
- (int) indexOffset
{
  return indexOffset;
}


//
//
//
- (void) setIndexOffset: (int) theIndexOffset
{
  indexOffset = theIndexOffset;
}


//
// delegate methods
//
- (void) windowDidLoad
{
#ifdef MACOSX
  [[self window] setFrameAutosaveName: @"MessageViewWindow"];
  [[self window] setFrameUsingName: @"MessageViewWindow"];
#endif  

  // We set the last window on top
  [GNUMail setLastMailWindowOnTop: [self window]]; 
}


//
//
//
- (void) windowDidBecomeKey: (NSNotification *) aNotification
{  
  // We clear our 'Save Attachment' menu
  [(GNUMail *)[NSApp delegate] removeAllItemsFromMenu: [(GNUMail *)[NSApp delegate] saveAttachmentMenu]];
  
  // We set the last window on top
  [GNUMail setLastMailWindowOnTop: [self window]];
  
  // Post a notification to GNUMail.m so that the menu gets updated.
  [[NSNotificationCenter defaultCenter]
    postNotificationName: SelectionOfMessageHasChanged
    object: nil
    userInfo: nil];

  // We adjust our showOrHideAllHeaders button
  [showOrHideAllHeaders setTag: [[(GNUMail *)[NSApp delegate] showAllHeadersMenuItem] tag]];
  [showOrHideAllHeaders setTitle: [[(GNUMail *)[NSApp delegate] showAllHeadersMenuItem] title]];
}


//
//
//
- (void) windowWillClose: (NSNotification *) theNotification
{
  // We update our last mail window on top
  if ( [GNUMail lastMailWindowOnTop] == [self window] )
    {
      [GNUMail setLastMailWindowOnTop: nil];
    }

  // We remove our window controller from the list of opened window controllers
  // of the "parent" MailWindowController object.
  [[[self mailWindowController] allMessageViewWindowControllers] removeObject: self];
  
  AUTORELEASE(self);
}


//
//
//
-  (void) textView: (NSTextView *) aTextView
     clickedOnCell: (id <NSTextAttachmentCell>) attachmentCell
	    inRect: (NSRect) cellFrame
	   atIndex: (unsigned) charIndex
  
{
  [Utilities clickedOnCell: attachmentCell
	     inRect: cellFrame
	     atIndex: charIndex
	     sender: self];
}


//
//
//
- (BOOL) textView: (NSTextView *) textView
    clickedOnLink: (id) link 
	  atIndex: (unsigned) charIndex
{
  NSDebugLog(@"Opening %@...", [link description]);
  return [[NSWorkspace sharedWorkspace] openURL: link];
}


@end


//
// Private interface
//
@implementation MessageViewWindowController (Private)

- (void) _loadAccessoryViews
{
  int i, index;

  index = 0;

  for (i = 0; i < [[GNUMail allBundles] count]; i++)
    {
      id<GNUMailBundle> aBundle;

      aBundle = [[GNUMail allBundles] objectAtIndex: i];
      
      if ( [aBundle hasViewingViewAccessory] )
	{
	  id aView;
	  
	  aView = [aBundle viewingViewAccessory];
	  
	  if ( [aBundle viewingViewAccessoryType] == ViewingViewTypeHeaderCell )
	    {
	      NSDebugLog(@"Adding ViewingViewTypeHeaderCell type of Bundle...");
	      [mailHeaderCell addView: aView];
	    }
	}

      // We also set the current superview
      [aBundle setCurrentSuperview: [[self window] contentView]];
    }
}

@end
