#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can` redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# DataSourceEditor.py
#
# DESCRIPTION:
# Class that implements a "Data Source" viewer/editor Tool Panel
#
# NOTES:
#

import sys, os, cPickle, traceback, StringIO, string
from wxPython.wx import *
from gnue.common.apps import GDebug, GConfig, RuntimeSettings
from gnue.common.datasources import GDataSource
from gnue.common import events
from gnue.designer.base.ToolBase import ToolBase

class DataSourceEditor(ToolBase):

  runtime_section = "DataSourceEditor"

  def init(self):
    self.connections = self.instance._app.connections

    self.combopanel = wxPanel(self, -1, pos=wxPoint(0,0))
    self.splitter = wxSplitterWindow(self, -1, style=wxSP_3D)
##    self.toolBar = ToolBar(self, parentWindow)

    self.list = wxListCtrl(self.splitter, -1, pos=wxPoint(0,0),
             style=wxLC_REPORT| # wxLC_EDIT_LABELS|
                   wxLC_HRULES|wxLC_VRULES|wxLC_SINGLE_SEL)

    self.list.InsertColumn(0,_('Name'))
    self.list.InsertColumn(1,_('Source'))
    self.list.InsertColumn(2,_('Connection'))

    self.notebook = wxNotebook(self.splitter, -1, style=wxNB_BOTTOM)

    self.propertyPanel = PropertyViewer(self, self.notebook)
    self.referencesPanel = ReferencesViewer(self, self.notebook)
    self.schemaPanel = SchemaViewer(self, self.notebook)

    self.notebook.AddPage(self.propertyPanel, _('Properties'))
    self.notebook.AddPage(self.referencesPanel, _('References'))
    self.notebook.AddPage(self.schemaPanel, _('Schema'))

    self.splitter.SplitHorizontally(self.list, self.notebook,200)

    EVT_SIZE(self, self.OnSize)

    self.datasourceMap = {}
    self.datasources = []

    self.current = None

    self.instance.rootObject.walk (self.initialInventory)
    self.instance.rootObject.walk (self.inventoryObject)
    self.fillList()

    # EventAware provided by ToolBase
    self.registerEventListeners({
                       'ObjectSelected'      : self.onSetCurrentObject,
                       'ObjectCreated'       : self.onCreateObject,
                       'ObjectModified'      : self.onModifyObject,
                       'ObjectDeleted'       : self.onDeleteObject,
                      })

    EVT_LIST_ITEM_SELECTED(self, self.list.GetId(), self.OnDSSelected)
    self._setCurrent(self.current)

    self.finalized = 0
    self.splitter.SetSize(self.GetSize())


  def fillList(self):

    self.list.DeleteAllItems()

    for datasource in self.datasources:
      if not datasource.__master:
        self._fillList(datasource, 0)


  def _fillList(self, object, indent):

    index = self.list.GetItemCount()
    self.list.InsertStringItem(index, " * " * indent + object.name)
    self.list.SetStringItem(index, 1,
       hasattr(object,'table') and object.table or '')
    self.list.SetStringItem(index, 2,
       hasattr(object,'database') and object.database or '')
    self.list.SetItemData(index, id(object))

    object.__listIndex = index

    for detail in object._datasourceReferences:
      self._fillList(detail, indent + 1)




  def initialInventory(self, object):
    if isinstance(object, GDataSource.GDataSource):
      self.datasourceMap[string.lower(object.name)] = object
      self.datasourceMap[id(object)] = object
      self.datasources.append(object)
      object._fieldFkReferences = {}
      object._blockReferences = []
      object._datasourceReferences = []
      if self.current is None:
        self.current = object


  def inventoryObject(self, object):
    if isinstance(object, GDataSource.GDataSource):
      if hasattr(object,'master') and object.master:
        try:
          object.__master = self.datasourceMap[string.lower(object.master)]
        except KeyError:
          # TODO: Something better than this
          print "DataSource %s references non-existent master %s" % \
              (object.name, object.master)
      else:
        object.__master = None

    #
    # Create a cross-reference list of objects referring to datasources
    # These are used by the ReferencesViewer
    #

    if object._type == 'GFEntry' and \
      hasattr(object,'foreign_key') and \
      len(object.foreign_key):

      ds, field = string.split(
             string.lower(object.foreign_key),'.')

      try:
        self.datasourceMap[ds]._fieldFkReferences[field].append(object)
      except KeyError:
        self.datasourceMap[ds]._fieldFkReferences[field] = [object]

    elif object._type == 'GFBlock' and \
       hasattr(object,'datasource') and \
       len(object.datasource):

      ds = self.datasourceMap[string.lower(object.datasource)]

      if not object in ds._blockReferences:
        ds._blockReferences.append(object)


    elif isinstance(object,GDataSource.GDataSource) and \
       hasattr(object,'master') and \
       object.master:

      ds = self.datasourceMap[string.lower(object.master)]

      if not object in ds._datasourceReferences:
        ds._datasourceReferences.append(object)



  def onSetCurrentObject (self, event):
    object = event.object
    handler = event.originator
    self._currentObject = object
    if object == None:
      return

    if handler != __name__:
      self._setCurrent(object)

  def _setCurrent(self, object):
    if isinstance(object, GDataSource.GDataSource):

      self.current = object
      self.list.SetItemState ( object.__listIndex,
                               wxLIST_STATE_SELECTED,
                               wxLIST_STATE_SELECTED)

      self.list.EnsureVisible(object.__listIndex)

    self.propertyPanel.setCurrent(object)
    self.referencesPanel.setCurrent(object)
    self.schemaPanel.setCurrent(object)


  def onCreateObject (self, event):
    object = event.object
    handler = event.originator
    self._currentSelection = {}
    if object == None:
      return

    if handler != __name__:
      self.initialInventory(object)
      self.inventoryObject(object)

    if isinstance(object, GDataSource.GDataSource):
      self.fillList()


  def onModifyObject (self, event):
    object = event.object
    handler = event.originator
    if object == None:
      return

    if handler != __name__:

      if isinstance(object, GDataSource.GDataSource):
        # TODO: Maintain self.datasourceMap
        self.fillList()

      # Dirty way to rebuild all our lists
      if object == self.current:
        self._setCurrent(object)


  def onDeleteObject (self, event):
    object = event.object
    handler = event.originator
    if object == None:
      return

    if isinstance(object, GDataSource.GDataSource):

      index = object.__listIndex

      i = j = 0
      for datasource in self.datasources:
        if datasource == object: j = i
        if datasource.__listIndex > index:
          datasource.__listIndex -= 1
          i += 1


      self.datasources.pop(j)
      self.list.DeleteItem(index)

      del self.datasourceMap[string.lower(object.name)]
      del self.datasourceMap[id(object)]


  def OnSize(self, event):
    if not self.finalized:
      self.finalize()
    self.splitter.SetSize(self.GetSize())


  # Runtime Setting Support

  def finalize(self):

    self.finalized = 1

    self.SetPosition((
       RuntimeSettings.getint(self.runtime_section, 'x', -1),
       RuntimeSettings.getint(self.runtime_section, 'y', -1)))

    self.splitter.SetSashPosition(
       RuntimeSettings.getint(
           self.runtime_section, 'sash', 100) or 100)

    self.notebook.SetSelection(
       RuntimeSettings.getint(
           self.runtime_section, 'visibletab', 0))

    for i in range(3):  # Should be the number of columns in the list
      self.list.SetColumnWidth(i,
        RuntimeSettings.getint(
           self.runtime_section, "col%s" % i, -2))

    for i in range(REF_COLUMNS):
      self.referencesPanel.list.SetColumnWidth(i,
        RuntimeSettings.getint(
           self.runtime_section, "refCol%s" % i, -2))

    for i in range(SCH_COLUMNS):
      self.schemaPanel.list.SetColumnWidth(i,
        RuntimeSettings.getint(
           self.runtime_section, "schemaCol%s" % i, -2))

    self.list.Refresh()
    self.Refresh()


  def saveRuntimeSettings(self):
    sash = self.GetPositionTuple()

    settings = {
              'sash': self.splitter.GetSashPosition(),
              'visibletab': self.notebook.GetSelection()
             }

    # Save the top-level list column sizes
    for i in range(3):  # Should be the number of columns in the list
      settings["col%s" % i] = self.list.GetColumnWidth(i)

    # Save the reference viewer column sizes
    for i in range(REF_COLUMNS):
      settings["refCol%s" % i] = self.referencesPanel.list.GetColumnWidth(i)

    # Save the schema viewer column sizes
    for i in range(SCH_COLUMNS):
      settings["schemaCol%s" % i] = self.schemaPanel.list.GetColumnWidth(i)

    return ( self.runtime_section,
             settings )


  # A Datasource was selected in the menu
  def OnDSSelected(self, event):
    self.current = self.datasourceMap[event.GetData()]
    self.instance.dispatchEvent(events.Event('ObjectSelected',object=self.current, originator=__name__))


  def OnAddDSSelected(self, event):

    attributes = {}
    if self.current is not None:
      attributes['database'] = \
          hasattr(self.current,'database') and self.current.database or ''

    self.instance.incubator.createObject(self.rootObject, 'datasource',
                           parent=self.rootObject, attributes=attributes)


  def OnDeleteDSSelected(self, event):

    if self.current is not None:
      self.dispatchEvent('ObjectDeleted',object=self.current, originator=__name__)



#
#
#
class ToolBar (wxToolBar):

  def __init__(self, editor, parent):
    wxToolBar.__init__(self, parent, -1, style=wxTB_DOCKABLE)
    self.editor = editor
    parent.SetToolBar(self)

    self.addButtonId = wxNewId()
    self.deleteButtonId = wxNewId()

    self.AddSimpleTool(self.addButtonId,
        wxImage(images_dir+gConfig('tb_insert'),
                wxBITMAP_TYPE_PNG).ConvertToBitmap(),
        _("Create Data Source"),
        _("Create a new data source"))

    self.AddSimpleTool(self.deleteButtonId,
        wxImage(images_dir+gConfig('tb_delete'),
                wxBITMAP_TYPE_PNG).ConvertToBitmap(),
        _("Delete Data Source"),
        _("Delete the currently selected data source"))

    EVT_TOOL(self, self.addButtonId, editor.OnAddDSSelected)
    EVT_TOOL(self, self.deleteButtonId, editor.OnDeleteDSSelected)


    self.dblinks = []


#############################################################################
#
# Properties Tab
#

class PropertyViewer(wxPanel):
  def __init__(self, editor, parentWindow):
    wxPanel.__init__(self, parentWindow, -1)
    self.editor = editor

    EVT_SIZE(self, self.OnSize)


  def setCurrent(self, object):
    if isinstance(object, GDataSource.GDataSource):
      self.current = object


  def OnSize(self, event):
    event.Skip()



#############################################################################
#
# References Tab
#

REF_NAME = 0
REF_REFR = 1
REF_TYPE = 2
REF_COLUMNS = REF_TYPE + 1

class ReferencesViewer(wxPanel):
  def __init__(self, editor, parentWindow):
    wxPanel.__init__(self, parentWindow, -1)
    self.editor = editor

    self.list = wxListCtrl(self, -1, pos=wxPoint(0,0),
             style=wxLC_REPORT|wxLC_HRULES|wxLC_VRULES)

    self.list.InsertColumn(REF_NAME,  _('Reference'))
    self.list.InsertColumn(REF_REFR,  _('Referrer'))
    self.list.InsertColumn(REF_TYPE,  _('Type'), wxLIST_FORMAT_RIGHT)

    EVT_SIZE(self, self.OnSize)

  def setCurrent(self, object):
    if isinstance(object, GDataSource.GDataSource):
      self.current = object
      self.fillList()

  def OnSize(self, event):
    self.list.SetSize(self.GetSize())

  def fillList(self):


    if not hasattr(self.current,'table'):
      return

    mastertable = self.current.table

    self.lines = {}

    def addLine(self, name, referrer, type):
      name = string.lower(name)

      if not self.lines.has_key(name):
        self.lines[name] = []

      self.lines[name].append((name, referrer, type))


    for item in self.current._datasourceReferences:

      mfields = string.split(item.masterlink,'.')
      dfields = string.split(item.detaillink,'.')

      for i in range(len(mfields)):
        addLine( self, "%s.%s" % (mastertable, mfields[i]),
                 "%s.%s" % (item.name, dfields[i]),
                 _("Detail Datasource") )


    for field in self.current._fieldFkReferences.keys():
      for item in self.current._fieldFkReferences[field]:
        addLine( self, "%s.%s" % (mastertable, field),
                 "%s.%s" % (item._parent.name, item.name),
                 _("Dropdown Entry") )


    for block in self.current._blockReferences:
      addLine (self, mastertable, block.name, "Block")

      for item in block._children:

        if item._type == 'GFEntry' and \
           hasattr(item,'field') and \
           len(item.field):

          addLine( self, "%s.%s" % (mastertable, item.field),
                   "%s.%s" % (block.name, item.name),
                   _("Entry"))

    index = 0
    self.list.DeleteAllItems()
    keys = self.lines.keys()
    keys.sort()

    index = 0
    for key in keys:

      newline = 0
      self.list.InsertStringItem(index, key)

      for line in self.lines[key]:
        if newline:
          self.list.InsertStringItem(index, "")
        else:
          newline = 1


        name, referrer, type = line
        self.list.SetStringItem(index, REF_REFR, referrer)
        self.list.SetStringItem(index, REF_TYPE, type)
        index += 1






#############################################################################
#
# Schema Viewer Tab
#

SCH_FIELD = 0
SCH_TYPE = 1
SCH_SIZE = 2
SCH_NATIVE = 3
SCH_REQ = 4
SCH_COLUMNS = SCH_REQ + 1

class SchemaViewer(wxPanel):
  def __init__(self, editor, parentWindow):
    wxPanel.__init__(self, parentWindow, -1)
    self.editor = editor

    self.list = wxListCtrl(self, -1, pos=wxPoint(0,0),
             style=wxLC_REPORT|wxLC_HRULES|wxLC_VRULES)

    self.list.InsertColumn(SCH_FIELD, _('Field'))
    self.list.InsertColumn(SCH_TYPE,  _('Base Type'))
    self.list.InsertColumn(SCH_SIZE,  _('Size'), wxLIST_FORMAT_RIGHT)
    self.list.InsertColumn(SCH_NATIVE,_('Native Type'))
    self.list.InsertColumn(SCH_REQ,   _('Required'))

    self.schemaMap = []

    EVT_SIZE(self, self.OnSize)
    EVT_LIST_BEGIN_DRAG(self, self.list.GetId(), self.OnBeginDrag)


  def fillList(self):

   try:
    conn = self.editor.current.database

    if not self.editor.connections.isConnectionActive(conn) and \
       not int(gConfig("AutoConnect",section="designer")):

      # TODO: Some other form of feedback
      if not hasattr(self.editor,'_schemaMessageDisplayed'):
        print _("You are not logged in to %s... not retrieving schema") % (conn)
        self.editor._schemaMessageDisplayed = 1

      return

    # TODO: Much error catching... any DB errors should be
    # TODO: trapped so the designing session can continue,
    # TODO: albeit without schema support.


    # TODO: This is the old, deprecated syntax
    dataObject = self.editor.connections.getDataObject(conn, "object")

    self.editor.connections.requestConnection(dataObject, conn)

    schema = dataObject._connection.introspector.findone(name=self.editor.current.table)


    index = 0
    self.list.DeleteAllItems()
    if schema:
      for field in schema.fields():
        self.list.InsertStringItem(index, field.name)
        self.list.SetStringItem(index, SCH_TYPE,
           string.upper(field.datatype[0]) + field.datatype[1:])
        self.list.SetStringItem(index, SCH_NATIVE,
           hasattr(field, 'nativetype') and field.nativetype or '(unknown)')
        self.list.SetStringItem(index, SCH_REQ,
           field.required and "Yes" or "No")
        if hasattr(field,'length'):
          if hasattr(field,'precision') and field.precision > 0:
            self.list.SetStringItem(index, SCH_SIZE, "%s;%s" % (
                  field.length, field.precision))
          else:
            self.list.SetStringItem(index, SCH_SIZE, "%s" % field.length)
        else:
          self.list.SetStringItem(index, SCH_SIZE,'-')
        self.list.SetItemData(index, index)
        self.schemaMap.append(field)
        index += 1

   except:

     buffer = StringIO.StringIO()
     traceback.print_exc(file=buffer)

     GDebug.printMesg(5,"Exception when retrieving schema (last 5 lines):")
     GDebug.printMesg(5,"%s"%buffer.getvalue())
     buffer.close()

  def setCurrent(self, object):
    if isinstance(object, GDataSource.GDataSource):
      self.current = object
      self.fillList()

  def OnSize(self, event):
    self.list.SetSize(self.GetSize())


  def OnBeginDrag(self, event):

    data = []

    item = -1
    # Cycle through each selected item
    while 1:
      item = self.list.GetNextItem(item,
                                   wxLIST_NEXT_ALL,
                                   wxLIST_STATE_SELECTED);
      if item == -1:
        break

      object = self.schemaMap[item]

      attributes = {
          "datasource": self.current.name,
          "name": self.editor.instance.getUniqueName(
                    "fld%s" % (
                     string.join(string.split(string.capwords( \
                       string.replace(object.name,'_',' '))),''))),
          "field": object.name,
          "datatype": object.datatype,
          "required" : object.required,
        }

      if hasattr(object,'length') and object.length:
        attributes['max_length'] = object.length
        attributes['width'] = object.length < 30 and object.length or 30


      if hasattr(object, 'label') and len(object.label):
        label = object.label
      elif len(object.name) == 1:
        label = string.upper(object.name)
      else:
        label = string.join(string.split(string.capwords( \
                     string.replace(object.name,'_',' '))),' ')

      data.append({ "Type" : "entry",
                    "Attributes": attributes,
                    "Label": label})


    do = wxCustomDataObject(wxCustomDataFormat("application/x-gnue-designer"))
    do.SetData(cPickle.dumps(data,1))

    dropSource = wxDropSource(self)
    dropSource.SetData(do)
    result = dropSource.DoDragDrop(false)


# Get around a few wx 2.2.x shortcomings:
try:
  wxLC_HRULES
  wxLC_VRULES
except NameError:
  wxLC_HRULES=0
  wxLC_VRULES=0


# Shortcut
images_dir = os.path.join(GConfig.getInstalledBase('designer_images','common_images'),'designer')


