#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can` redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# DataSourceEditor.py
#
# DESCRIPTION:
# Class that implements a generic grid editor in which
# each row of the grid corresponds to GFObject
#
# NOTES:
#

__all__ = ['MultiObjectGridEditor']

import sys, os, cPickle, traceback, StringIO, string
import weakref, types
from wxPython.wx import *
from wxPython.grid import *
from gnue.common.apps import GDebug, RuntimeSettings


class MultiObjectGridEditor(wxGrid):

  ##
  ##  Stuff that needs to be implemented by the subclasses
  ##

  base_gobject = None  # GObject

  # Any initialization stuff
  def init(self):
    pass

  # Should return a tuple of columns
  def getColumnDefinitions(self):
    return ()


  ##
  ## Internal Stuff
  ##
  def __init__(self, parent, instance):
    wxGrid.__init__(self, parent, -1)

    self.instance = instance
    self.objects = []
    self.rows = []
    self.current = None
    self.lastSort = -1
    self.lastReverse = 0

    # Set up our grid


    # Register the Designer event handlers
    instance.registerEventListeners({
                       'ObjectSelected'      : self._objectSelected,
                       'ObjectCreated'       : self._objectCreated,
                       'ObjectModified'      : self._objectModified,
                       'ObjectDeleted'       : self._objectDeleted,
                      })

    #
    # Register the wx event handlers
    #
    EVT_GRID_CELL_CHANGE(self, self.__OnCellChange)
    EVT_GRID_SELECT_CELL(self, self.__OnSelectCell)
    EVT_GRID_LABEL_LEFT_CLICK(self, self.__OnLabelLeftClick)

    self.init()
    self.columns = columns = self.getColumnDefinitions()

    #
    # Create the grid
    #

    self.CreateGrid(1,len(columns))
#    self.SetEditable(1)
    self.SetSelectionMode(wxGrid.wxGridSelectRows)
    self.BeginBatch()

    self.SetRowLabelSize(0)

    index = 0
    for column in columns:
      attr = wxGridCellAttr()
      try:
        readonly = column['ReadOnly']
        attr.SetReadOnly(column['ReadOnly'])
      except KeyError:
        pass

      self.SetColLabelValue(index, column['Label'])
      self.SetColAttr(index, attr)

      index += 1



    # Initial inventory
    instance.rootObject.walk (self.inventoryObject)

    self.__rebuildGrid()
    self.EndBatch()


  #
  # Rebuild an individual row in the visible grid
  #
  def __rebuildRow(self, ri):
    self.BeginBatch()

    # Add any necessary rows
    if ri + 1 > self.GetNumberRows():
      self.AppendRows(ri - self.GetNumberRows() + 1)

    row = self.rows[ri]
    ci = 0
    for column in self.columns:
      v = column['Attribute']
      if type(v) == types.MethodType:
        value = v(row)
      else:
        try:
          value = str(row.__dict__[v])
        except KeyError:
          value = ""
      self.SetCellValue(ri, ci, value)
      ci += 1
    self.EndBatch()


  #
  # Rebuild the visible grid.
  # If sort != -1, then sort is
  # the column # to sort by.
  #
  def __rebuildGrid(self, sort=-1, reverse=0):

    # Save the sorting preference
    self.lastSort = sort
    self.lastReverse = reverse

    # Sort, if appropriate
    if sort >= 0:
      column = self.columns[sort]['Attribute']
      rows = []
      for object in self.objects:
        # If key is dynamically created, create it
        if type(column) == types.MethodType:
          key = column(object)
        else:
        # Otherwise get the value, if available
          try:
            key = object.__dict__[column]
          except KeyError:
            key = None
        try:
          # If the key is a string, lowercase it
          key = key.lower()
        except AttributeError:
          pass
        rows.append ( (key, object) )
      rows.sort()
      if reverse:
        rows.reverse()
      self.rows = []
      for key, object in rows:
        self.rows.append(object)
    else:
      self.rows = self.objects[:]

    # .. and update the visible grid
    num = len(self.rows)
    self.BeginBatch()

    # Create or delete any excess rows
    existing = self.GetNumberRows()
    if num < existing:
      self.DeleteRows(0, existing - num)
    elif num > existing:
      self.AppendRows(num - existing)

    # And update each individual row
    for i in range(len(self.rows)):
      self.__rebuildRow(i)

    if self.current in self.rows:
      self.SelectRow(self.rows.index(self.current))
    else:
      try:
        self.current = self.rows[0]
      except IndexError:
        self.current = None

    self.EndBatch()


  #
  #
  #
  def saveRuntimeSettings(self):
    settings = {}

    # Save the top-level list column sizes
    for i in range(3):  # Should be the number of columns in the list
      settings["col%s" % i] = self.list.GetColumnWidth(i)

    # Save the reference viewer column sizes
    for i in range(REF_COLUMNS):
      settings["refCol%s" % i] = self.referencesPanel.list.GetColumnWidth(i)

    # Save the schema viewer column sizes
    for i in range(SCH_COLUMNS):
      settings["schemaCol%s" % i] = self.schemaPanel.list.GetColumnWidth(i)

    return ( self.runtime_section,
             settings )


  #
  # Add the object to our inventory, if applicable
  # If rebuild=1, then the visible grid is rebuilt
  #
  def inventoryObject(self, object, rebuild=0):
    if isinstance(object, self.base_gobject):
      self.objects.append(object)
      if rebuild:
        if self.lastSort > 0:
          self.__rebuildGrid(sort=self.lastSort, reverse=self.lastReverse)
        else:
          self.rows.append(object)
          self.__rebuildRow(len(self.rows)-1)


  #
  # ObjectSelected event
  #
  def _objectSelected(self, event):
    if event.originator == self:
      return
    object = event.object
    if object in self.objects:
      self.current = object
      self.SelectRow(self.rows.index(object))
      index = self.objects.index(object)


  #
  # ObjectModified event
  #
  def _objectModified(self, event):
    if event.originator == self:
      return
    if event.object in self.objects:
      if self.lastSort > 0:
        self.__rebuildGrid(sort=self.lastSort, reverse=self.lastReverse)
      else:
        self.__rebuildRow(self.rows.index(event.object))

  #
  # ObjectCreated event
  #
  def _objectCreated(self, event):
    self.inventoryObject(event.object, 1)


  #
  # ObjectDeleted event
  #
  def _objectDeleted(self, event):
    object = event.object
    if object in self.objects:
      self.BeginBatch()
      self.objects.remove(object)
      index = self.rows.index(object)
      self.rows.remove(object)
      self.DeleteRows(index,1)
      self.EndBatch()


  #
  # Called by the subclass to notify us that
  # a valid set has changed
  #
  def updateValidSet(self, column):
    i = 0
    while i < len(self.columns):
      if self.columns[i]['Name'] == column:
        break
      i += 1
    values = self.columns[i]['ValidSet']
    if type(values) == types.MethodType:
      values = values()
    print "TODO: UPDATING VALID SET"


  ##
  ## Wx event handlers
  ##
  def __OnCellChange(self, event):
    r = event.GetRow()
    c = event.GetCol()
    val = self.GetCellValue(r,c)
    object = self.rows[r]
    column = self.columns[c]
    attribute = column['Attribute']
    try:
      old = {attribute: object.__dict__[attribute] }
    except KeyError:
      old = {}

    self.instance.dispatchEvent('ObjectModified',
                                object=object, originator=self,
                                old=old, new = {attribute: val})


  #
  # User selected a new cell (row)
  #
  def __OnSelectCell(self, event):
    # Fire an ObjectSelected event
    object = self.rows[event.GetRow()]
    if object != self.current:
      self.instance.dispatchEvent('ObjectSelected',
                                  object=self.rows[event.GetRow()],
                                  originator = self)
      self.current = object
    event.Skip()

  #
  # User clicked the top label; resort by that column
  #
  def __OnLabelLeftClick(self, event):
    sort = event.GetCol()
    reverse = (sort == self.lastSort) and not self.lastReverse
    self.__rebuildGrid(sort=sort, reverse=reverse)


