/* vi:set ts=8 sts=0 sw=8:
 * $Id: file.c,v 1.5 2001/01/27 17:31:31 kahn Exp kahn $
 *
 * Copyright (C) 2000 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <pwd.h>
#include <grp.h>
#include <gnome.h>
#include "file.h"
#include "win.h"
#include "results.h"
#include "misc.h"
#include "prefs.h"
#include "msgbar.h"
#include "pathmax.h"
#include "debug.h"


/*** PRIVATE FUNCTION PROTOTYPES ***/
static void file_info_stat(win_t *w, GtkWidget **wgts, char *fname);
static void file_copy_ok_cb(GtkWidget *wgt, gpointer cbdata);
static void file_copy_cancel_cb(GtkWidget *wgt, gpointer cbdata);
static void cmd_archive_common(char *msg, win_t *w, char *cmd,
			       gboolean repl_files, gboolean report_err);
static void file_list_read(win_t *w, char *fname);


/*
 * Data needed in the callback when "Ok" is clicked after entering the
 * destination name for copying files.
 */
struct file_copy_data_s {
	win_t *w;
	GtkWidget *entry;	/* text entry widget */
	GtkWidget *dlg;		/* toplevel dialog widget */
};
typedef struct file_copy_data_s file_copy_data_t;


enum fileinfo_e {
	Ef_finfo_toplev,	/* dialog */
	Ef_finfo_vbox,		/* vbox */
	Ef_finfo_ok,		/* button */
	Ef_finfo_MAX
};
typedef enum fileinfo_e fileinfo_t;

enum fileinfo_data_e {
	Ef_fidata_toplev,	/* dialog */
	Ef_fidata_frame,	/* table */
	Ef_fidata_fname,	/* label */
	Ef_fidata_fsize,	/* label */
	Ef_fidata_perm,		/* label */
	Ef_fidata_inode,	/* label */
	Ef_fidata_links,	/* label */
	Ef_fidata_uid,		/* label */
	Ef_fidata_gid,		/* label */
	Ef_fidata_atime,	/* label */
	Ef_fidata_mtime,	/* label */
	Ef_fidata_ctime,	/* label */
	Ef_fidata_bsize,	/* label */
	Ef_fidata_blocks,	/* label */
	Ef_fidata_MAX
};
typedef enum fileinfo_data_e fileinfo_data_t;

static ef_t ef_fileinfo_widgets[] = {
	{
		"File Properties",
		Ef_finfo_toplev,
		Ef_wgt_dialog,
	},
	{
		"vbox_file_info",
		Ef_finfo_vbox,
		Ef_wgt_vbox,
	},
	{
		"but_finfo_ok",
		Ef_finfo_ok,
		Ef_wgt_button,
	},
	{ NULL }
}; /* ef_fileinfo_widgets */

static ef_t ef_fileinfo_data[] = {
	{
		"File Info Data",
		Ef_fidata_toplev,
		Ef_wgt_dialog,
	},
	{
		"frame_file_info",
		Ef_fidata_frame,
		Ef_wgt_frame,
	},
	{
		"label_fname",
		Ef_fidata_fname,
		Ef_wgt_label,
	},
	{
		"label_fsize",
		Ef_fidata_fsize,
		Ef_wgt_label,
	},
	{
		"label_perm",
		Ef_fidata_perm,
		Ef_wgt_label,
	},
	{
		"label_inode",
		Ef_fidata_inode,
		Ef_wgt_label,
	},
	{
		"label_links",
		Ef_fidata_links,
		Ef_wgt_label,
	},
	{
		"label_uid",
		Ef_fidata_uid,
		Ef_wgt_label,
	},
	{
		"label_gid",
		Ef_fidata_gid,
		Ef_wgt_label,
	},
	{
		"label_atime",
		Ef_fidata_atime,
		Ef_wgt_label,
	},
	{
		"label_mtime",
		Ef_fidata_mtime,
		Ef_wgt_label,
	},
	{
		"label_ctime",
		Ef_fidata_ctime,
		Ef_wgt_label,
	},
	{
		"label_bsize",
		Ef_fidata_bsize,
		Ef_wgt_label,
	},
	{
		"label_blocks",
		Ef_fidata_blocks,
		Ef_wgt_label,
	},
	{ NULL }
}; /* ef_fileinfo_data[] */

enum filecopy_e {
	Ef_fcopy_toplev,	/* dialog */
	Ef_fcopy_msg,		/* label */
	Ef_fcopy_dest,		/* entry */
	Ef_fcopy_ok,		/* button */
	Ef_fcopy_cancel,	/* button */
	Ef_fcopy_MAX
};
typedef enum filecopy_e filecopy_t;

static ef_t ef_filecopy_widgets[] = {
	{
		"File Entry Dialog",
		Ef_fcopy_toplev,
		Ef_wgt_dialog,
	},
	{
		"entry_dlg_label",
		Ef_fcopy_msg,
		Ef_wgt_label,
	},
	{
		"entry_dlg_entry",
		Ef_fcopy_dest,
		Ef_wgt_entry,
	},
	{
		"entry_dlg_but_ok",
		Ef_fcopy_ok,
		Ef_wgt_button,
	},
	{
		"entry_dlg_but_cancel",
		Ef_fcopy_cancel,
		Ef_wgt_button,
	},
	{ NULL }
}; /* ef_filecopy_widgets[] */

enum shellcmd_e {
	Ef_shell_dialog,	/* dialog */
	Ef_shell_msg,		/* label */
	Ef_shell_entry,		/* entry */
	Ef_shell_ok,		/* button */
	Ef_shell_cancel,	/* button */
	Ef_shell_MAX
};
typedef enum shellcmd_e shellcmd_t;

static ef_t ef_shell_widgets[] = {
	{
		"Text Entry Dialog",
		Ef_shell_dialog,
		Ef_wgt_dialog,
	},
	{
		"entry_dlg_label",
		Ef_shell_msg,
		Ef_wgt_label,
	},
	{
		"entry_dlg_entry",
		Ef_shell_entry,
		Ef_wgt_entry,
	},
	{
		"entry_dlg_ok",
		Ef_shell_ok,
		Ef_wgt_button,
	},
	{
		"entry_dlg_cancel",
		Ef_shell_cancel,
		Ef_wgt_button,
	},
	{ NULL }
}; /* ef_shell_widgets[] */



/*** PUBLIC FUNCTION DEFINITIONS ***/
FILE *
file_open_fp(const char *fname, const char *mode)
{
	FILE *fp;

	if ((fp = fopen(fname, mode)) == NULL) {
		int val = errno;
		char *buf = g_strdup_printf(_("Could not open '%s'"), fname);
		cmd_err(buf, val);
		g_free(buf);
	}

	return fp;
} /* file_open_fp */


/*
 * The public functions here are all callbacks for the toolbar buttons in the
 * results window.
 */

void
file_open_cb(GtkWidget *wgt, gpointer cbdata)
{
	cmd_exec(_("Opening selected files..."), cbdata,
		 prefs_string_get(Prefs_file_open), TRUE, TRUE);
} /* file_open_cb */


void
file_delete_cb(GtkWidget *wgt, gpointer cbdata)
{
	cmd_exec(_("Deleting selected files..."), cbdata,
		 prefs_string_get(Prefs_file_delete), TRUE, TRUE);
} /* file_delete_cb */


void
file_copy_cb(GtkWidget *wgt, gpointer cbdata)
{
	GtkWidget **wgts;
	int num;
	file_copy_data_t *fcdp;
	win_t *w = (win_t *)cbdata;

	num = (sizeof(ef_filecopy_widgets) / sizeof(ef_filecopy_widgets[0])) -1;
	wgts = my_widgets_setup(w, NULL, num, ef_filecopy_widgets,
				"File Entry Dialog", Ef_fcopy_toplev,
				Ef_fcopy_MAX, FALSE);

	gtk_window_set_title(GTK_WINDOW(wgts[Ef_fcopy_toplev]),
			     _("Copy Files..."));
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fcopy_msg]),
			   _("Enter destination location to copy file(s)"));
	fcdp = g_new(file_copy_data_t, 1);
	fcdp->w = cbdata;
	fcdp->entry = wgts[Ef_fcopy_dest];
	fcdp->dlg = wgts[Ef_fcopy_toplev];

	gtk_signal_connect(GTK_OBJECT(wgts[Ef_fcopy_ok]), "clicked",
			   file_copy_ok_cb, fcdp);
	gtk_signal_connect(GTK_OBJECT(wgts[Ef_fcopy_cancel]), "clicked",
			   file_copy_cancel_cb, fcdp);
	gtk_widget_show(wgts[Ef_fcopy_toplev]);
	g_free(wgts);
} /* file_copy_cb */


void
file_print_cb(GtkWidget *wgt, gpointer cbdata)
{
	cmd_exec(_("Printing selected files"), cbdata,
		 prefs_string_get(Prefs_file_print), TRUE, TRUE);
} /* file_print_cb */


void
file_execute_cb(GtkWidget *wgt, gpointer cbdata)
{
	cmd_exec(_("Executing selected files"), cbdata,
		 prefs_string_get(Prefs_file_exec), TRUE, TRUE);
} /* file_execute_cb */


void
file_add_cb(GtkWidget *wgt, gpointer w)
{
	cmd_archive_common(_("Adding selected files to archive..."), w,
			   prefs_string_get(Prefs_archive_add), TRUE, FALSE);
} /* file_add_cb */


void
file_remove_cb(GtkWidget *wgt, gpointer w)
{
	cmd_archive_common(_("Removing selected files from archive..."), w,
			   prefs_string_get(Prefs_archive_remove), TRUE, FALSE);
} /* file_remove_cb */


struct shell_cmd_data_s {
	win_t *w;
	GtkWidget *entry;
	GtkWidget *dlg;
};
typedef struct shell_cmd_data_s shell_cmd_data_t;


static void
file_shell_cancel_cb(GtkWidget *wgt, gpointer cbdata)
{
	shell_cmd_data_t *scdp = (shell_cmd_data_t *)cbdata;

	gtk_widget_destroy(scdp->dlg);
	g_free(scdp);
}


static void
file_shell_ok_cb(GtkWidget *wgt, gpointer cbdata)
{
	char *txt;
	shell_cmd_data_t *scdp = (shell_cmd_data_t *)cbdata;

	txt = gtk_entry_get_text(GTK_ENTRY(scdp->entry));
	if (!txt || strlen(txt) < 1)
		goto file_shell_out;

	cmd_exec(_("Executing shell command on selected files"), scdp->w,
		 txt, TRUE, TRUE);

file_shell_out:
	file_shell_cancel_cb(NULL, cbdata);
}


void
file_shell_cb(GtkWidget *wgt, gpointer cbdata)
{
	GtkWidget **wgts;
	char *cmd;
	int num;
	shell_cmd_data_t *scdp;
	win_t *w = (win_t *)cbdata;

	num = (sizeof(ef_shell_widgets) / sizeof(ef_shell_widgets[0])) - 1;
	wgts = my_widgets_setup(w, NULL, num, ef_shell_widgets,
				"Text Entry Dialog",
				Ef_shell_MAX, Ef_shell_MAX, TRUE);
	gtk_window_set_title(GTK_WINDOW(wgts[Ef_shell_dialog]),
			     _("Execute a shell command on selected files..."));
	gnome_dialog_set_parent(GNOME_DIALOG(wgts[Ef_shell_dialog]),
				GTK_WINDOW(win_main_toplev(w)));

	gtk_label_set_text(GTK_LABEL(wgts[Ef_shell_msg]),
			   _("Enter the shell command to be exected.\n"
			     "Use $files to indicate selected files."));

	cmd = prefs_string_get(Prefs_file_shell);
	if (cmd && strlen(cmd) > 0)
		gtk_entry_set_text(GTK_ENTRY(wgts[Ef_shell_entry]), cmd);

	scdp = g_new(shell_cmd_data_t, 1);
	scdp->w = w;
	scdp->entry = wgts[Ef_shell_entry];
	scdp->dlg = wgts[Ef_shell_dialog];

	gtk_signal_connect(GTK_OBJECT(wgts[Ef_shell_ok]), "clicked",
			   GTK_SIGNAL_FUNC(file_shell_ok_cb), scdp);
	gtk_signal_connect(GTK_OBJECT(wgts[Ef_shell_cancel]), "clicked",
			   GTK_SIGNAL_FUNC(file_shell_cancel_cb), scdp);
	gtk_widget_show(wgts[Ef_shell_dialog]);
	g_free(wgts);
} /* file_shell_cb */


void
file_list_cb(GtkWidget *wgt, gpointer cbdata)
{
	char *prefstr, *cmd, *tmpfile;

	tmpfile = prefs_string_get(Prefs_archive_fname);
	if (!tmpfile || strlen(tmpfile) < 1) {
		cmd_err(_("No archive file specified"), 0);
		return;
	}
	if (!g_file_exists(tmpfile)) {
		char *buf = g_strdup_printf(
					_("Archive file '%s' does not exist"),
					tmpfile);
		cmd_err(buf, 0);
		g_free(buf);
		return;
	}

	prefstr = prefs_string_get(Prefs_archive_list);
	if (!prefstr || strlen(prefstr) < 1) {
		cmd_err(_("No command prefs specified"), 0);
		return;
	}
 
	if (prefs_bool_get(Prefs_archive_builtin)) {
		tmpfile = my_tempname();
		APPDBG_FILE(("file_list_cb: tmpfile is '%s'\n", tmpfile));
		cmd = g_strdup_printf("%s > %s", prefstr, tmpfile);
		cmd_archive_common(_("Getting archive contents..."), cbdata,
				   cmd, FALSE, TRUE);
		g_free(cmd);
		file_list_read(cbdata, tmpfile);
		(void)unlink(tmpfile);
		g_free(tmpfile);
	} else {
		cmd_archive_common(_("Getting archive contents..."), cbdata,
				   prefstr, FALSE, TRUE);
	}
} /* file_list_cb */


static void
file_info_destroy(GtkWidget *widget, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_free(w->finfowgts);
	w->finfowgts = NULL;
} /* file_info_destroy */


void
file_info_cb(GtkWidget *widget, gpointer cbdata)
{
	int num;
	char *fname;
	GList *l;
	GtkCList *clist;
	GtkWidget *nb, *label, **wgts;
	win_t *w = (win_t *)cbdata;

	num = (sizeof(ef_fileinfo_widgets) /sizeof(ef_fileinfo_widgets[0])) - 1;
	w->finfowgts = my_widgets_setup(w, w->finfowgts, num,
					ef_fileinfo_widgets, "File Properties",
					Ef_finfo_toplev, Ef_finfo_MAX, TRUE);
	gtk_signal_connect(GTK_OBJECT(w->finfowgts[Ef_finfo_toplev]),
			   "delete_event", GTK_SIGNAL_FUNC(gtk_true), NULL);
	gtk_signal_connect(GTK_OBJECT(w->finfowgts[Ef_finfo_toplev]),
			   "destroy", GTK_SIGNAL_FUNC(file_info_destroy), w);
	nb = gtk_notebook_new();
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(nb), TRUE);
	gtk_notebook_popup_enable(GTK_NOTEBOOK(nb));
	gtk_box_pack_start(GTK_BOX(w->finfowgts[Ef_finfo_vbox]),
			   nb, TRUE, TRUE, 0);

	/* for each selected file, create a new notebook page and add it */
	clist = GTK_CLIST(results_clist_widget(w));
	for (l = clist->selection; l; l = l->next) {
		GtkCTreeNode *cnode = (GtkCTreeNode *)l->data;
		GtkCListRow *clrow = (GtkCListRow *)cnode->list.data;
		fname = clrow->cell->u.text;

		/*
		 * We used glade to create the widgets that will actually store
		 * the file's statbuf info.  However, since all we're
		 * interested in are the label widgets (which are stored in a
		 * frame widget), we need to get rid of the top level dialog
		 * window.  To do this, we ref the frame which contains all the
		 * label widgets, unparent the frame from its current parent (a
		 * temporary top-level, dialog widget), add the frame as a
		 * child to a notebook page and then unref the frame widget.
		 */
		num = (sizeof(ef_fileinfo_data) /
					sizeof(ef_fileinfo_data[0])) - 1;
		wgts = my_widgets_setup(w, NULL, num,
					ef_fileinfo_data,
					"File Info Data",
					Ef_fidata_toplev,
					Ef_fidata_MAX, FALSE);

		file_info_stat(w, wgts, fname);
		label = gtk_label_new(g_basename(fname));
		gtk_widget_ref(wgts[Ef_fidata_frame]);
		gtk_widget_unparent(wgts[Ef_fidata_frame]);
		gtk_notebook_append_page(GTK_NOTEBOOK(nb),
					 wgts[Ef_fidata_frame], label);
#if 0
		/*
		 * XXX - would really like to get rid of top-level dialog
		 * widget, but this also kills the frame, table, and labels
		 * which were originally attached... even though they were
		 * unparented and given a new parent just a few lines ago!  I'm
		 * probably misunderstanding something about parent/child
		 * relationships in GTK widgets.
		 */
		gtk_widget_destroy(wgts[Ef_fidata_toplev]);
#endif
		gtk_widget_unref(wgts[Ef_fidata_frame]);
		g_free(wgts);
	}

	gtk_widget_show_all(w->finfowgts[Ef_finfo_toplev]);
} /* file_info_cb */


/*** PRIVATE FUNCTION DEFINITIONS ***/
/*
 * stat() 'fname' and fill in wgts[] with corresponding info.
 */
#ifndef S_IFLNK
#define lstat stat
#endif

#define BUFLEN 64
static void
file_info_stat(win_t *w, GtkWidget **wgts, char *fname)
{
	extern void filemodestring(struct stat *statp, char *str);
	struct passwd *pwd;
	struct group *grp;
	struct stat lsb;
	char buf[BUFLEN];
	char *timestr;
	char *tmpbuf;
	int err;
#ifdef S_ISLNK
	char buf2[BUFLEN];
	struct stat sb, *sbp;
	char *link;
	extern char *get_link_name(win_t *w, char *name, char *relname);
#endif

	if (!fname) {
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_fname]),
				   _("No filename specified"));
		return;
	}

	if ((err = lstat(fname, &lsb)) == -1) {
		err = errno;
		tmpbuf = g_strdup_printf("%s", g_strerror(err));
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_fname]), tmpbuf);
		g_free(tmpbuf);
		return;
	}

#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode) && ((err = stat(fname, &sb)) == -1))
		sbp = NULL;
	sbp = &sb;
#endif

	/* skip leading "./" */
	if (fname[0] == '.' && fname[1] == '/')
		fname += 2;
#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		link = get_link_name(w, fname, fname);
		tmpbuf = g_strdup_printf("%s -> %s", fname, link);
		g_free(link);
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_fname]), tmpbuf);
		g_free(tmpbuf);
	} else {
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_fname]), fname);
	}
#else
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_fname]), fname);
#endif

#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		if (sbp)
			g_snprintf(buf, BUFLEN, "%lu -> %lu",
				   (gulong)lsb.st_size, (gulong)sb.st_size);
		else
			g_snprintf(buf, BUFLEN, "%lu -> ???",
				   (gulong)lsb.st_size);
	} else {
		g_snprintf(buf, BUFLEN, "%lu", (gulong)lsb.st_size);
	}
#else
	g_snprintf(buf, BUFLEN, "%lu", (gulong)lsb.st_size);
#endif
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_fsize]), buf);

	(void)memset(buf, 0, BUFLEN);
	filemodestring(&lsb, buf);
#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		if (sbp) {
			(void)memset(buf2, 0, BUFLEN);
			filemodestring(&sb, buf2);
			tmpbuf = g_strdup_printf("%s -> %s", buf, buf2);
		} else {
			tmpbuf = g_strdup_printf("%s -> ???", buf);
		}
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_perm]), tmpbuf);
		g_free(tmpbuf);
	} else {
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_perm]), buf);
	}
#else
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_perm]), buf);
#endif

#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		if (sbp)
			g_snprintf(buf, BUFLEN, "%lu -> %lu",
				   (gulong)lsb.st_ino, (gulong)sb.st_ino);
		else
			g_snprintf(buf, BUFLEN, "%lu -> ???",
				   (gulong)lsb.st_ino);
	} else {
		g_snprintf(buf, BUFLEN, "%lu", (gulong)lsb.st_ino);
	}
#else
	g_snprintf(buf, BUFLEN, "%lu", (gulong)lsb.st_ino);
#endif
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_inode]), buf);

#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		if (sbp)
			g_snprintf(buf, BUFLEN, "%lu -> %lu",
				   (gulong)lsb.st_nlink, (gulong)sb.st_nlink);
		else
			g_snprintf(buf, BUFLEN, "%lu -> ???",
				   (gulong)lsb.st_nlink);
	} else {
		g_snprintf(buf, BUFLEN, "%lu", (gulong)lsb.st_nlink);
	}
#else
	g_snprintf(buf, BUFLEN, "%lu", (gulong)lsb.st_nlink);
#endif
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_links]), buf);

	pwd = getpwuid(lsb.st_uid);
#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		if (sbp) {
			struct passwd *pwd2 = getpwuid(sb.st_uid);
			g_snprintf(buf, BUFLEN, "%s (%lu) -> %s (%lu)",
				   (pwd && pwd->pw_name) ?
				   		pwd->pw_name : "<UNKNOWN>",
				   (gulong)lsb.st_uid,
				   (pwd2 && pwd2->pw_name) ?
				   		pwd2->pw_name : "<UNKNOWN>",
				   (gulong)sb.st_uid);
		} else {
			g_snprintf(buf, BUFLEN, "%s (%lu) -> ??? (???)",
				   (pwd && pwd->pw_name) ?
				   		pwd->pw_name : "<UNKNOWN>",
				   (gulong)lsb.st_uid);
		}
	} else {
		g_snprintf(buf, BUFLEN, "%s (%lu)",
			   (pwd && pwd->pw_name) ? pwd->pw_name : "<UNKNOWN>",
			   (gulong)lsb.st_uid);
	}
#else
	g_snprintf(buf, BUFLEN, "%s (%lu)",
		   (pwd && pwd->pw_name) ? pwd->pw_name : "<UNKNOWN>",
		   (gulong)lsb.st_uid);
#endif
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_uid]), buf);

	grp = getgrgid(lsb.st_gid);
#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		if (sbp) {
			struct group *grp2 = getgrgid(sb.st_gid);
			g_snprintf(buf, BUFLEN, "%s (%lu) -> %s (%lu)",
				   (grp && grp->gr_name) ?
				   		grp->gr_name : "<UNKNOWN>",
				   (gulong)lsb.st_uid,
				   (grp2 && grp2->gr_name) ?
				   		grp2->gr_name : "<UNKNOWN>",
				   (gulong)sb.st_uid);
		} else {
			g_snprintf(buf, BUFLEN, "%s (%lu) -> ??? (???)",
				   (grp && grp->gr_name) ?
				   		grp->gr_name : "<UNKNOWN>",
				   (gulong)lsb.st_uid);
		}
	} else {
		g_snprintf(buf, BUFLEN, "%s (%lu)",
			   (grp && grp->gr_name) ? grp->gr_name : "<UNKNOWN>",
			   (gulong)lsb.st_gid);
	}
#else
	g_snprintf(buf, BUFLEN, "%s (%lu)",
		   (grp && grp->gr_name) ? grp->gr_name : "<UNKNOWN>",
		   (gulong)lsb.st_gid);
#endif
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_gid]), buf);

	timestr = ctime(&lsb.st_atime);
	if (timestr[strlen(timestr) - 1] == '\n')
		timestr[strlen(timestr) - 1] = '\0';
#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		if (sbp) {
			char *timestr2 = ctime(&sb.st_atime);
			if (timestr2[strlen(timestr2) - 1] == '\n')
				timestr2[strlen(timestr2) - 1] = '\0';
			tmpbuf = g_strdup_printf("%s -> %s", timestr, timestr2);
		} else {
			tmpbuf = g_strdup_printf("%s -> ???", timestr);
		}
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_atime]), tmpbuf);
		g_free(tmpbuf);
	} else {
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_atime]), timestr);
	}
#else
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_atime]), timestr);
#endif

	timestr = ctime(&lsb.st_mtime);
	if (timestr[strlen(timestr) - 1] == '\n')
		timestr[strlen(timestr) - 1] = '\0';
#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		if (sbp) {
			char *timestr2 = ctime(&sb.st_mtime);
			if (timestr2[strlen(timestr2) - 1] == '\n')
				timestr2[strlen(timestr2) - 1] = '\0';
			tmpbuf = g_strdup_printf("%s -> %s", timestr, timestr2);
		} else {
			tmpbuf = g_strdup_printf("%s -> ???", timestr);
		}
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_mtime]), tmpbuf);
		g_free(tmpbuf);
	} else {
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_mtime]), timestr);
	}
#else
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_mtime]), timestr);
#endif

	timestr = ctime(&lsb.st_ctime);
	if (timestr[strlen(timestr) - 1] == '\n')
		timestr[strlen(timestr) - 1] = '\0';
#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		if (sbp) {
			char *timestr2 = ctime(&sb.st_ctime);
			if (timestr2[strlen(timestr2) - 1] == '\n')
				timestr2[strlen(timestr2) - 1] = '\0';
			tmpbuf = g_strdup_printf("%s -> %s", timestr, timestr2);
		} else {
			tmpbuf = g_strdup_printf("%s -> ???", timestr);
		}
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_ctime]), tmpbuf);
		g_free(tmpbuf);
	} else {
		gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_ctime]), timestr);
	}
#else
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_ctime]), timestr);
#endif

#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		if (sbp)
			g_snprintf(buf, BUFLEN, "%lu -> %lu",
						   (gulong)lsb.st_blksize,
						   (gulong)sb.st_blksize);
		else
			g_snprintf(buf, BUFLEN, "%lu -> ???",
						   (gulong)lsb.st_blksize);
	} else {
		g_snprintf(buf, BUFLEN, "%lu", (gulong)lsb.st_blksize);
	}
#else
	g_snprintf(buf, BUFLEN, "%lu", (gulong)lsb.st_blksize);
#endif
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_bsize]), buf);

#ifdef S_ISLNK
	if (S_ISLNK(lsb.st_mode)) {
		if (sbp)
			g_snprintf(buf, BUFLEN, "%lu -> %lu",
						   (gulong)lsb.st_blocks,
						   (gulong)sb.st_blocks);
		else
			g_snprintf(buf, BUFLEN, "%lu -> ???",
						   (gulong)lsb.st_blocks);
	} else {
		g_snprintf(buf, BUFLEN, "%lu", (gulong)lsb.st_blocks);
	}
#else
	g_snprintf(buf, BUFLEN, "%lu", (gulong)lsb.st_blocks);
#endif
	gtk_label_set_text(GTK_LABEL(wgts[Ef_fidata_blocks]), buf);
} /* file_info_stat */


static void
file_copy_cancel_cb(GtkWidget *wgt, gpointer cbdata)
{
	file_copy_data_t *fcdp = (file_copy_data_t *)cbdata;

	gtk_widget_destroy(fcdp->dlg);
	g_free(fcdp);
} /* file_copy_cancel_cb */


static void
file_copy_ok_cb(GtkWidget *wgt, gpointer cbdata)
{
	char *dest, *cmd, *pref;
	file_copy_data_t *fcdp = (file_copy_data_t *)cbdata;

	gtk_widget_hide(fcdp->dlg);

	dest = gtk_entry_get_text(GTK_ENTRY(fcdp->entry));
	APPDBG_FILE(("file_copy_ok_cb: entry dest is '%s'\n", dest));
	if (!dest || strlen(dest) < 1)
		goto file_copy_ok_out;

	pref = prefs_string_get(Prefs_file_copy);
	if (!pref || strlen(pref) < 1)
		goto file_copy_ok_out;

	cmd = strrepl(pref, "$dest", dest);
	APPDBG_FILE(("file_copy_ok_cb: cmd after $dest repl '%s'\n", cmd));
	if (cmd) {
		cmd_exec(_("Copying selected files..."), fcdp->w, cmd,
			 TRUE, TRUE);
		g_free(cmd);
	}

file_copy_ok_out:
	file_copy_cancel_cb(NULL, fcdp);
} /* file_copy_ok_cb */


static void
cmd_archive_common(char *msg, win_t *w, char *cmd, gboolean repl_files,
		   gboolean report_err)
{
	char *finalcmd, *archive;

	archive = prefs_string_get(Prefs_archive_fname);
	if (!archive || strlen(archive) < 1) {
		cmd_err(_("No archive file specified"), 0);
		return;
	}

	finalcmd = strrepl(cmd, "$archive", archive);
	if (!finalcmd) {
		cmd_err(_("$archive not specified in command prefs"), 0);
		return;
	}

	cmd_exec(msg, w, finalcmd, repl_files, report_err);
	g_free(finalcmd);
} /* cmd_archive_common */


static void
file_list_ok(GtkWidget *wgt, gpointer cbdata)
{
	int width, height;
	GtkWidget *dlg = GTK_WIDGET(cbdata);

	gdk_window_get_size(dlg->window, &width, &height);
	prefs_int_set(Prefs_builtin_win_w, width);
	prefs_int_set(Prefs_builtin_win_h, height);
	gtk_widget_destroy(dlg);
} /* file_list_ok */


static void
file_list_read(win_t *w, char *fname)
{
	GtkWidget *dlg, *list, *tmp;
	GladeXML *xml;
	char *line;
	FILE *fp;

	if ((fp = file_open_fp(fname, "r")) == NULL)
		return;

	xml = my_glade_xml_get("List Dialog");
	g_assert(xml);
	tmp = my_glade_widget_get(xml, "list_dlg_apply", 0);
	gtk_widget_hide(tmp);
	dlg = my_glade_widget_get(xml, "List Dialog", 0);
	gnome_dialog_set_parent(GNOME_DIALOG(dlg),
				GTK_WINDOW(win_main_toplev(w)));
	gtk_window_set_title(GTK_WINDOW(dlg), _("Current Archive Contents"));
	my_prefs_window_set_size(dlg,
				 Prefs_save_builtin_win,
				 Prefs_builtin_win_w,
				 Prefs_builtin_win_h);

	tmp = my_glade_widget_get(xml, "list_dlg_ok", 0);
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked", file_list_ok, dlg);

	list = my_glade_widget_get(xml, "list_dlg_list", 0);
	gtk_object_destroy(GTK_OBJECT(xml));

	line = g_new(char, PATH_MAX * 2);
	while ((fgets(line, (PATH_MAX * 2) - 1, fp))) {
		line[strlen(line) - 1] = '\0';	/* drop trailing \n */
		tmp = gtk_list_item_new_with_label(line);
		gtk_container_add(GTK_CONTAINER(list), tmp);
		gtk_widget_show(tmp);
	}
	fclose(fp);
	g_free(line);

	gtk_widget_show(dlg);
} /* file_list_read */


/* the end */
