/* sqlquery.c
 *
 * Copyright (C) 1999 - 2001 Vivien Malerba
 * Copyright (C) 2001 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "sqlquery.h"
#include "sqlqueryenv.h"

static void sql_query_class_init (SqlQueryClass * class);
static void sql_query_init (SqlQuery * srv);
static void sql_query_destroy (GtkObject * object);
static void m_changed (SqlQuery * q, gpointer data);

/* 
 * Main static functions 
 */
static GtkObject *sql_query_new_id (gchar * name, gASQL_Main_Config * conf,
				    guint id, gboolean ref);
static void sql_query_table_dropped_cb (GtkObject * obj,
					SqlMemTable * table, SqlQuery * q);
static void sql_query_field_dropped_cb (GtkObject * obj,
					SqlMemTable * table,
					SqlMemField * field, SqlQuery * q);
static void sql_query_datatypes_updated_cb (SqlAccess * srv, SqlQuery * q);
static void sql_query_destroy (GtkObject * object);



/* get a pointer to the parents to be able to call their destructor */
static GtkObject *parent_class = NULL;

enum
{
	ELT_CREATED,
	ELT_DROPPED,
	ELT_MODIF,
	CHANGED,
	NAME_CHANGED,
	AUTO_LINK_ADDED,
	AUTO_LINK_REMOVED,
	WH_COND_CREATED,
	WH_COND_DROPPED,
	WH_COND_MODIFIED,
	WH_COND_MOVED,
	LAST_SIGNAL
};

static gint sql_query_signals[LAST_SIGNAL] =
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

/* Descriptive strings for enum JoinType */
static gchar *join_type_labels[LAST_JOIN_TYPE] =
{N_("Inner"), N_("Left Outer"), N_("Right Outer"), N_("Full Outer"), N_("Cross")}; 

static gchar *join_type_xml[LAST_JOIN_TYPE] =
{"INNER_JOIN", "LEFT_JOIN", "RIGHT_JOIN", "FULL_JOIN", "CROSS_JOIN"}; 

static gchar *join_type_sql[LAST_JOIN_TYPE] =
{N_("INNER JOIN"), N_("LEFT JOIN"), N_("RIGHT JOIN"), N_("FULL JOIN"), ", "}; 

static gchar *join_type_operator[LAST_JOIN_TYPE] =
{"=", "*=", "=*", "*=*", ""}; 

/* returns the string for a join type button label */
gchar *join_type_get_label(JoinType join_type)
{
  return _(join_type_labels[join_type]);
}

/* returns the string for the join type XML name */
gchar *join_type_get_xml(JoinType join_type)
{
  return join_type_xml[join_type];
}

/* returns the string for the join type SQL operator */
gchar *join_type_get_sql(JoinType join_type)
{
  return join_type_sql[join_type];
}

/* returns the join type from the xml label */
JoinType join_type_from_xml(gchar *jt_xml)
{
  JoinType jt;

  switch (jt_xml[0]){
  case 'I':
	  jt=JOIN_INNER;
          break;
  case 'L':
	  jt=JOIN_LEFT_OUTER;
          break;
  case 'R':
	  jt=JOIN_RIGHT_OUTER;
          break;
  case 'F':
	  jt=JOIN_FULL_OUTER;
          break;
  case 'C':
	  jt=JOIN_CROSS;
          break;
  default:
	  jt=JOIN_INNER; /* TODO FM: error handling? */
  }
  return jt;
}

/* returns the SQL operator that defines a join in the WHERE part */
gchar *join_type_get_operator(JoinType join_type)
{
  return join_type_operator[join_type];
}


guint
sql_query_get_type (void)
{
	static guint f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Query",
			sizeof (SqlQuery),
			sizeof (SqlQueryClass),
			(GtkClassInitFunc) sql_query_class_init,
			(GtkObjectInitFunc) sql_query_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_object_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_query_class_init (SqlQueryClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;

	sql_query_signals[ELT_CREATED] =
		gtk_signal_new ("elt_created",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryClass,
						   elt_created),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);
	sql_query_signals[ELT_DROPPED] =
		gtk_signal_new ("elt_dropped", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryClass,
						   elt_dropped),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);
	sql_query_signals[ELT_MODIF] =
		gtk_signal_new ("elt_modified", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryClass,
						   elt_modified),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);
	sql_query_signals[CHANGED] =
		gtk_signal_new ("changed", GTK_RUN_FIRST, object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryClass, changed),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);
	sql_query_signals[NAME_CHANGED] =
		gtk_signal_new ("name_changed", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryClass,
						   name_changed),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);
	sql_query_signals[AUTO_LINK_ADDED] =
		gtk_signal_new ("auto_link_added", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryClass,
						   auto_link_added),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);
	sql_query_signals[AUTO_LINK_REMOVED] =
		gtk_signal_new ("auto_link_removed", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryClass,
						   auto_link_removed),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);
	sql_query_signals[WH_COND_CREATED] =
		gtk_signal_new ("wh_cond_created", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryClass,
						   wh_cond_created),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);
	sql_query_signals[WH_COND_DROPPED] =
		gtk_signal_new ("wh_cond_dropped", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryClass,
						   wh_cond_dropped),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);
	sql_query_signals[WH_COND_MODIFIED] =
		gtk_signal_new ("wh_cond_modified", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryClass,
						   wh_cond_modified),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);
	sql_query_signals[WH_COND_MOVED] =
		gtk_signal_new ("wh_cond_moved", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryClass,
						   wh_cond_moved),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, sql_query_signals,
				      LAST_SIGNAL);
	class->elt_created =
		(void (*)(SqlQuery * q, QueryElement * elt)) m_changed;
	class->elt_dropped =
		(void (*)(SqlQuery * q, QueryElement * elt)) m_changed;
	class->elt_modified =
		(void (*)(SqlQuery * q, QueryElement * elt)) m_changed;
	class->auto_link_added = (void (*)(SqlQuery * q, QueryLink * ql))
		m_changed;
	class->auto_link_removed = (void (*)(SqlQuery * q, QueryLink * ql))
		m_changed;
	class->changed = NULL;
	class->name_changed = NULL;
	class->wh_cond_created =
		(void (*)(SqlQuery * q, QueryWhereNode * node)) m_changed;
	class->wh_cond_dropped =
		(void (*)(SqlQuery * q, QueryWhereNode * node)) m_changed;
	class->wh_cond_modified =
		(void (*)(SqlQuery * q, QueryWhereNode * node)) m_changed;
	class->wh_cond_moved =
		(void (*)(SqlQuery * q, QueryWhereNode * node)) m_changed;

	object_class->destroy = sql_query_destroy;
	parent_class = gtk_type_class (gtk_object_get_type ());
}

static void
sql_query_init (SqlQuery * q)
{
	q->name = NULL;
	q->descr = NULL;
	q->objects = NULL;
	q->group_by_list = NULL;
	q->order_by_list = NULL;
	q->where_tree = g_node_new (q);
	q->depend_list = NULL;
	q->text_only = FALSE;
	q->text_sql = NULL;
	q->envs = NULL;
}

GtkObject *
sql_query_new_noref (gchar * name, gASQL_Main_Config * conf)
{
	return sql_query_new_id (name, conf, 0, FALSE);
}

GtkObject *
sql_query_new (gchar * name, gASQL_Main_Config * conf)
{
	return sql_query_new_id (name, conf, get_id_serial (conf), TRUE);
}

static GtkObject *
sql_query_new_id (gchar * name, gASQL_Main_Config * conf,
		  guint id, gboolean ref)
{
	GtkObject *obj;

	SqlQuery *q;
	obj = gtk_type_new (sql_query_get_type ());
	q = SQL_QUERY (obj);

	q->conf = conf;
	q->id = id;		/* next value for the sequence */
	q->name = g_strdup (name);

	if (ref) {
		/* adding to the list of queries */
		conf->queries = g_slist_append (conf->queries, q);
		conf->save_up_to_date = FALSE;
	}

	/* connection to signals coming from SqlDb */
	gtk_signal_connect_while_alive (GTK_OBJECT (q->conf->db),
					"table_dropped",
					GTK_SIGNAL_FUNC
					(sql_query_table_dropped_cb), q, obj);
	gtk_signal_connect_while_alive (GTK_OBJECT (q->conf->db),
					"field_dropped",
					GTK_SIGNAL_FUNC
					(sql_query_field_dropped_cb), q, obj);
	gtk_signal_connect_while_alive (GTK_OBJECT (q->conf->db),
					"ff_link_created",
					GTK_SIGNAL_FUNC
					(sql_query_auto_links_add_link), q,
					obj);
	gtk_signal_connect_while_alive (GTK_OBJECT (q->conf->db),
					"ff_link_dropped",
					GTK_SIGNAL_FUNC
					(sql_query_auto_links_del_link), q,
					obj);
	gtk_signal_connect_while_alive (GTK_OBJECT (q->conf->srv),
					"data_types_updated",
					GTK_SIGNAL_FUNC
					(sql_query_datatypes_updated_cb), q,
					obj);
	return obj;
}

static void
copy_recursive_where_nodes (SqlQuery * oldq, SqlQuery * newq,
			    GNode * oldnode, GNode * newnode);

static GtkObject *sql_query_copy_common (SqlQuery * q, gboolean ref);
GtkObject *
sql_query_copy (SqlQuery * q)
{
	return sql_query_copy_common (q, TRUE);
}

GtkObject *
sql_query_copy_noref (SqlQuery * q)
{
	return sql_query_copy_common (q, FALSE);
}

static GtkObject *
sql_query_copy_common (SqlQuery * q, gboolean ref)
{
	SqlQuery *newquery;
	gchar *str;
	gint i = 1;
	gboolean found;
	GSList *list;

	g_return_val_if_fail (q, NULL);

	/* choose a name for the new query */
	if (ref)
		do {
			str = g_strdup_printf (_("%s: copy %d"), q->name, i);
			list = q->conf->queries;
			found = FALSE;
			while (list && !found) {
				if (!strcmp
				    (SQL_QUERY (list->data)->name, str)) {
					i++;
					found = TRUE;
					g_free (str);
				}
				list = g_slist_next (list);
			}
		} while (found);
	else
		str = g_strdup (q->name);

	if (ref)
		newquery = SQL_QUERY (sql_query_new (str, q->conf));
	else
		newquery = SQL_QUERY (sql_query_new_noref (str, q->conf));
	g_free (str);

	/* description copy */
	if (q->descr)
		newquery->descr = g_strdup (q->descr);

	/* query elements copy */
	list = q->objects;
	while (list) {
		QueryElement *qe, *nqe;

		qe = (QueryElement *) list->data;
		nqe = g_new0 (QueryElement, 1);
		nqe->type = qe->type;
		if (qe->type == QUERY_ELT_VALUE) {
			/* we need to copy the QueryElementValue */
			QueryElementValue *qev, *nqev;

			qev = (QueryElementValue *) qe->main;
			nqev = g_new0 (QueryElementValue, 1);
			nqev->type_val = qev->type_val;
			switch (nqev->type_val) {
			case QUERY_VALUE_STR:
				nqev->val.str = g_strdup (qev->val.str);
				break;
			}
			if (qev->default_val)
				nqev->default_val =
					g_strdup (qev->default_val);
			nqev->data_type = qev->data_type;
			nqev->ask_at_exec = qev->ask_at_exec;
			nqe->main = nqev;
		}
		else
			nqe->main = qe->main;
		if (qe->print_name)
			nqe->print_name = g_strdup (qe->print_name);
		if (qe->name)
			nqe->name = g_strdup (qe->name);
		newquery->objects = g_slist_append (newquery->objects, nqe);
		list = g_slist_next (list);
	}
	/* setting the args correctly: could not be done before */
	list = q->objects;
	i = 0;
	while (list) {
		GSList *args;
		QueryElement *qe, *nqe;

		qe = (QueryElement *) list->data;
		nqe = (QueryElement *) (g_slist_nth (newquery->objects, i)->
					data);
		args = qe->args;
		while (args) {
			nqe->args = g_slist_append (nqe->args,
						    g_slist_nth (newquery->
								 objects,
								 g_slist_index
								 (q->objects,
								  args->
								  data))->
						    data);
			args = g_slist_next (args);
		}
		list = g_slist_next (list);
		i++;
	}

	/* automatic links copy */
	list = q->depend_list;
	while (list) {
		QueryLink *ql, *nql;

		ql = (QueryLink *) list->data;
		nql = g_new0 (QueryLink, 1);
		nql->link = ql->link;
		/*nql->useit = ql->useit;*/
		nql->join_type = ql->join_type;
		newquery->depend_list =
			g_slist_append (newquery->depend_list, nql);
		list = g_slist_next (list);
	}

	/* WHERE tree copy */
	copy_recursive_where_nodes (q, newquery, q->where_tree,
				    newquery->where_tree);

	/* last remaining things */
	newquery->text_only = q->text_only;
	if (q->text_sql)
		newquery->text_sql = g_strdup (q->text_sql);

	return GTK_OBJECT (newquery);
}

/* for this recursive function, we assume it is called with 
   existing GNodes (and their data correctely set), 
   and it creates and update the GNodes and data of the new
   query for all the children of the GNodes 
*/
static void
copy_recursive_where_nodes (SqlQuery * oldq, SqlQuery * newq,
			    GNode * oldnode, GNode * newnode)
{
	GNode *onde, *nnde;
	QueryWhereNode *owh, *nwh;

	if (!oldnode || !newnode)
		return;

	onde = oldnode->children;
	while (onde) {
		/* data part */
		owh = (QueryWhereNode *) onde->data;
		nwh = g_new0 (QueryWhereNode, 1);
		nwh->is_node = owh->is_node;
		if (!nwh->is_node) {
			nwh->val.leaf.l_elt =
				g_slist_nth (newq->objects,
					     g_slist_index (oldq->objects,
							    owh->val.leaf.
							    l_elt))->data;
			nwh->val.leaf.op = owh->val.leaf.op;
			nwh->val.leaf.r_type = owh->val.leaf.r_type;
			switch (nwh->val.leaf.r_type) {
			case QUERY_WHERE_ELT:
				nwh->val.leaf.r_object =
					g_slist_nth (newq->objects,
						     g_slist_index (oldq->
								    objects,
								    owh->val.
								    leaf.
								    r_object))->
					data;
				break;
			case QUERY_WHERE_ASK:
				nwh->val.leaf.r_object = NULL;
				break;
			}
		}
		else
			nwh->val.logic = owh->val.logic;

		/* GNodes part */
		nnde = g_node_append_data (newnode, nwh);
		copy_recursive_where_nodes (oldq, newq, onde, nnde);	/* children */

		onde = onde->next;
	}
}

static void remove_elt (SqlQuery * q, QueryElement * object);
static void
sql_query_destroy (GtkObject * object)
{
	SqlQuery *q;
	GSList *list;
	guint sigid;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_QUERY (object));

	q = SQL_QUERY (object);

	/* list of dependencies */
	while (q->depend_list) {
		GSList *hold = q->depend_list;
		g_free (q->depend_list->data);
		q->depend_list = g_slist_remove_link (q->depend_list, q->depend_list);
		g_slist_free_1 (hold);
	}

	/* minor memory allocations */
	if (q->name)
		g_free (q->name);
	if (q->descr)
		g_free (q->descr);

	/* removing objects */
	while (q->objects) {
		remove_elt (q, (QueryElement *) (q->objects->data));
	}			/* q->objects is now NULL and
				   rem: other lists are cleared by the previous loop */

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
sql_query_table_dropped_cb (GtkObject * obj,
			    SqlMemTable * table, SqlQuery * q)
{
	GSList *list;
	QueryElement *qe;

	list = q->objects;
	while (list) {
		qe = (QueryElement *) (list->data);

		if (qe->type == QUERY_ELT_FIELD) {
			if (g_slist_find (table->fields, qe->main)) {
				remove_elt (q, qe);
				list = q->objects;
			}
			else
				list = g_slist_next (list);
		}
		else
			list = g_slist_next (list);
	}
}

static gboolean remove_elt_where_check_foreach (SqlQuery * q, GNode * node,
						QueryElement * object);
/* removes recursively everything referencing that object in all the 
   Query object' lists.
   Frees the memory associated to the object passed as argument */
static void
remove_elt (SqlQuery * q, QueryElement * object)
{
	GSList *list;
	QueryElement *qe;

	/* GROUP BY list */
	q->group_by_list = g_slist_remove (q->group_by_list, object);

	/* ORDER BY list */
	q->order_by_list = g_slist_remove (q->order_by_list, object);

	/* WHERE tree */
	remove_elt_where_check_foreach (q, NULL, object);

	/* DEPEND list */
	q->depend_list = g_slist_remove (q->depend_list, object);

	/* objects list */
	/* the top level list of elements */
	q->objects = g_slist_remove (q->objects, object);

	/* elements referencing that object */
	list = q->objects;
	while (list) {
		qe = (QueryElement *) (list->data);
		if ((qe->type != QUERY_ELT_FIELD)
		    && (qe->type != QUERY_ELT_VALUE)) {
			if (g_slist_find (qe->args, object)) {
				/* del the element which refers to object */
				remove_elt (q, list->data);
				list = q->objects;
			}
			else
				list = g_slist_next (list);
		}
		else
			list = g_slist_next (list);
	}
#ifdef debug_signal
	g_print (">> 'ELT_DROPPED' from sqlquery->remove_elt\n");
#endif
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[ELT_DROPPED],
			 object);
#ifdef debug_signal
	g_print ("<< 'ELT_DROPPED' from sqlquery->remove_elt\n");
#endif

	/* freeing the memory associated to the object */
	switch (object->type) {
	case QUERY_ELT_VALUE:
		{
			QueryElementValue *val;
			val = (QueryElementValue *) (object->main);
			if ((val->type_val == QUERY_VALUE_STR)
			    && (val->val.str))
				g_free (val->val.str);
			if (val->default_val)
				g_free (val->default_val);
			g_free (val);
		}
	case QUERY_ELT_AGGREGATE:
	case QUERY_ELT_FUNCTION:
	default:
		if (object->name)
			g_free (object->name);
		if (object->print_name)
			g_free (object->print_name);
		if (object->args)
			g_slist_free (object->args);
		break;
	}
	g_free (object);
}

/* returns TRUE if there HAS been some modifications in the tree */
static gboolean
remove_elt_where_check_foreach (SqlQuery * q, GNode * node,
				QueryElement * object)
{
	GNode *gnode;
	QueryWhereNode *wh;

	if (node)
		gnode = node;
	else
		gnode = q->where_tree->children;

	while (gnode) {
		wh = (QueryWhereNode *) gnode->data;

		if ((!wh->is_node) && (wh->val.leaf.l_elt == object)) {
			/* simply remove the condition */
			sql_query_cond_del (q, wh);
			remove_elt_where_check_foreach (q, NULL, object);
			return TRUE;
		}

		if ((!wh->is_node) &&
		    (wh->val.leaf.r_type == QUERY_WHERE_ELT) &&
		    ((QueryElement *) wh->val.leaf.r_object == object)) {
			/* change the type of the right operand */
			sql_query_cond_modif_leaf (q, wh, wh->val.leaf.l_elt,
						   wh->val.leaf.op,
						   QUERY_WHERE_ASK, NULL);
		}

		/* the children last this time ! */
		if (gnode->children) {
			if (remove_elt_where_check_foreach
			    (q, gnode->children, object)) {
				return TRUE;
			}
		}
		gnode = gnode->next;
	}

	return FALSE;
}

static void
sql_query_field_dropped_cb (GtkObject * obj,
			    SqlMemTable * table,
			    SqlMemField * field, SqlQuery * q)
{
	GSList *list;
	QueryElement *qe;

	list = q->objects;
	while (list) {
		qe = (QueryElement *) (list->data);

		if (qe->type == QUERY_ELT_FIELD)
			if (SQL_MEM_FIELD (qe->main) == field) {
				remove_elt (q, qe);
				list = q->objects;
			}
			else
				list = g_slist_next (list);
		else
			list = g_slist_next (list);
	}
}

static void
sql_query_datatypes_updated_cb (SqlAccess * srv, SqlQuery * q)
{
	GSList *list;
	QueryElement *qe;

	list = q->objects;
	while (list) {
		qe = (QueryElement *) (list->data);

		if (qe->type == QUERY_ELT_VALUE)
			if (!g_slist_find (q->conf->srv->data_types,
					   ((QueryElementValue *) (qe->
								   main))->
					   data_type)) {
				remove_elt (q, qe);
				list = q->objects;
			}
			else
				list = g_slist_next (list);
		else
			list = g_slist_next (list);
	}
}

static void
m_changed (SqlQuery * q, gpointer data)
{
#ifdef debug_signal
	g_print (">> 'CHANGED' from sqlquery->m_changed\n");
#endif
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[CHANGED]);
#ifdef debug_signal
	g_print ("<< 'CHANGED' from sqlquery->m_changed\n");
#endif
}



/*
 * Helper functions
 */

void
sql_query_set_name (SqlQuery * q, gchar * name, gchar * descr)
{
	gboolean changed = FALSE;

	if (name) {
		if (q->name) {
			changed = strcmp (q->name, name) ? TRUE : FALSE;
			g_free (q->name);
		}
		else
			changed = TRUE;
		q->name = g_strdup (name);
	}

	if (descr) {
		if (q->descr) {
			changed = strcmp (q->descr, descr) ? TRUE : changed;
			g_free (q->descr);
		}
		else
			changed = TRUE;
		q->descr = g_strdup (descr);
	}

	if (changed) {
#ifdef debug_signal
		g_print (">> 'NAME_CHANGED' from sql_query_set_name\n");
#endif
		gtk_signal_emit (GTK_OBJECT (q),
				 sql_query_signals[NAME_CHANGED]);
#ifdef debug_signal
		g_print ("<< 'NAME_CHANGED' from sql_query_set_name\n");
#endif
	}
}

void
sql_query_set_text_sql (SqlQuery * q, gchar * sql)
{
	if (q->text_sql) {
		g_free (q->text_sql);
		q->text_sql = NULL;
		q->text_only = FALSE;
	}

	if (sql) {
		q->text_sql = g_strdup (sql);
		q->text_only = TRUE;
	}

#ifdef debug_signal
	g_print (">> 'CHANGED' from sql_query_set_text_sql\n");
#endif
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[CHANGED]);
#ifdef debug_signal
	g_print ("<< 'CHANGED' from sql_query_set_text_sql\n");
#endif
}

gchar *
sql_query_render_operator (QueryWhereOpType op)
{
	gchar *str;

	switch (op) {
	case QUERY_OP_EQUAL:
		str = "=";
		break;
	case QUERY_OP_DIFF:
		str = "!=";
		break;
	case QUERY_OP_SUP:
		str = ">";
		break;
	case QUERY_OP_SUPEQUAL:
		str = ">=";
		break;
	case QUERY_OP_INF:
		str = "<";
		break;
	case QUERY_OP_INFEQUAL:
		str = "<=";
		break;
	case QUERY_OP_LIKE:
		str = "LIKE";
		break;
	case QUERY_OP_REGEX:
		str = "MATCH";
		break;
	case QUERY_OP_IN:
		str = "IN";
		break;
	default:
		str = "???";
	}
	return str;
}

/*
 * Loading/saving as XML
 */

static void build_where_xml_recursive (SqlQuery * q, GNode * node,
				       xmlNodePtr tree);
void
sql_query_build_xml_tree (SqlQuery * q,
			  xmlNodePtr maintree, gASQL_Main_Config * conf)
{
	xmlNodePtr toptree, tree, subtree;
	gchar *str;
	GSList *list, *arglist;
	QueryElement *qe;
	QueryLink *ql;
	gint order;

	toptree = xmlNewChild (maintree, NULL, "query", NULL);
	str = g_strdup_printf ("QU%d", q->id);
	xmlSetProp (toptree, "id", str);
	g_free (str);
	xmlSetProp (toptree, "name", q->name);
	xmlSetProp (toptree, "descr", q->descr);
	if (q->text_only)
		xmlSetProp (toptree, "text_query", "t");

	/* list of QueryElementValue objects, stored in their own element */
	list = q->objects;
	while (list) {
		qe = (QueryElement *) (list->data);
		if (qe->type == QUERY_ELT_VALUE) {
			tree = xmlNewChild (toptree, NULL, "QueryEltValue",
					    NULL);
			str = g_strdup_printf ("QU%d:QV%d", q->id,
					       g_slist_position (q->objects,
								 list));
			xmlSetProp (tree, "id", str);
			g_free (str);
			str = g_strdup_printf ("DT%s",
					       (((QueryElementValue *) (qe->
									main))->
						data_type)->sqlname);
			xmlSetProp (tree, "type", str);
			g_free (str);
			if (((QueryElementValue *) (qe->main))->default_val)
				xmlSetProp (tree, "default",
					    ((QueryElementValue *) (qe->
								    main))->
					    default_val);
			if (((QueryElementValue *) (qe->main))->ask_at_exec)
				xmlSetProp (tree, "exec_modif", "t");
			else
				xmlSetProp (tree, "exec_modif", "f");
		}
		list = g_slist_next (list);
	}

	/* list of objects */
	list = q->objects;
	order = 0;
	while (list) {
		qe = (QueryElement *) (list->data);
		tree = xmlNewChild (toptree, NULL, "QueryElt", NULL);
		str = g_strdup_printf ("QU%d:QE%d", q->id, order++);
		xmlSetProp (tree, "id", str);
		g_free (str);

		xmlSetProp (tree, "name", qe->name);
		switch (qe->type) {
		case QUERY_ELT_FIELD:
			str = g_strdup_printf ("TV%s:FI%s",
					       sql_db_find_table_from_field
					       (q->conf->db,
						SQL_MEM_FIELD (qe->main))->
					       name,
					       SQL_MEM_FIELD (qe->main)->
					       name);
			break;
		case QUERY_ELT_AGGREGATE:
			str = g_strdup_printf ("AG%s",
					       SQL_DATA_AGGREGATE (qe->main)->
					       objectid);
			break;
		case QUERY_ELT_FUNCTION:
			str = g_strdup_printf ("PR%s",
					       SQL_DATA_FUNCTION (qe->main)->
					       objectid);
			break;
		case QUERY_ELT_VALUE:
			str = g_strdup_printf ("QU%d:QV%d", q->id,
					       g_slist_index (q->objects,
							      qe));
			break;
		}
		xmlSetProp (tree, "object", str);
		g_free (str);

		if (qe->print_name)
			xmlSetProp (tree, "print", qe->print_name);
		/* arguments... */
		arglist = qe->args;
		while (arglist) {
			qe = (QueryElement *) (arglist->data);
			subtree =
				xmlNewChild (tree, NULL, "QueryEltArg", NULL);
			str = g_strdup_printf ("QU%d:QE%d", q->id,
					       g_slist_index (q->objects,
							      qe));
			xmlSetProp (subtree, "ref", str);
			g_free (str);
			arglist = g_slist_next (arglist);
		}
		list = g_slist_next (list);
	}

	/* list of automatic dependencies */
	list = q->depend_list;
	while (list) {
		ql = (QueryLink *) (list->data);
		tree = xmlNewChild (toptree, NULL, "depend", NULL);
		str = g_strdup_printf ("LI%d",
				       g_slist_index (conf->db->field_links,
						      ql->link));
		xmlSetProp (tree, "link", str);
		g_free (str);
		/*xmlSetProp (tree, "useit", (ql->useit) ? "t" : "f");*/
		xmlSetProp(tree, "join_type", join_type_get_xml(ql->join_type));
		list = g_slist_next (list);
	}

	/* tree of where clauses */
	tree = xmlNewChild (toptree, NULL, "QueryWherePart", NULL);
	build_where_xml_recursive (q, q->where_tree, tree);


	/* The text of the query if specified */
	if (q->text_only && q->text_sql)
		tree = xmlNewChild (toptree, NULL, "QueryText", q->text_sql);

	/* the Query environments */
	list = q->envs;
	while (list) {
		SqlQueryEnv *env;
		env = SQL_QUERY_ENV (list->data);

		tree = xmlNewChild (toptree, NULL, "QueryEnv", NULL);
		str = g_strdup_printf ("QU%d:QF%d", q->id,
				       g_slist_position (q->envs, list));
		xmlSetProp (tree, "id", str);
		g_free (str);
		if (env->modif_table) {
			str = g_strdup_printf ("TV%s",
					       env->modif_table->name);
			xmlSetProp (tree, "modif_table", str);
			g_free (str);
		}

		if (env->name)
			xmlSetProp (tree, "name", env->name);

		if (env->descr)
			xmlSetProp (tree, "descr", env->descr);

		str = g_strdup_printf ("%ld", env->actions);
		xmlSetProp (tree, "actions", str);
		g_free (str);

		if (env->form_is_default)
			xmlSetProp (tree, "form_default", "t");
		else
			xmlSetProp (tree, "form_default", "f");

		/* FIXME: if we have set up a form from a glade dialog, then
		   save it! */

		list = g_slist_next (list);
	}
}

static void
build_where_xml_recursive (SqlQuery * q, GNode * node, xmlNodePtr tree)
{
	GNode *gnode;
	xmlNodePtr ntree;
	QueryWhereNode *wh;
	gchar *str;

	gnode = node->children;

	while (gnode) {
		wh = (QueryWhereNode *) gnode->data;
		if (wh->is_node) {
			ntree = xmlNewChild (tree, NULL, "QueryWhereNode",
					     NULL);
			switch (wh->val.logic) {
			case QUERY_WHERE_AND:
				str = "AND";
				break;
			case QUERY_WHERE_OR:
				str = "OR";
				break;
			case QUERY_WHERE_NOT:
				str = "NOT";
				break;
			}
			xmlSetProp (ntree, "type", str);
		}
		else {
			ntree = xmlNewChild (tree, NULL, "QueryWhere", NULL);
			str = g_strdup_printf ("QU%d:QE%d", q->id,
					       g_slist_index (q->objects,
							      wh->val.leaf.
							      l_elt));
			xmlSetProp (ntree, "loperand", str);
			g_free (str);
			switch (wh->val.leaf.op) {
			case QUERY_OP_EQUAL:
				str = "EQUAL";
				break;
			case QUERY_OP_DIFF:
				str = "DIFF";
				break;
			case QUERY_OP_SUP:
				str = "SUP";
				break;
			case QUERY_OP_SUPEQUAL:
				str = "SUPEQUAL";
				break;
			case QUERY_OP_INF:
				str = "INF";
				break;
			case QUERY_OP_INFEQUAL:
				str = "INFEQUAL";
				break;
			case QUERY_OP_LIKE:
				str = "LIKE";
				break;
			case QUERY_OP_REGEX:
				str = "MATCH";
				break;
			case QUERY_OP_IN:
				str = "IN";
				break;
			default:
				str = "???";
			}
			xmlSetProp (ntree, "op", str);

			switch (wh->val.leaf.r_type) {
			case QUERY_WHERE_ELT:
				str = g_strdup_printf ("QU%d:QE%d", q->id,
						       g_slist_index (q->
								      objects,
								      (QueryElement
								       *) wh->
								      val.
								      leaf.
								      r_object));
				xmlSetProp (ntree, "roperand", str);
				g_free (str);
				break;
			case QUERY_WHERE_ASK:
				/* nothing to do */
				break;
			}
		}

		build_where_xml_recursive (q, gnode, ntree);
		gnode = gnode->next;
	}
}

static SqlQuery *sql_query_build_from_xml_tree (gASQL_Main_Config * conf,
						xmlNodePtr node);
static QueryElement *find_query_elt (SqlQuery * q, gchar * xmlid);
gboolean
sql_queries_build_from_xml_tree (gASQL_Main_Config * conf, xmlNodePtr node)
{
	xmlNodePtr subtree;

	subtree = node->xmlChildrenNode;
	while (subtree) {
		sql_query_build_from_xml_tree (conf, subtree);
		subtree = subtree->next;
	}
}

static QueryElement *
find_query_elt (SqlQuery * q, gchar * xmlid)
{
	gint i;
	gchar *ptr;
	ptr = strstr (xmlid, ":QE");
	i = atoi (ptr + 3);
	return (QueryElement *) (g_slist_nth_data (q->objects, i));
}

static void build_recursive_where_tree_from_xml (SqlQuery * q, GNode * node,
						 xmlNodePtr tree);
static SqlQuery *
sql_query_build_from_xml_tree (gASQL_Main_Config * conf, xmlNodePtr node)
{
	xmlNodePtr tree;
	gchar *str;
	guint id;
	gboolean ok = TRUE;
	SqlQuery *q;
	SqlMemField *field;
	GSList *list, *elts_ref = NULL;
	QueryElement *qe;
	QueryLink *ql;

	str = xmlGetProp (node, "id");
	id = atoi (str + 2);
	g_free (str);

	str = xmlGetProp (node, "name");
	q = SQL_QUERY (sql_query_new_id (str, conf, id, TRUE));
	g_free (str);

	str = xmlGetProp (node, "descr");
	if (str) {
		sql_query_set_name (q, NULL, str);
		g_free (str);
	}

	str = xmlGetProp (node, "text_query");
	q->text_only = FALSE;
	if (str && (*str == 't')) {
		q->text_only = TRUE;
		g_free (str);
	}

	/* First pass through all the query elements, just to make sure they are
	   in memory when, in the second pass, the links between them are set up */
	tree = node->xmlChildrenNode;
	while (tree) {
		if (!strcmp (tree->name, "QueryElt")) {
			gboolean found = FALSE;

			qe = g_new0 (QueryElement, 1);
			str = xmlGetProp (tree, "name");
			qe->name = str;

			str = xmlGetProp (tree, "print");
			qe->print_name = str;
			q->objects = g_slist_append (q->objects, qe);

			str = xmlGetProp (tree, "object");
			if ((*str == 'T') && (*(str + 1) == 'V')) {
				SqlMemField *field;
				field = sql_db_find_field_from_xml_name
					(conf->db, str);
				qe->type = QUERY_ELT_FIELD;
				qe->main = (gpointer) field;
				found = TRUE;
			}
			if ((!found) && (*str == 'A') && (*(str + 1) == 'G')) {
				SqlDataAggregate *agg;
				agg = sql_data_aggregate_get_from_xml_id
					(conf->srv, str);
				qe->type = QUERY_ELT_AGGREGATE;
				qe->main = (gpointer) agg;
				found = TRUE;
			}
			if ((!found) && (*str == 'P') && (*(str + 1) == 'R')) {
				SqlDataFunction *func;
				func = sql_data_function_get_from_xml_id
					(conf->srv, str);
				qe->type = QUERY_ELT_FUNCTION;
				qe->main = (gpointer) func;
				found = TRUE;
			}
			if ((!found) && (*str == 'Q') && (*(str + 1) == 'U')) {
				QueryElementValue *eltval;
				xmlNodePtr valtree;
				gchar *str2;
				SqlDataType *type;

				eltval = g_new0 (QueryElementValue, 1);
				valtree = xmlGetID (tree->doc, str)->node;
				str2 = xmlGetProp (valtree, "type");
				type = sql_data_type_get_from_xml_id (conf->
								      srv,
								      str2);
				g_free (str2);
				eltval->data_type = type;
				str2 = xmlGetProp (valtree, "default");
				if (str2)
					eltval->default_val = str2;
				else
					eltval->default_val = NULL;
				str2 = xmlGetProp (valtree, "exec_modif");
				if (str2 && (*str2 == 't'))
					eltval->ask_at_exec = TRUE;
				else
					eltval->ask_at_exec = FALSE;
				if (str2)
					g_free (str2);

				qe->type = QUERY_ELT_VALUE;
				qe->main = (gpointer) eltval;
				found = TRUE;
			}
			g_free (str);
		}

		if (!strcmp (tree->name, "QueryText")) {
			str = xmlNodeGetContent (tree);
			q->text_sql = str;
		}

		tree = tree->next;
	}

	/* Second pass through all the query elements and dependencies to make the
	   links between Query elements */
	tree = node->xmlChildrenNode;

	while (tree) {
		if (!strcmp (tree->name, "QueryElt")) {
			xmlNodePtr subtree;
			QueryElement *argqe;

			str = xmlGetProp (tree, "id");
			qe = find_query_elt (q, str);
			g_free (str);
			subtree = tree->xmlChildrenNode;
			while (subtree) {	/* QueryEltArg nodes */
				str = xmlGetProp (subtree, "ref");
				argqe = find_query_elt (q, str);
				g_free (str);
				qe->args = g_slist_append (qe->args, argqe);
				subtree = subtree->next;
			}
		}

		if (!strcmp (tree->name, "depend")) {
			str = xmlGetProp (tree, "link");
			ql = g_new0 (QueryLink, 1);
			ql->link =
				sql_db_find_flink_from_xml_name (conf->db,
								 str);
			g_free (str);

			/*
			str = xmlGetProp (tree, "useit");
			if (*str == 't')
				ql->useit = TRUE;
			else
				ql->useit = FALSE;
			*/
			str = xmlGetProp(tree, "join_type");
			ql->join_type = join_type_from_xml(str);
			
			g_free (str);

			q->depend_list = g_slist_append (q->depend_list, ql);
		}

		if (!strcmp (tree->name, "QueryWherePart"))
			build_recursive_where_tree_from_xml (q, q->where_tree,
							     tree->
							     xmlChildrenNode);

		if (!strcmp (tree->name, "QueryEnv")) {
			SqlQueryEnv *env;

			env = SQL_QUERY_ENV (sql_query_env_new (q));
			str = xmlGetProp (tree, "modif_table");
			if (str) {
				SqlMemTable *table;
				table = sql_db_find_table_by_xml_name (q->
								       conf->
								       db,
								       str);
				sql_query_env_set_modif_table (env, table);
				g_free (str);
			}

			str = xmlGetProp (tree, "name");
			if (str) {
				sql_query_env_set_name (env, str);
				g_free (str);
			}

			str = xmlGetProp (tree, "descr");
			if (str) {
				sql_query_env_set_descr (env, str);
				g_free (str);
			}

			str = xmlGetProp (tree, "actions");
			if (str) {
				env->actions = atol (str);
				g_free (str);
			}

			str = xmlGetProp (tree, "form_default");
			if (str) {
				if (*str == 't')
					env->form_is_default = TRUE;
				else
					env->form_is_default = FALSE;
				g_free (str);
			}

		}

		tree = tree->next;
	}


	if (!ok) {
		conf->queries = g_slist_remove (conf->queries, q);
		gtk_object_destroy (GTK_OBJECT (q));
		q = NULL;
	}
	return q;
}

static void
build_recursive_where_tree_from_xml (SqlQuery * q, GNode * node,
				     xmlNodePtr tree)
{
	GNode *nnode;
	QueryWhereNode *wh;
	gchar *str;
	xmlNodePtr treelist;

	treelist = tree;
	while (treelist) {
		wh = g_new0 (QueryWhereNode, 1);
		if (!strcmp (treelist->name, "QueryWhereNode")) {
			wh->is_node = TRUE;
			str = xmlGetProp (treelist, "type");
			if (!strcmp (str, "AND"))
				wh->val.logic = QUERY_WHERE_AND;
			if (!strcmp (str, "OR"))
				wh->val.logic = QUERY_WHERE_OR;
			if (!strcmp (str, "NOT"))
				wh->val.logic = QUERY_WHERE_NOT;
			g_free (str);
		}
		else {		/* we have a QueryWhere */
			QueryElement *qe;
			wh->is_node = FALSE;
			str = xmlGetProp (treelist, "loperand");
			qe = find_query_elt (q, str);
			wh->val.leaf.l_elt = qe;
			g_free (str);
			str = xmlGetProp (treelist, "op");
			if (!strcmp (str, "EQUAL"))
				wh->val.leaf.op = QUERY_OP_EQUAL;
			if (!strcmp (str, "DIFF"))
				wh->val.leaf.op = QUERY_OP_DIFF;
			if (!strcmp (str, "SUP"))
				wh->val.leaf.op = QUERY_OP_SUP;
			if (!strcmp (str, "SUPEQUAL"))
				wh->val.leaf.op = QUERY_OP_SUPEQUAL;
			if (!strcmp (str, "INF"))
				wh->val.leaf.op = QUERY_OP_INF;
			if (!strcmp (str, "INFEQUAL"))
				wh->val.leaf.op = QUERY_OP_INFEQUAL;
			if (!strcmp (str, "LIKE"))
				wh->val.leaf.op = QUERY_OP_LIKE;
			if (!strcmp (str, "REGEX"))
				wh->val.leaf.op = QUERY_OP_REGEX;
			if (!strcmp (str, "IN"))
				wh->val.leaf.op = QUERY_OP_IN;
			g_free (str);
			str = xmlGetProp (treelist, "roperand");
			if (str) {
				qe = find_query_elt (q, str);
				wh->val.leaf.r_type = QUERY_WHERE_ELT;
				wh->val.leaf.r_object = qe;
				g_free (str);
			}
			else {
				wh->val.leaf.r_type = QUERY_WHERE_ASK;
				wh->val.leaf.r_object = NULL;
			}
		}

		nnode = g_node_new (wh);
		g_node_insert_before (node, NULL, nnode);
		if (treelist->xmlChildrenNode)
			build_recursive_where_tree_from_xml (q, nnode,
							     treelist->
							     xmlChildrenNode);

		treelist = treelist->next;
	}
}

/* 
 * Query contents manipulation
 */
QueryElement *
sql_query_new_elt_field (SqlQuery * q,
			 gchar * name,
			 gchar * print_name, SqlMemField * field)
{
	QueryElement *elt;

	elt = (QueryElement *) g_malloc (sizeof (QueryElement));
	elt->type = QUERY_ELT_FIELD;
	elt->main = field;
	elt->args = NULL;
	elt->name = g_strdup (name);
	if (*print_name != '\0')
		elt->print_name = g_strdup (print_name);
	else
		elt->print_name = NULL;

	q->objects = g_slist_append (q->objects, elt);
#ifdef debug_signal
	g_print (">> 'ELT_CREATED' from sql_query_new_elt_field\n");
#endif
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[ELT_CREATED], elt);
#ifdef debug_signal
	g_print ("<< 'ELT_CREATED' from sql_query_new_elt_field\n");
#endif
	return elt;
}

QueryElement *
sql_query_new_elt_aggregate (SqlQuery * q,
			     gchar * name,
			     gchar * print_name,
			     SqlDataAggregate * agg, GSList * args)
{
	QueryElement *elt;

	elt = (QueryElement *) g_malloc (sizeof (QueryElement));
	elt->type = QUERY_ELT_AGGREGATE;
	elt->main = agg;
	elt->args = g_slist_copy (args);
	elt->name = g_strdup (name);
	if (*print_name != '\0')
		elt->print_name = g_strdup (print_name);
	else
		elt->print_name = NULL;

	q->objects = g_slist_append (q->objects, elt);
#ifdef debug_signal
	g_print (">> 'ELT_CREATED' from sql_query_new_elt_aggregate\n");
#endif
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[ELT_CREATED], elt);
#ifdef debug_signal
	g_print ("<< 'ELT_CREATED' from sql_query_new_elt_aggregate\n");
#endif
	return elt;
}

QueryElement *
sql_query_new_elt_function (SqlQuery * q,
			    gchar * name,
			    gchar * print_name,
			    SqlDataFunction * func, GSList * args)
{
	QueryElement *elt;

	elt = (QueryElement *) g_malloc (sizeof (QueryElement));
	elt->type = QUERY_ELT_FUNCTION;
	elt->main = func;
	elt->args = g_slist_copy (args);
	elt->name = g_strdup (name);
	if (*print_name != '\0')
		elt->print_name = g_strdup (print_name);
	else
		elt->print_name = NULL;

	q->objects = g_slist_append (q->objects, elt);
#ifdef debug_signal
	g_print (">> 'ELT_CREATED' from sql_query_new_elt_function\n");
#endif
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[ELT_CREATED], elt);
#ifdef debug_signal
	g_print ("<< 'ELT_CREATED' from sql_query_new_elt_function\n");
#endif
	if (print_name) {
#ifdef debug_signal
		g_print (">> 'CHANGED' from sql_query_new_elt_function\n");
#endif
		gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[CHANGED]);
#ifdef debug_signal
		g_print ("<< 'CHANGED' from sql_query_new_elt_function\n");
#endif
	}
	return elt;
}

QueryElement *
sql_query_new_elt_value (SqlQuery * q,
			 gchar * name,
			 gchar * print_name,
			 SqlDataType * data_type,
			 gchar * defval, gboolean modif)
{
	QueryElement *elt;
	QueryElementValue *val;

	val = g_new0 (QueryElementValue, 1);
	val->data_type = data_type;
	val->default_val = g_strdup (defval);
	val->ask_at_exec = modif;

	elt = g_new0 (QueryElement, 1);
	elt->type = QUERY_ELT_VALUE;
	elt->main = val;
	elt->args = NULL;
	elt->name = g_strdup (name);
	if (*print_name != '\0')
		elt->print_name = g_strdup (print_name);
	else
		elt->print_name = NULL;

	q->objects = g_slist_append (q->objects, elt);
#ifdef debug_signal
	g_print (">> 'ELT_CREATED' from sql_query_new_elt_value\n");
#endif
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[ELT_CREATED], elt);
#ifdef debug_signal
	g_print ("<< 'ELT_CREATED' from sql_query_new_elt_value\n");
#endif
	return elt;
}

QueryElement *
sql_query_get_element (SqlQuery * q, gchar * name)
{
	QueryElement *qe = NULL;
	GSList *list;

	list = q->objects;
	while (list && !qe) {
		if (!strcmp (((QueryElement *) (list->data))->name, name))
			qe = (QueryElement *) (list->data);
		else
			list = g_slist_next (list);
	}

	return qe;
}

void
sql_query_del_elt (SqlQuery * q, QueryElement * elt)
{
	remove_elt (q, elt);
}

void
sql_query_swap_objects (SqlQuery * q, QueryElement *o1, QueryElement *o2)
{
	GSList *list1, *list2;
	gpointer data;

	list1 = g_slist_find (q->objects, o1);
	list2 = g_slist_find (q->objects, o2);

	if (list1 && list2) {
		/* swapping list1->data and list2->data */
		data = list1->data;
		list1->data = list2->data;
		list2->data = data;
	}
#ifdef debug_signal
	g_print (">> 'CHANGED' from sql_query_swap_objects\n");
#endif
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[CHANGED]);
#ifdef debug_signal
	g_print ("<< 'CHANGED' from sql_query_swap_objects\n");
#endif
}

void
sql_query_modif_elt (SqlQuery * q, QueryElement * elt,
		     gchar * name, gchar * print_name)
{
	if (name) {
		if (elt->name)
			g_free (elt->name);
		elt->name = g_strdup (name);
	}
	if (print_name) {
		if (elt->print_name)
			g_free (elt->print_name);
		if (*print_name == '\0')
			elt->print_name = NULL;
		else
			elt->print_name = g_strdup (print_name);
	}

	if (name || print_name) {
#ifdef debug_signal
		g_print (">> 'ELT_MODIF' from sql_query_modif_elt\n");
#endif
		gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[ELT_MODIF],
				 elt);
#ifdef debug_signal
		g_print ("<< 'ELT_MODIF' from sql_query_modif_elt\n");
#endif
		if (print_name) {
#ifdef debug_signal
			g_print (">> 'CHANGED' from sql_query_modif_elt\n");
#endif
			gtk_signal_emit (GTK_OBJECT (q),
					 sql_query_signals[CHANGED]);
#ifdef debug_signal
			g_print ("<< 'CHANGED' from sql_query_modif_elt\n");
#endif
		}
	}
}



/*
 * Query links usage management
 */
static void refresh_link (SqlQuery * q, QueryLink * ql);
static gboolean table_is_in_query (SqlQuery * q, SqlMemTable * table);
void
sql_query_auto_links_refresh (SqlQuery * q)
{
	GSList *list;

	list = q->depend_list;
	while (list) {
		refresh_link (q, (QueryLink *) (list->data));
		list = g_slist_next (list);
	}
}

/* TODO FM: revise; check who is calling and join_type setting */
static void
refresh_link (SqlQuery * q, QueryLink * ql)
{
	SqlMemTable *fromt, *tot;

	fromt = sql_db_find_table_from_field (q->conf->db, ql->link->from);
	tot = sql_db_find_table_from_field (q->conf->db, ql->link->to);
	if (fromt && tot) {
		if (table_is_in_query (q, fromt)
		    && table_is_in_query (q, tot))
			/*ql->useit = TRUE;*/
			ql->join_type = JOIN_INNER;
		else
			/*ql->useit = FALSE;*/
			ql->join_type = JOIN_CROSS;
	}
#ifdef debug_signal
	g_print (">> 'CHANGED' from sqldb->refresh_link\n");
#endif
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[CHANGED]);
#ifdef debug_signal
	g_print ("<< 'CHANGED' from sqldb->refresh_link\n");
#endif
}

static gboolean
table_is_in_query (SqlQuery * q, SqlMemTable * table)
{
	gboolean found = FALSE;
	GSList *list;
	QueryElement *qe;

	list = q->objects;
	while (list && !found) {
		qe = (QueryElement *) (list->data);
		if (qe->type == QUERY_ELT_FIELD)
			if (sql_db_find_table_from_field (q->conf->db,
							  SQL_MEM_FIELD (qe->
									 main))
			    == table)
				found = TRUE;
		list = g_slist_next (list);
	}
	return found;
}


/* CBs to be connected to SqlDb::"ff_link_created/dropped" */
void
sql_query_auto_links_add_link (GtkObject * obj, SqlMemFlink * link,
			       SqlQuery * q)
{
	SqlMemTable *fromt, *tot;
	GSList *list;
	QueryLink *ql;
	gboolean found;

	/* trying to find the link to see if needed to insert it */
	list = q->depend_list;
	found = FALSE;
	while (list && !found) {
		ql = (QueryLink *) (list->data);
		if (ql->link == link)
			found = TRUE;
		list = g_slist_next (list);
	}

	if (!found) {
		/* creating a new element */
		fromt = sql_db_find_table_from_field (q->conf->db,
						      link->from);
		tot = sql_db_find_table_from_field (q->conf->db, link->to);
		if (fromt && tot) {
			ql = (QueryLink *) g_malloc (sizeof (QueryLink));
			ql->link = link;
			refresh_link (q, ql);
			/* insert it into the list */
			q->depend_list = g_slist_append (q->depend_list, ql);
#ifdef debug_signal
			g_print (">> 'AUTO_LINK_ADDED' from sql_query_auto_links_add_link\n");
#endif
			gtk_signal_emit (GTK_OBJECT (q),
					 sql_query_signals[AUTO_LINK_ADDED],
					 ql);
#ifdef debug_signal
			g_print ("<< 'AUTO_LINK_ADDED' from sql_query_auto_links_add_link\n");
#endif
		}
	}
}

void
sql_query_auto_links_del_link (GtkObject * obj, SqlMemFlink * link,
			       SqlQuery * q)
{
	GSList *list;
	QueryLink *ql;
	gboolean found;

	list = q->depend_list;
	found = FALSE;
	while (list && !found) {
		ql = (QueryLink *) (list->data);
		if (ql->link == link) {
			found = TRUE;
			q->depend_list = g_slist_remove_link (q->depend_list, list);
			g_slist_free_1 (list);
#ifdef debug_signal
			g_print (">> 'AUTO_LINK_REMOVED' from sql_query_auto_links_del_link\n");
#endif
			gtk_signal_emit (GTK_OBJECT (q),
					 sql_query_signals[AUTO_LINK_REMOVED],
					 ql);
#ifdef debug_signal
			g_print ("<< 'AUTO_LINK_REMOVED' from sql_query_auto_links_del_link\n");
#endif
			g_free (ql);
		}
		else
			list = g_slist_next (list);
	}
}

/*
 * Query WHERE/HAVING management
 */
QueryWhereNode *
sql_query_cond_new_node (SqlQuery * q,
			 QueryWhereLogicType logic,
			 QueryWhereNode * parent, QueryWhereNode * sibling)
{
	/* emit the wh_cond_created signal */
	QueryWhereNode *node;
	GNode *n_parent, *n_sibling;

	g_return_val_if_fail (q != NULL, NULL);

	/* tests to assert the validity of parameters */
	n_parent = sql_query_gnode_find (q, parent);
	n_sibling = sql_query_gnode_find (q, sibling);
	if (n_parent && !g_node_is_ancestor (q->where_tree, n_parent))
		return NULL;
	if (n_parent && n_sibling && (n_sibling->parent != n_parent))
		return NULL;

	/* Ok for the insertion */
	node = g_new0 (QueryWhereNode, 1);
	node->is_node = TRUE;
	node->val.logic = logic;

	if (n_parent)
		g_node_insert_data_before (n_parent, n_sibling, node);
	else
		g_node_insert_data_before (q->where_tree, n_sibling, node);

#ifdef debug_signal
	g_print (">> 'WH_COND_CREATED' from sql_query_cond_new_node\n");
#endif
	g_print (">> 'WH_COND_CREATED' from sql_query_cond_new_node\n");
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[WH_COND_CREATED],
			 node);
#ifdef debug_signal
	g_print ("<< 'WH_COND_CREATED' from sql_query_cond_new_node\n");
#endif

	return node;
}

GNode *
sql_query_gnode_find (SqlQuery * q, QueryWhereNode * node)
{
	g_return_val_if_fail (q != NULL, NULL);
	return g_node_find (q->where_tree, G_LEVEL_ORDER, G_TRAVERSE_ALL,
			    node);
}

QueryWhereNode *
sql_query_cond_new_leaf (SqlQuery * q,
			 QueryElement * l_elt,
			 QueryWhereOpType op,
			 QueryWhereRightOperandType r_type,
			 gpointer r_object,
			 QueryWhereNode * parent, QueryWhereNode * sibling)
{
	/* emit the wh_cond_created signal */
	QueryWhereNode *node;
	GNode *n_parent, *n_sibling;

	g_return_val_if_fail (q != NULL, NULL);

	/* tests to assert the validity of parameters */
	n_parent = sql_query_gnode_find (q, parent);
	n_sibling = sql_query_gnode_find (q, sibling);
	if (n_parent && n_sibling && G_NODE_IS_LEAF (n_parent))
		return NULL;
	if (!n_parent && n_sibling)
		return NULL;
	if (n_parent && !g_node_is_ancestor (q->where_tree, n_parent))
		return NULL;
	if (n_parent && n_sibling && (n_sibling->parent != n_parent))
		return NULL;

	/* Ok for the insertion */
	node = g_new0 (QueryWhereNode, 1);
	node->is_node = FALSE;
	node->val.leaf.l_elt = l_elt;
	node->val.leaf.op = op;
	node->val.leaf.r_type = r_type;
	node->val.leaf.r_object = r_object;

	if (n_parent)
		g_node_insert_data_before (n_parent, n_sibling, node);
	else
		g_node_insert_data_before (q->where_tree, n_sibling, node);

#ifdef debug_signal
	g_print (">> 'WH_COND_CREATED' from sql_query_cond_new_node\n");
#endif
	g_print (">> 'WH_COND_CREATED' from sql_query_cond_new_node\n");
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[WH_COND_CREATED],
			 node);
#ifdef debug_signal
	g_print ("<< 'WH_COND_CREATED' from sql_query_cond_new_node\n");
#endif

	return node;
}

static void actual_sql_query_cond_del (SqlQuery * q, QueryWhereNode * node);
void
sql_query_cond_del (SqlQuery * q, QueryWhereNode * node)
{
	GNode *gnode, *gnode_parent;
	g_return_if_fail (q != NULL);
	g_return_if_fail (node != NULL);

	gnode = sql_query_gnode_find (q, node);
	gnode_parent = gnode->parent;
	actual_sql_query_cond_del (q, node);

	/* remove the parent GNode as well if it has no more children */
	if (gnode_parent && !gnode_parent->children && (gnode_parent->data != q) &&	/* not first node */
	    ((QueryWhereNode *) gnode_parent->data)->is_node)
		sql_query_cond_del (q, (QueryWhereNode *) gnode_parent->data);
}

static void
actual_sql_query_cond_del (SqlQuery * q, QueryWhereNode * node)
{
	GNode *gnode, *gnode_parent;

	gnode = sql_query_gnode_find (q, node);
	gnode_parent = gnode->parent;
	/* parsing the tree to destroy all its children (recursive) */
	if (gnode) {
		while (gnode->children)
			actual_sql_query_cond_del (q,
						   (QueryWhereNode *) gnode->
						   children->data);

		/* WARNING here: we emit the signal while the GNode is still in the tree! */
#ifdef debug_signal
		g_print (">> 'WH_COND_DROPPED' from sql_query_cond_del\n");
#endif
		gtk_signal_emit (GTK_OBJECT (q),
				 sql_query_signals[WH_COND_DROPPED], node);
#ifdef debug_signal
		g_print ("<< 'WH_COND_DROPPED' from sql_query_cond_del\n");
#endif

		/* now that there are not any more children, destroy that node */
		g_node_unlink (gnode);
		g_node_destroy (gnode);
	}
}

void
sql_query_cond_move (SqlQuery * q,
		     QueryWhereNode * node,
		     QueryWhereNode * to_parent, QueryWhereNode * to_sibling)
{
	/* emit the wh_cond_moved signal */
	GNode *gnode;

	g_return_if_fail (q != NULL);
	gnode = sql_query_gnode_find (q, node);
	if (gnode) {
		GNode *n_parent, *n_sibling = NULL;

		g_node_unlink (gnode);
		n_parent = sql_query_gnode_find (q, to_parent);
		if (to_sibling)
			n_sibling = sql_query_gnode_find (q, to_sibling);
		if (n_parent)
			g_node_insert_before (n_parent, n_sibling, gnode);
		else
			g_node_insert_before (q->where_tree, n_sibling,
					      gnode);
#ifdef debug_signal
		g_print (">> 'WH_COND_MOVED' from sql_query_cond_move\n");
#endif
		g_print (">> 'WH_COND_MOVED' from sql_query_cond_move\n");
		gtk_signal_emit (GTK_OBJECT (q),
				 sql_query_signals[WH_COND_MOVED], node);
#ifdef debug_signal
		g_print ("<< 'WH_COND_MOVED' from sql_query_cond_move\n");
#endif
	}
}

void
sql_query_cond_modif_node (SqlQuery * q,
			   QueryWhereNode * node, QueryWhereLogicType logic)
{
	g_return_if_fail (q != NULL);
	g_return_if_fail (node != NULL);
	g_return_if_fail (node->is_node == TRUE);

	node->val.logic = logic;

#ifdef debug_signal
	g_print (">> 'WH_COND_MODIFIED' from sql_query_cond_modif_node\n");
#endif
	g_print (">> 'WH_COND_MODIFIED' from sql_query_cond_modif_node\n");
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[WH_COND_MODIFIED],
			 node);
#ifdef debug_signal
	g_print ("<< 'WH_COND_MODIFIED' from sql_query_cond_modif_node\n");
#endif
}

void
sql_query_cond_modif_leaf (SqlQuery * q,
			   QueryWhereNode * node,
			   QueryElement * l_elt,
			   QueryWhereOpType op,
			   QueryWhereRightOperandType r_type,
			   gpointer r_object)
{
	g_return_if_fail (q != NULL);
	g_return_if_fail (node != NULL);
	g_return_if_fail (node->is_node == FALSE);

	node->val.leaf.l_elt = l_elt;
	node->val.leaf.op = op;
	node->val.leaf.r_type = r_type;
	node->val.leaf.r_object = r_object;

#ifdef debug_signal
	g_print (">> 'WH_COND_MODIFIED' from sql_query_cond_modif_leaf\n");
#endif
	g_print (">> 'WH_COND_MODIFIED' from sql_query_cond_modif_leaf\n");
	gtk_signal_emit (GTK_OBJECT (q), sql_query_signals[WH_COND_MODIFIED],
			 node);
#ifdef debug_signal
	g_print ("<< 'WH_COND_MODIFIED' from sql_query_cond_modif_leaf\n");
#endif
}

/*
 * Interrogation helper functions 
 */

gint
sql_query_is_field_printed (SqlQuery * q, SqlMemField * field)
{
	gint result = -1, i;
	GSList *list;
	QueryElement *elt;

	list = q->objects;
	i = 0;
	while (list && (result == -1)) {
		elt = (QueryElement *) (list->data);
		if ((elt->type == QUERY_ELT_FIELD) && (elt->main == field) &&
		    (elt->print_name) && (*(elt->print_name) != '\0'))
			result = i;
		if ((elt->print_name) && (*(elt->print_name) != '\0'))
			i++;
		list = g_slist_next (list);
	}

	return result;
}

QueryElement *
sql_query_get_query_element (SqlQuery * q, gint pos)
{
	QueryElement *qe = NULL;
	gint i;
	GSList *list;

	if (pos < 0)
		return NULL;
	list = q->objects;
	i = -1;
	while (list && !qe) {
		if ((((QueryElement *) (list->data))->print_name) &&
		    (*(((QueryElement *) (list->data))->print_name) !=
		     '\0')) {
			i++;
			if (i == pos)
				qe = (QueryElement *) (list->data);
		}
		list = g_slist_next (list);
	}

	return qe;
}

gint
sql_query_get_queryres_pos (SqlQuery * q, QueryElement * qe)
{
	return g_slist_index (q->objects, qe);
}

gint
sql_query_get_queryres_pos_f (SqlQuery * q, SqlMemField * field)
{
	GSList *list;
	gboolean found;
	list = q->objects;
	found = FALSE;
	while (list && !found) {
		if ((((QueryElement *) (list->data))->type == QUERY_ELT_FIELD)
		    && (((QueryElement *) (list->data))->main == field))
			found = TRUE;
		else
			list = g_slist_next (list);
	}
	if (!list) {
		g_warning ("sql_query_get_queryres_pos_f: field not found");
		return -1;
	}
	else
		return g_slist_index (q->objects, list->data);
}

gboolean
sql_query_is_table_concerned (SqlQuery * q, SqlMemTable * table)
{
	GSList *list;
	QueryElement *qe;
	gboolean found = FALSE;
	g_return_val_if_fail (table != NULL, FALSE);

	list = q->objects;
	while (list && !found) {
		qe = (QueryElement *) (list->data);
		if ((qe->type == QUERY_ELT_FIELD) &&
		    (g_slist_find (table->fields, SQL_MEM_FIELD (qe->main))))
			found = TRUE;
		list = g_slist_next (list);
	}

	return found;
}


static gchar *get_select_part_of_sql (SqlQuery * q, GSList * missing_values,
				      gboolean * any);
static gchar *get_from_part_of_sql (SqlQuery * q, gboolean * any);
static gchar *get_where_part_of_sql (SqlQuery * q, GSList * missing_values,
				     gboolean * any);

/*
 * Getting the SQL version of the query
 */
gchar *
sql_query_get_select_query (SqlQuery * q, GSList * missing_values)
{
	gchar *str, *retval = NULL;
	GString *SQL;
	GSList *elts, *tables;
	gboolean first, any;

	if (q->text_only)
		retval = g_strdup (q->text_sql);
	else {
		SQL = g_string_new ("SELECT\n");

		/* SELECT objects */
		str = get_select_part_of_sql (q, missing_values, &any);
		if (any) {
			g_string_append (SQL, str);
		}
		g_free (str);

		/* FROM relations */
		str = get_from_part_of_sql (q, &any);
		if (any) {
			g_string_sprintfa (SQL, "\nFROM\n%s", str);
		}
		g_free (str);

		/* WHERE conditions */
		str = get_where_part_of_sql (q, missing_values, &any);
		if (any) {
			g_string_sprintfa (SQL, "\nWHERE\n%s", str);
		}
		g_free (str);

		retval = SQL->str;
		g_string_free (SQL, FALSE);
	}

	return retval;
}

static gchar *object_to_str (SqlQuery * q, QueryElement * elt,
			     GSList * missing_values);

/* creates a new string with the objects that go in the SELECT part to SQL */
static gchar *
get_select_part_of_sql (SqlQuery * q, GSList * missing_values, gboolean * any)
{
	gchar *str, *retval = NULL;
	GString *objects;
	gboolean first;
	GSList *elts;

		elts = q->objects;
		first = TRUE;
	objects = g_string_new ("");
		while (elts) {
			if (((QueryElement *) (elts->data))->print_name) {
				if (first)
					first = FALSE;
				else
				g_string_append (objects, ",\n");

			str = object_to_str (q, (QueryElement *) (elts->data),
						     missing_values);
			g_string_sprintfa (objects, "%s as %s", str,
					   ((QueryElement *) (elts->data))->
						   print_name);
				g_free (str);
			}
			elts = g_slist_next (elts);
		}

	retval = objects->str;
	g_string_free (objects, FALSE);
	*any = !first;
	return retval;
}

/* WARNING: Allocates memory */
static gchar *
object_to_str (SqlQuery * q, QueryElement * elt, GSList * missing_values)
{
	gchar *str = NULL;
	GString *string;
	SqlMemTable *table;
	gboolean first;
	GSList *args;
	QueryElementValue *qv;

	switch (elt->type) {
	case QUERY_ELT_FIELD:
		table = sql_db_find_table_from_field (q->conf->db,
						      SQL_MEM_FIELD (elt->
								     main));
		str = g_strdup_printf ("%s.%s", table->name,
				       SQL_MEM_FIELD (elt->main)->name);
		break;
	case QUERY_ELT_FUNCTION:
		string = g_string_new (SQL_DATA_FUNCTION (elt->main)->
				       sqlname);
		first = TRUE;
		args = elt->args;
		g_string_append (string, "(");
		while (args) {
			if (first)
				first = FALSE;
			else
				g_string_append (string, ", ");

			str = object_to_str (q, (QueryElement *) (args->data),
					     missing_values);
			g_string_append (string, str);
			g_free (str);

			args = g_slist_next (args);
		}
		g_string_append (string, ")");

		str = string->str;
		g_string_free (string, FALSE);
		break;
	case QUERY_ELT_AGGREGATE:
		string = g_string_new (SQL_DATA_AGGREGATE (elt->main)->
				       sqlname);
		args = elt->args;
		g_string_append (string, "(");
		str = object_to_str (q, (QueryElement *) (args->data),
				     missing_values);
		g_string_append (string, str);
		g_free (str);
		g_string_append (string, ")");

		str = string->str;
		g_string_free (string, FALSE);
		break;
	case QUERY_ELT_VALUE:
		str = NULL;
		if (missing_values) {
			GSList *list = missing_values;
			QueryMissingValue *qmv;

			while (list && !str) {
				qmv = (QueryMissingValue *) (list->data);
				if ((qmv->type == QUERY_WHERE_ELT)
				    && (qmv->node == elt))
					str = g_strdup (qmv->sql);
				list = g_slist_next (list);
			}
		}
		if (!str) {
			qv = (QueryElementValue *) elt->main;
			if (qv->default_val) {
				if (qv->ask_at_exec)
					str = g_strdup_printf ("%s_?",
							       qv->
							       default_val);
				else
					str = g_strdup (qv->default_val);
			}
			else
				str = g_strdup ("???");
		}
	}

	return str;
}

static GSList *complement_tables_list (SqlQuery * q,
				       GSList * list, QueryElement * elt);

static GSList *
complement_tables_list (SqlQuery * q, GSList * list, QueryElement * elt)
{
	SqlMemTable *table;
	GSList *nlist, *args;

	nlist = list;
	switch (elt->type) {
	case QUERY_ELT_FIELD:
		table = sql_db_find_table_from_field (q->conf->db,
						      SQL_MEM_FIELD (elt->
								     main));
		if (!g_slist_find (nlist, table))
			nlist = g_slist_append (nlist, table);
		break;
	case QUERY_ELT_FUNCTION:
	case QUERY_ELT_AGGREGATE:
		args = elt->args;
		while (args) {
			nlist = complement_tables_list (q, nlist,
							(QueryElement
							 *) (args->data));
			args = g_slist_next (args);
		}
		break;
	case QUERY_ELT_VALUE:
	}

	return nlist;
}

/* recursive func */
static gchar *sql_render_a_where_node (SqlQuery * q, GNode * gnode,
				       GSList * missing_values);
/* returns the WHERE part of a query while taking care of the automatic links
   and any other WHERE constraint imposed to the query 
   The allocated string has to be freed */
static gchar *
get_where_part_of_sql (SqlQuery * q, GSList * missing_values, gboolean * any)
{
	gboolean firstnode = TRUE;
		GString *gstr;
		gchar *wh;

		gstr = g_string_new ("");
	if (q->where_tree->children) {
		GNode *gnode = q->where_tree->children;

		while (gnode) {
			if (firstnode)
				firstnode = FALSE;
			else
				g_string_append (gstr, "AND ");
			wh = sql_render_a_where_node (q, gnode,
						      missing_values);
			g_string_append (gstr, wh);
			g_free (wh);
			g_string_append (gstr, "\n");
			gnode = gnode->next;
		}
	}
		wh = gstr->str;
		g_string_free (gstr, FALSE);
	*any = !firstnode;
	return wh;
}

/* render a node and its children */
static gchar *
sql_render_a_where_node (SqlQuery * q, GNode * gnode, GSList * missing_values)
{
	GString *whs;
	QueryWhereNode *wh;
	gchar *str;

	wh = (QueryWhereNode *) gnode->data;
	whs = g_string_new ("");

	if (wh->is_node) {
		if (wh->val.logic == QUERY_WHERE_NOT)
			g_string_append (whs, "NOT ");

		if (gnode->children && gnode->children->next) {	/* two or more children */
			GNode *nde = gnode->children;
			gboolean first = TRUE;

			g_string_append (whs, "(");
			while (nde) {
				if (first)
					first = FALSE;
				else {
					gchar *ope;
					switch (wh->val.logic) {
					case QUERY_WHERE_AND:
						ope = " AND ";
						break;
					case QUERY_WHERE_OR:
						ope = " OR ";
						break;
					case QUERY_WHERE_NOT:
						ope = " NOT ";
						break;
					}
					g_string_append (whs, ope);
				}
				str = sql_render_a_where_node (q, nde,
							       missing_values);
				g_string_append (whs, str);
				g_free (str);
				nde = nde->next;
			}
			g_string_append (whs, ")");
		}
		if (gnode->children && !gnode->children->next) {	/* only one child */
			str = sql_render_a_where_node (q, gnode->children,
						       missing_values);
			g_string_append (whs, str);
			g_free (str);
		}
		if (!gnode->children) {
			/* if no child, then do nothing, but signal an error */
			g_warning
				("A node has no children. Should not have happened");
		}
	}
	else {			/* here we have a leaf */
		str = object_to_str (q, wh->val.leaf.l_elt, missing_values);
		g_string_append (whs, str);
		g_free (str);
		switch (wh->val.leaf.op) {
		case QUERY_OP_EQUAL:
			str = "=";
			break;
		case QUERY_OP_DIFF:
			str = "!=";
			break;
		case QUERY_OP_SUP:
			str = ">";
			break;
		case QUERY_OP_SUPEQUAL:
			str = ">=";
			break;
		case QUERY_OP_INF:
			str = "<";
			break;
		case QUERY_OP_INFEQUAL:
			str = "<=";
			break;
		case QUERY_OP_LIKE:
			str = "LIKE";
			break;
		case QUERY_OP_REGEX:
			str = "MATCH";
			break;
		case QUERY_OP_IN:
			str = "IN";
			break;
		default:
			str = "???";
		}
		g_string_sprintfa (whs, " %s ", str);
		str = NULL;
		if (wh->val.leaf.r_type == QUERY_WHERE_ASK) {
			if (missing_values) {
				GSList *list = missing_values;
				QueryMissingValue *qmv;
				while (list && !str) {
					qmv = (QueryMissingValue *) (list->
								     data);
					if ((qmv->type == wh->val.leaf.r_type)
					    && (qmv->node == wh))
						str = g_strdup (qmv->sql);
					list = g_slist_next (list);
				}
			}
			if (!str)
				str = g_strdup ("???");
		}
		else
			str = object_to_str (q,
					     (QueryElement *) wh->val.leaf.
					     r_object, missing_values);
		g_string_append (whs, str);
		g_free (str);
	}

	str = whs->str;
	g_string_free (whs, FALSE);
	return str;
}


/* Creates the FROM part of the SQL statement
 * The general case is a comma separated list of relations (views) where a
 * relation can be a table or colection of joined tables.
 * The joins will be associated from left to right.
 * The token FROM is not returned, only the list of tables
 *
 * ex: tblAlone, (tblA INNER JOIN tblB ON tblA.ID=tblB.AID) 
 *     LEFT JOIN tblC ON tblB.ID=tblC.BID
 *
 * Restrictions:
 *  - only equi-joins; TODO FM: introduce theta joins (add new condition 
 *    column in the Joins' clist and edit it in a dialog box, or inline
 *    if widget supports it)
 *  - no complex keys! TODO FM: relationship widget must support complex
 *    keys and in the Joins' clist widget, the FROM and TO columns could have
 *    only the tables's names and the fields appear in the new condition column
 *  - aliased tables not supported. Requires support for aliased tables in the
 *    relationships widget: if a table is added a 2nd time, it gets an alias.
 *    This limitation forbides self joins, recursive joins involving a 2nd
 *    table (A-B-A) and more than one relationship per same pair of tables
 *    eg, (A.b1id-B.id ; A.b2id-B.id)
 *  - the links list must be in the intended order of joining, which is
 *    particularly important if outer joins are involved. In this case, 
 *    associating joins with parentesis is needed to avoid ambiguities
 *   
 * Note 1: those remaining tables (without join conditions) will create a
 *   cartesian product (CROSS), unless the user manually specifys joining
 *   conditions in the WHERE part of the query)
 *
 * Algorithm:
 * - obtain the list of tables used in the objects
 * - loop the joins to create the SQL join statements
 * - during the former loop, removes joined tables from the objects table list
 * - the remaining tables from the initial list, if any, will be appended
 *   to the begining of the FROM part, with tables separated by commas
 * - next, the string with joined tables will be added to the statement
 *  
 */

#define JOINED(tables, tbl) (g_slist_find(tables, tbl) != NULL)

static gchar *
get_from_part_of_sql (SqlQuery * q, gboolean * any)
{
	GSList *links;
	gboolean first;
	GString *sql_from, *joins, *all_joins;
	gchar *retval;
	QueryLink *ql;
	SqlMemTable *fromt, *tot;
	GSList *elts, *object_tables, *joined_tables;

	/* look for tables in the objects */
	elts = q->objects;
	object_tables = NULL;
	while (elts) {
		if (((QueryElement *) (elts->data))->print_name)
			object_tables =
				complement_tables_list (q, object_tables,
							(QueryElement
							 *) (elts->data));
		elts = g_slist_next (elts);
	}

	joins = g_string_new ("");
	all_joins = g_string_new ("");
	joined_tables = NULL;	/* keeps list of tables that were joined */
	links = q->depend_list;
	/* create the SQL joins and remove the processed tables from the tables list
	 * This loop assumes that the links list is sorted to create the
	 * joins by associating tables from left to right */
	while (links) {
		ql = (QueryLink *) (links->data);
		if (ql->join_type != JOIN_CROSS) {
			fromt = sql_db_find_table_from_field (q->conf->db,
							      ql->link->from);
			tot = sql_db_find_table_from_field (q->conf->db,
							    ql->link->to);
			if (fromt && tot) {
				if (JOINED (joined_tables, fromt)) {	/* hence, time to join tot */
					object_tables =
						g_slist_remove (object_tables,
								tot);
					joined_tables =
						g_slist_prepend
						(joined_tables, tot);
					g_string_sprintfa (joins,
							   " %s %s ON (%s.%s = %s.%s)",
							   join_type_get_sql
							   (ql->join_type),
							   tot->name,
							   fromt->name,
							   ql->link->from->
							   name, tot->name,
							   ql->link->to->
							   name);
				}
				else if (JOINED (joined_tables, tot)) {	/* hence, time to join fromt */
					object_tables =
						g_slist_remove (object_tables,
								fromt);
					joined_tables =
						g_slist_prepend
						(joined_tables, fromt);
					g_string_sprintfa (joins,
							   " %s %s ON (%s.%s = %s.%s)",
							   join_type_get_sql
							   (ql->join_type),
							   fromt->name,
							   tot->name,
							   ql->link->to->name,
							   fromt->name,
							   ql->link->from->
							   name);
				}
				else {	/* fromt, tot were not joined, hence new join set */
					if (joins->len != 0)
						g_string_append (all_joins,
								 ", ");
					g_string_append (all_joins,
							 joins->str);
					object_tables =
						g_slist_remove (object_tables,
								fromt);
					object_tables =
						g_slist_remove (object_tables,
								tot);
					joined_tables =
						g_slist_prepend
						(joined_tables, fromt);
					joined_tables =
						g_slist_prepend
						(joined_tables, tot);
					g_string_sprintf (joins,
							  "%s %s %s ON (%s.%s = %s.%s)",
							  fromt->name,
							  join_type_get_sql
							  (ql->join_type),
							  tot->name,
							  fromt->name,
							  ql->link->from->
							  name, tot->name,
							  ql->link->to->name);
				}
			}
		}
		links = g_slist_next (links);
	}
	if ((all_joins->len != 0) && (joins->len != 0))
		g_string_append (all_joins, ", ");
	g_string_append (all_joins, joins->str);
	g_string_free (joins, TRUE);
	g_slist_free (joined_tables);
	*any = all_joins->len > 0;

	/* Start the FROM part with the object tables not taking part of a join */
	first = TRUE;
	elts = object_tables;
	sql_from = g_string_new ("");
	while (elts) {
		if (first) {
			first = FALSE;
		}
		else
			g_string_append (sql_from, ", ");
		g_string_append (sql_from, SQL_MEM_TABLE (elts->data)->name);
		elts = g_slist_next (elts);
	}
	g_slist_free (object_tables);
	*any = *any || !first;

	/* Append all joins to the FROM part */
	g_string_append (sql_from, all_joins->str);
	g_string_free (all_joins, TRUE);

	retval = sql_from->str;
	g_string_free (sql_from, FALSE);
	return retval;
}

gchar *
sql_query_get_select_all_query_with_values (SqlQuery * q, gint row)
{
	GString *str;
	gchar *retval;
	gboolean wherearg, first;
	gint i;
	GSList *list;
	QueryElement *qe;
	SqlMemTable *t;
	gchar *strchar, *tmpstr;
	SqlDataDisplayFns *fns;
	GdaField *gfield;

	str = g_string_new ("");
	/*g_string_sprintfa(str, "%s ", q->modif_table->name); */

	/* WHERE list of automatic links */
	retval = get_where_part_of_sql (q, NULL, &wherearg);
	if (wherearg) {
		g_string_sprintfa (str, " WHERE\n%s", retval);
	}
	g_free (retval);

	/* WHERE part with the values provided by the SqlQueryRes */
	i = 0;
	first = !wherearg;
	list = q->objects;
	while (list) {
		qe = (QueryElement *) (list->data);
		if (qe->type == QUERY_ELT_FIELD) {
			t = sql_db_find_table_from_field (q->conf->db,
							  SQL_MEM_FIELD (qe->
									 main));
			if (!wherearg) {
				g_string_append (str, " WHERE");
				wherearg = TRUE;
			}

			fns = sql_access_get_object_display_fns (q->conf->srv,
								 GTK_OBJECT
								 (qe->main));
			/*GTK_OBJECT(SQL_MEM_FIELD(qe->main)->type)); */
			/*gfield = sql_query_res_get_gdafield(q->res, row, i); */
			strchar = (fns->gdafield_to_sql) (gfield);
			tmpstr = sql_query_res_stringify (gfield);
			if (!tmpstr) {
				if (!first)
					g_string_append (str, " AND");
				else
					first = FALSE;
				g_string_sprintfa (str, " %s.%s is null",
						   t->name,
						   SQL_MEM_FIELD (qe->main)->
						   name);
			}
			else {
				if (strchar) {
					if (!first)
						g_string_append (str, " AND");
					else
						first = FALSE;
					g_string_sprintfa (str, " %s.%s=%s",
							   t->name,
							   SQL_MEM_FIELD (qe->
									  main)->
							   name, strchar);
				}
			}
			if (strchar)
				g_free (strchar);
			g_free (tmpstr);
		}
		i++;
		list = g_slist_next (list);
	}

	retval = str->str;
	g_string_free (str, FALSE);
	return retval;
}
