# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the highlevel wizard functions.
"""

import sys
import os
import glob

from xml.sax import make_parser

from qt import *

# import the standard wizards
import MessageBoxWizard.MessageBoxWizard
import FileDialogWizard.FileDialogWizard
import InputDialogWizard.InputDialogWizard
import ColorDialogWizard.ColorDialogWizard
import FontDialogWizard.FontDialogWizard
import QRegExpWizard.QRegExpWizard
import PyRegExpWizard.PyRegExpWizard

from eric3config import getConfig

class Wizards(QObject):
    """
    Class implementing the highlevel wizard functions.
    """
    def __init__(self, loc, parent = None):
        """
        Constructor
        
        @param loc the localizer string as determined by the main script (string)
        """
        QObject.__init__(self, parent)
        self.registeredWizards = []
        self.registeredTranslators = []
        self.locale = loc
        self.init()
       
    def init(self):
        """
        Private method to register all the wizards.
        """
        self.actions = []
        
        self.wizDir = os.path.abspath(getConfig('ericWizardsDir'))
        
        wizard = MessageBoxWizard.MessageBoxWizard.MessageBoxWizard(self)
        self.registeredWizards.append(wizard)
        self.actions.append(wizard.initAction())
        self.loadWizardTranslator(wizard)
        
        wizard = FileDialogWizard.FileDialogWizard.FileDialogWizard(self)
        self.registeredWizards.append(wizard)
        self.actions.append(wizard.initAction())
        self.loadWizardTranslator(wizard)
        
        wizard = InputDialogWizard.InputDialogWizard.InputDialogWizard(self)
        self.registeredWizards.append(wizard)
        self.actions.append(wizard.initAction())
        self.loadWizardTranslator(wizard)
        
        self.insertSeparator()
        
        wizard = ColorDialogWizard.ColorDialogWizard.ColorDialogWizard(self)
        self.registeredWizards.append(wizard)
        self.actions.append(wizard.initAction())
        self.loadWizardTranslator(wizard)
        
        wizard = FontDialogWizard.FontDialogWizard.FontDialogWizard(self)
        self.registeredWizards.append(wizard)
        self.actions.append(wizard.initAction())
        self.loadWizardTranslator(wizard)
        
        self.insertSeparator()
        
        wizard = QRegExpWizard.QRegExpWizard.QRegExpWizard(self)
        self.registeredWizards.append(wizard)
        self.actions.append(wizard.initAction())
        self.loadWizardTranslator(wizard)
        
        wizard = PyRegExpWizard.PyRegExpWizard.PyRegExpWizard(self)
        self.registeredWizards.append(wizard)
        self.actions.append(wizard.initAction())
        self.loadWizardTranslator(wizard)
        
    def loadWizardTranslator(self, wizard):
        """
        Private method to load a wizard's translator.
        
        @param wizard reference to the wizard object.
        """
        trans = wizard.getTranslationFileBaseName()
        if trans is not None:
            trans = "%s%s" % (trans, self.locale)
            qtrans = QTranslator(None)
            dir, trans = os.path.split('%s.%s' % (self.wizDir, trans))
            dir = dir.replace('.', os.sep)
            if qtrans.load(trans, dir):
                self.registeredTranslators.append(qtrans)
                qApp.installTranslator(qtrans)
        
    def insertSeparator(self):
        """
        Private method to prepare a menu separator.
        """
        self.registeredWizards.append(None)
        
    def initMenu(self):
        """
        Public slot to initialize the wizards menu.
        
        @return the generated menu
        """
        menu = QPopupMenu(self.parent())
        menu.insertTearOffHandle()
        
        for wizard in self.registeredWizards:
            if wizard == None:
                menu.insertSeparator()
            else:
                action = wizard.getAction()
                action.addTo(menu)
            
        return menu
        
    def getCurrentEditor(self):
        """
        Public method to get a reference to the current editor.
        
        @return the current editor
        """
        return  self.parent().getViewManager().activeWindow() 
        
    def getActions(self):
        """
        Public method to get a list of all actions.
        
        @return list of all actions (list of E3Action)
        """
        return self.actions[:]
