;;; emms.el --- Emacs MultiMedia System

;; Copyright (C) 2003  Jorgen Schfer

;; Author: Jorgen Schfer <forcer@forcix.cx>
;; Keywords: emms, mp3, mpeg, multimedia

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This is the very core of EMMS. It provides ways to play a track
;; using `emms-start', to go through the playlist using the commands
;; `emms-next' and `emms-previous', stop the playing using
;; `emms-stop', and see what is currently playing using `emms-show'.

;; But in itself, this core is useless. It does not know how to play
;; anything - you need players for this - nor does it know any way how
;; to get tracks to play - you need sources for this.

;; A sample configuration is offered in emms-defaults.el, so you might
;; just want to use that file.

;;; Code:

;; Version control
(defvar emms-version "0.2 $Revision: 1.26 $"
  "EMMS version string.")
;; $Id: emms.el,v 1.26 2004/06/07 12:27:22 Lukhas Exp $

;;; User Customization
(defgroup emms nil
  "*EMMS, the Emacs MultiMedia System for playing multimedia tracks
using external players."
  :prefix "emms-"
  :group 'applications)

(defgroup emms-player nil
  "*Players for EMMS."
  :prefix "emms-player-"
  :group 'emms)

(defgroup emms-source nil
  "*Track sources for EMMS."
  :prefix "emms-source-"
  :group 'emms)

(defcustom emms-source-list nil
  "*A list of sources EMMS can get tracks from. You need to set this
to use `emms-play-all'.
For example, to play all the media files in ~/media/, you can use the
following setting (this uses emms-source-file.el):

(setq emms-source-list '((emms-source-directory-tree \"~/media\")))"
  :group 'emms
  :type '(repeat sexp))

(defcustom emms-player-list nil
  "*A list of players EMMS can use. You need to set this!"
  :group 'emms
  :type '(repeat function))

(defcustom emms-show-format "Currently playing %s"
  "*The format to use for `emms-show'. The only argument is the string
returned by `emms-track-description'."
  :group 'emms
  :type 'string)

(defcustom emms-repeat-playlist nil
  "*Non-nil for repeating the playlist after playing the last track."
  :group 'emms
  :type 'boolean)

(defcustom emms-track-description-function 'emms-track-description
  "*A function to be called to give a nice, user-friendly description
of the track passed to it as an argument."
  :group 'emms
  :type 'function)

(defcustom emms-play-all-preparation-function 'identity
  "*A function that prepares the list of tracks for `emms-play-all'.
This can be used to sort or shuffle."
  :group 'emms
  :type 'function
  :options '(emms-play-all-sort
             emms-play-all-shuffle))

(defcustom emms-play-all-except nil
  "*A list of functions that return non-nil if their argument track
should not be played using `emms-play-all'."
  :group 'emms
  :type '(repeat function))

(defcustom emms-sort-lessp-function 'emms-sort-track-name-less-p
  "*A function that compares two tracks, and returns non-nil if the
first track should be sorted before the second (see `sort')."
  :group 'emms
  :type 'function)

(defcustom emms-playlist-changed-hook nil
  "*A hook run when the playlist of EMMS has changed."
  :group 'emms
  :type 'hook)

(defcustom emms-playlist-current-changed-hook nil
  "*A hook run when the current track in the playlist of EMMS has
changed."
  :group 'emms
  :type 'hook)

(defcustom emms-player-started-hook nil
  "*A hook run when an EMMS player started playing."
  :group 'emms
  :type 'hook
  :options '(emms-show))

(defcustom emms-player-stopped-hook nil
  "*A hook run when an EMMS player stopped playing.
See `emms-player-finished-hook'."
  :group 'emms
  :type 'hook)

(defcustom emms-player-finished-hook '(emms-next-noerror)
  "*A hook run when an EMMS player stopped playing.
Please pay attention to the differences between
`emms-player-finished-hook' and `emms-player-stopped-hook'. The
former is called when the player is stopped interactivly and the
latter when the player returned."
  :group 'emms
  :type 'hook
  :options '(emms-next-noerror))

(defvar emms-player-playing-p nil
  "Nil if there is no player playing right now. Else, it's the
currently playing player.")

(defvar emms-playlist []
  "The current playlist, a vector of tracks.")
(defvar emms-playlist-current nil
  "The current track, or nil if there is no playlist.")

;;; User Interface
(defun emms-start ()
  "Start EMMS and play the current track in the playlist."
  (interactive)
  (unless emms-player-playing-p
    (emms-player-start (emms-playlist-current-track))))

(defun emms-stop ()
  "Stop EMMS."
  (interactive)
  (when emms-player-playing-p
    (emms-player-stop)))

(defun emms-next ()
  "Play the next track in the playlist.
This might behave funny in `emms-player-finished-hook', use
`emms-next-noerror' instead for that."
  (interactive)
  (when emms-player-playing-p
    (emms-stop))
  (if (emms-playlist-next)
      (emms-start)
    (error "No next track in playlist.")))

(defun emms-next-noerror ()
  "Play the next track in the playlist, but don't signal an error when
we're at the end. This should be called when no player is playing.
This is a suitable function to put in `emms-player-finished-hook'."
  (interactive)
  (when emms-player-playing-p
    (error "A track is already playing."))
  (cond ((emms-playlist-next)
	 (emms-start))
	(emms-repeat-playlist
	 (setq emms-playlist-current 0)
	 (emms-start))
	(t
	 (message "No next track in playlist."))))

(defun emms-previous ()
  "Play the previous track in the playlist."
  (interactive)
  (when emms-player-playing-p
    (emms-stop))
  (if (emms-playlist-previous)
      (emms-start)
    (error "No previous track in playlist.")))

;;;###autoload
(defun emms-play-all ()
  "Play all sources in `emms-source-list'."
  (interactive)
  (let ((new (emms-play-all-except
              emms-play-all-except
              (funcall emms-play-all-preparation-function
                       (apply #'append
                              (mapcar (lambda (source)
                                        (apply (car source)
                                               (cdr source)))
                                      emms-source-list))))))
    (when (null new)
      (error "No tracks found for playing! Look at `emms-source-list'."))
    (emms-stop)
    (emms-playlist-set-playlist (vconcat new))
    (emms-playlist-set-current 0)
    (emms-start)))

(defun emms-play-all-except (funclist tracklist)
  "Delete all elements in TRACKLIST for which any element of FUNCLIST
returns non-nil."
  (let ((tracks nil)
        (funcs nil))
    (while tracklist
      (setq funcs funclist)
      (let ((add t))
        (while funcs
          (when (funcall (car funcs) (car tracklist))
            (setq add nil))
          (setq funcs (cdr funcs)))
        (when add
          (setq tracks (cons (car tracklist) tracks))))
      (setq tracklist (cdr tracklist)))
    (reverse tracks)))

(defun emms-show (&optional insertp)
  "Show the currently playing track in the minibuffer, or insert it if
INSERTP is non-nil. This uses `emms-show-format' to format the current
track."
  (interactive "P")
  (let ((string (format emms-show-format (emms-playlist-current))))
    (if insertp
        (insert string)
      (message "%s" string))))

(defun emms-shuffle ()
  "Shuffle the playlist."
  (interactive)
  (emms-playlist-shuffle))

(defun emms-sort ()
  "Sort the playlist."
  (interactive)
  (emms-playlist-sort))

(defun emms-play-all-sort (list)
  "Sort LIST according to `emms-sort-lessp-function'."
  (sort list emms-sort-lessp-function))

(defun emms-play-all-shuffle (list)
  "Shuffle LIST."
  (append (emms-playlist-shuffle-vector (vconcat list)) nil))

(defun emms-sort-track-name-less-p (a b)
  "Return non-nil if the track name of a is before b."
  (string< (emms-track-name a)
           (emms-track-name b)))

;;; A Track
;; This is a simple datatype to store track information. A track
;; consists of a type (a symbol) and a name (a string).

(defun emms-track (type name)
  "Create a track with type TYPE and name NAME."
  (list (cons 'type type)
        (cons 'name name)))

(defun emms-track-type (track)
  "Return the type of TRACK."
  (emms-track-get track 'type))

(defun emms-track-name (track)
  "Return the name of TRACK."
  (emms-track-get track 'name))

(defun emms-track-get (track name &optional inexistent)
  "Return the value of NAME for TRACK. If there is no value, return
INEXISTENT (or nil if not given)."
  (let ((val (assq name track)))
    (if val
        (cdr val)
      inexistent)))

(defun emms-track-set (track name value)
  "Set the value of NAME for TRACK to VALUE."
  (let ((cell (assq name track)))
    (if cell
        (setcdr cell value)
      (nconc track
             (list (cons name value))))))

(defun emms-track-description (track)
  "A simple function to give a user-readable description of a track.
If it's a file track, it's just the filename.
Else it's the type and the name with a colon in between."
  (if (eq 'file (emms-track-type track))
      (emms-track-name track)
    (concat (symbol-name (emms-track-type track))
            ":"
            (emms-track-name track))))

;;; The Playlist
;; This is a simple vector storing the current playlist. If at all
;; possible, don't access the vector directly but use the functions
;; provided here. If you can't avoid accessing the vector directly, be
;; careful to call the right hooks.

(defun emms-playlist-current ()
  "Return a description of the currently playing track.
This uses `emms-track-description-function'."
  (funcall emms-track-description-function
           (emms-playlist-current-track)))

(defun emms-playlist-current-track ()
  "Return the currently playing track."
  (when emms-playlist-current
    (emms-playlist-get-track emms-playlist-current)))

(defun emms-playlist-get-track-description (track)
  "Return a description of TRACK.
This uses `emms-track-description-function'."
  (funcall emms-track-description-function track))

(defun emms-playlist-get (number)
  "Return a description of the NUMBERth entry of the current playlist.
This uses `emms-track-description-function'"
  (funcall emms-track-description-function
           (emms-playlist-get-track number)))

(defun emms-playlist-get-track (number)
  "Return the NUMBERth track of the current playlist."
  (aref emms-playlist number))

(defun emms-playlist-set-playlist (new)
  "Set the playlist to NEW.
This runs `emms-playlist-changed-hook'."
  (setq emms-playlist new)
  (cond
   ((= 0 (length new))
    (setq emms-playlist-current nil))
   ((null emms-playlist-current)
    (setq emms-playlist-current 0))
   ((>= emms-playlist-current (length emms-playlist))
    (setq emms-playlist-current (- (length emms-playlist) 1))))
  (run-hooks 'emms-playlist-changed-hook))

(defun emms-playlist-get-playlist ()
  "Return the current playlist."
  emms-playlist)

(defun emms-playlist-set-current (new)
  "Set the current track in the playlist to NEW (a number).
This runs `emms-playlist-current-changed-hook'."
  (setq emms-playlist-current new)
  (run-hooks 'emms-playlist-current-changed-hook))

(defun emms-playlist-get-current ()
  "Return the number of the current track, or nil if the playlist is
empty."
  emms-playlist-current)

(defun emms-playlist-next ()
  "Advance the current track to the next entry in the playlist and
return non-nil. Return nil if there is no next track."
  (let ((cur (emms-playlist-get-current)))
    (when (and cur
               (< cur (- (length (emms-playlist-get-playlist)) 1)))
      (emms-playlist-set-current (+ 1 cur))
      t)))

(defun emms-playlist-previous ()
  "Set the current track to the previous entry in the playlist and
return non-nil. Return nil if there is no previous track."
  (let ((cur (emms-playlist-get-current)))
    (when (and cur 
               (> cur 0))
      (emms-playlist-set-current (- cur 1))
      t)))

(defun emms-playlist-add (seq &optional idx)
  "Add each track of the sequence SEQ to the current playlist.
Insert at IDX, which defaults to the end."
  (let ((idx (or idx (length emms-playlist))))
    (emms-playlist-set-playlist
     (vconcat (substring emms-playlist 0 idx)
              seq
              (substring emms-playlist idx)))))

(defun emms-playlist-remove (idx)
  "Remove track at IDX from playlist."
  (emms-playlist-set-playlist
   (vconcat (substring emms-playlist 0 idx)
	    (substring emms-playlist (1+ idx)))))
  
(defun emms-playlist-shuffle ()
  "Shuffle the current playlist."
  (emms-playlist-set-playlist
   (emms-playlist-shuffle-vector
    (emms-playlist-get-playlist))))

(defun emms-playlist-sort ()
  "Sort the current playlist according to `emms-sort-lessp-function'"
  (emms-playlist-set-playlist
   (emms-playlist-sort-vector
    (emms-playlist-get-playlist))))

(defun emms-playlist-shuffle-vector (vector)
  "Shuffle VECTOR."
  (let ((i (- (length vector) 1)))
    (while (>= i 0)
      (let* ((r (random (1+ i)))
             (old (aref vector r)))
        (aset vector r (aref vector i))
        (aset vector i old))
      (setq i (- i 1))))
  vector)

(defun emms-playlist-sort-vector (vector)
  "Sort VECTOR according to `emms-sort-lessp-function'."
  (vconcat (sort (append vector nil)
                 emms-sort-lessp-function)))

;;; Sources
;; A source is a source for tracks. It's just a function that returns
;; a list. The define-emms-source macro also defines emms-play-SOURCE
;; and emms-add-SOURCE. The former will replace the current playlist,
;; the latter add to the end.

(defmacro define-emms-source (name arglist &rest body)
  "Define a new source named NAME. This will define the function
emms-source-NAME and commands emms-play-NAME and emms-add-NAME. The
source function will return a list to be played, while the play and
add commands will be interactive (if you specify an interactive form
in the BODY) to be used by the user, both calling the source function
with the appropriate arguments. See emms-source-file.el for some
examples."
  (let ((source-name (intern (format "emms-source-%s" name)))
        (source-play (intern (format "emms-play-%s" name)))
        (source-add (intern (format "emms-add-%s" name)))
        (docstring "A source of tracks for EMMS.")
        (interactive nil)
        (call-args (delete '&rest
                           (delete '&optional
                                   arglist))))
    (when (stringp (car body))
      (setq docstring (car body)
            body (cdr body)))
    (when (eq 'interactive (caar body))
      (setq interactive (car body)
            body (cdr body)))
    `(progn
       (defun ,source-name ,arglist
         ,docstring
         ,@body)
       (defun ,source-play ,arglist
         ,docstring
         ,interactive
         (emms-source-play (,source-name ,@call-args)))
       (defun ,source-add ,arglist
         ,docstring
         ,interactive
         (emms-source-add (,source-name ,@call-args))))))

(defun emms-source-play (lis)
  "Play the tracks in LIS."
  (let ((new (vconcat lis)))
    (when (zerop (length new))
      (error "No tracks found." source))
    (emms-stop)
    (emms-playlist-set-playlist new)
    (emms-playlist-set-current 0)
    (emms-start)))

(defun emms-source-add (lis)
  "Add the tracks in LIS to the playlist."
  (emms-playlist-add lis))

;;; Players
;; A player is a function that gets called with an action and possible
;; arguments to that action, and evaluates that action.
;; See `emms-define-player' for more information.

(defun emms-player-for (track)
  "Return the player that feels responsible for TRACK, or nil if there
is none."
  (let ((lis emms-player-list))
    (while (and lis
                (not (emms-player-do (car lis) 'playablep track)))
      (setq lis (cdr lis)))
    (if lis
        (car lis)
      nil)))

(defun emms-player-do (player method &rest args)
  "Run the method METHOD on player PLAYER, passing any additional ARGS
on the way."
  (apply #'funcall player method args))

(defun emms-player-start (track)
  "Start playing TRACK."
  (if emms-player-playing-p
      (error "A player is already playing - stop it first.")
    (let ((player (emms-player-for track)))
      (if (not player)
          (error "I can't play %s!" track)
        (emms-player-do player 'start track)
        (setq emms-player-playing-p player)
        (run-hooks 'emms-player-started-hook)))))

(defun emms-player-stop ()
  "Stop the currently playing player."
  (when emms-player-playing-p
    (emms-player-do emms-player-playing-p 'stop)
    (setq emms-player-playing-p nil)))

(defun emms-player-stopped (&optional stopped-p)
  "Declare that the current player has finished playing.
This should only be called by a player. If STOPPED-P is non-nil
run `emms-player-stopped-hook' otherwise run
`emms-player-finished-hook'."
  (setq emms-player-playing-p nil)
  (if stopped-p 
      (run-hooks 'emms-player-stopped-hook)
    (run-hooks 'emms-player-finished-hook)))

(defmacro define-emms-player (name &rest methods)
  "Define a player for EMMS named NAME with the methods METHODS, which
is a list of method names as keywords followed by the functions to
call for them. Use `emms-player-do' to run the methods of a player.

These are the predefined methods for EMMS.

    :start track ....... Start playing this track. (Required)
    :stop .............. Stop this player. (Required)
    :playablep track ... Return non-nil if the player can play this
                         track. (Required)

    :regex ............. Return a regexp of files this player can
                         play. Used by another part of EMMS to build
                         useful file lists. (Optional)

    :has method ........ Return non-nil if this player knows about
                         METHOD. (This is defined by this macro)

A player can define any additional method it wants.

When a player has stopped playing, it should call
`emms-player-stopped' as the last action before returning."
  (unless (plist-get methods :start)
    (error "Player definition %s missing a :start method." name))
  (unless (plist-get methods :stop)
    (error "Player definition %s missing a :stop method." name))
  (unless (plist-get methods :playablep)
    (error "Player definition %s missing a :playablep method." name))
  `(defun ,name (method &rest args)
     (let ((plist ,(cons 'list methods)))
       (if (eq method 'has)
           (let ((methodname (intern (concat ":" (symbol-name (car args))))))
             (or (plist-get plist methodname)
                 (eq (car args) :has)))
         (let* ((methodname (intern (concat ":" (symbol-name method))))
                (function (plist-get plist methodname)))
           (cond
            ((functionp function)
             (apply #'funcall function args))
            (function
             function)
            (t
             (error "Player %s does not know how to %s!"
                    ',name
                    method))))))))

(provide 'emms)
;;; emms.el ends here
