/*
 * SBus detection feature for Discover
 *
 * Copyright (C) 2004 Jurij Smakov <jurij@wooyd.org>
 *                      and Joshua Kwan <joshk@triplehelix.org>
 *
 * Based largely on prtconf, which is
 *   Copyright (C) 1998 Jakub Jelinek (jj@ultra.linux.cz)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#define _GNU_SOURCE

#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/utsname.h>
#include <sys/ioctl.h>

#include <asm/openpromio.h>

#include "discover.h"
#include "utils.h"

#define DECL_OP(size) struct openpromio *op = (struct openpromio *)buf; \
                      op->oprom_size = (size)
#define MAX_PROP        128
#define MAX_VAL         (4096-128-4)

int promfd = 0;
int prom_current_node = 0;
char buf[4096];
struct sbus_info *sbus_first, *sbus_result;

static int prom_getsibling(int node)
{
  DECL_OP(sizeof(int));
        
  if (node == -1) return 0;
  *(int *)op->oprom_array = node;
  if (ioctl (promfd, OPROMNEXT, op) < 0)
    return 0;
  prom_current_node = *(int *)op->oprom_array;
  return *(int *)op->oprom_array;
}

static int prom_getchild(int node)
{
  DECL_OP(sizeof(int));
        
  if (!node || node == -1) return 0;
  *(int *)op->oprom_array = node;
  if (ioctl (promfd, OPROMCHILD, op) < 0)
    return 0;
  prom_current_node = *(int *)op->oprom_array;
  return *(int *)op->oprom_array;
}

static char *prom_getproperty(char *prop, int *lenp)
{
  DECL_OP(MAX_VAL);
        
  strcpy (op->oprom_array, prop);
  if (ioctl (promfd, OPROMGETPROP, op) < 0)
    return 0;
  if (lenp) *lenp = op->oprom_size;
  return op->oprom_array;
}

static int prom_searchsiblings(char *name)
{
  char *prop;
  int len;
        
  for (;;) {
    if (!(prop = prom_getproperty("name", &len)))
      return 0;
    prop[len] = 0;
    if (!strcmp(prop, name))
      return prom_current_node;
    if (!prom_getsibling(prom_current_node))
      return 0;
  }
}
 
static void prom_walk(int node, struct cards_lst *lst)
{
  int nextnode;
  int len;        
  char *prop;
  struct cards_lst *walk_lst;

  prop = strndup(prom_getproperty("name", &len), MAX_PROP);
  if (prop && len > 0)
  {
    if ((nextnode = prom_getchild(node)) > 0)
      prom_walk(nextnode, lst);
    
    if(!sbus_first) {
      sbus_first = sbus_result = 
	(struct sbus_info *) my_malloc(sizeof(struct sbus_info));
      sbus_result->next = (struct sbus_info *) NULL;
    } else {
      sbus_result->next = (struct sbus_info *) my_malloc(sizeof(struct sbus_info));
      sbus_result = sbus_result->next;
      sbus_result->next = (struct sbus_info *) NULL;
    }
    sbus_result->dev_id = prop;
    sbus_result->modulename = s_unknown;
    sbus_result->model = s_unknown;
    for(walk_lst = lst; walk_lst; walk_lst = walk_lst->next) {
      if((walk_lst->bus != SBUS) || (walk_lst->dev_id == NULL)) continue;
      if(strncmp(sbus_result->dev_id, walk_lst->dev_id, MAX_PROP) == 0) {
	sbus_result->vendor = walk_lst->vendor;
	sbus_result->modulename = walk_lst->modulename;
	sbus_result->model = walk_lst->model;
	sbus_result->type = walk_lst->type;
        if (debug)
          printf("\t\tFound %s %s\n", sbus_result->vendor, sbus_result->model);
      }
    }
    /* Oh well.. */
    free(prop);
  }
  nextnode = prom_getsibling(node);
  if (nextnode) prom_walk(nextnode, lst);
}


struct sbus_info *sbus_detect(struct cards_lst *lst)
{
  int node, root;

  if(debug) {
    fprintf(stdout, "\nProbing SBUS devices...\n");
  }
  /********************************************************************/
  /********************* SBUS DEVICE DETECTION ************************/
  /********************************************************************/
  sbus_first = sbus_result = (struct sbus_info *) NULL;
  promfd = open(PATH_OPENPROM, O_RDONLY);
  
  if (promfd == -1) /* devfs? */
    promfd = open(PATH_OPENPROM_DEVFS, O_RDONLY);
  
  if (promfd == -1) /* nothing? doh! */
    return sbus_result;
  
  node = prom_getsibling(0);
  if(!node) return sbus_result;                 /* No root node */
  if(!prom_getchild(node)) return sbus_result;  /* No root children */
  
  root = prom_current_node;

  /* sparc64 has its sbus node in the root. */
  node = prom_searchsiblings("sbus");
  if (node)
  {
    node = prom_getchild(node);
    if (node)
    {
      prom_walk(node, lst);
      return sbus_first;
    }
    else
      return sbus_result;
  }
  else
  {
    /* force return to root, it existed 1ms before, why not now */
    prom_current_node = root;
    
    node = prom_searchsiblings("iommu");
    if(!node) return sbus_result;               /* No iommu node */
    node = prom_getchild(node);                  
    if(!node) return sbus_result;               /* No iommu children */
    node = prom_searchsiblings("sbus");
    if(!node) return sbus_result;               /* No sbus node */
    node = prom_getchild(node);
    if(!node) return sbus_result;               /* No sbus children */

    /* prom_walk walks the sbus node children and fills out the fields */

    prom_walk(node, lst);
    return sbus_first;
  }
}
