/**
 * Manipulate context free grammars.
 * @author Shaun Jackman <sdj@sfu.ca>
 * @copyright Copyright 2004 Shaun Jackman
 */


#include <stdio.h>
#include <stdlib.h>
#include "cfg.h"
#include "dictionary.h"
#include "set.h"
#include "util.h"


/** Prints the specified production. */
void
print_production( const Production* production,
		const Dictionary* symbols)
{
	int i;
	if( symbols != NULL) {
		printf( "%s -> ", get_key( symbols, production->lhs));
		for( i = 0; i < production->count; i++)
			printf( "%s ", get_key( symbols, production->rhs[i]));
		putchar( '\n');
	} else {
		printf( "%d -> ", production->lhs);
		for( i = 0; i < production->count; i++)
			printf( "%d ", production->rhs[i]);
		putchar( '\n');
	}
}


/** Prints the specified CFG. */
void
print_cfg( const CFG* cfg)
{
	Set set;
	int i;

	puts( "# Non-terminals.");
	set = cfg->nonterminal;
	while( (i = remove_first_element( &set)) != EMPTY_SET)
		printf( "%s %d\n", get_key( &cfg->symbols, i), i);

	puts( "# Terminals.");
	set = cfg->terminal;
	while( (i = remove_first_element( &set)) != EMPTY_SET)
		printf( "%s %d\n", get_key( &cfg->symbols, i), i);

	puts( "# Productions.");
	for( i = 0; i < cfg->count; i++)
		print_production( &cfg->productions[i], NULL);
}


/** Clears this CFG. */
void
clear_cfg( CFG* cfg)
{
	clear_dictionary( &cfg->symbols);
	clear_set( &cfg->nonterminal);
	clear_set( &cfg->terminal);
	memset( cfg->precedence, 0, sizeof cfg->precedence);
	cfg->max = 1;
	cfg->count = 0;
	cfg->productions = allocate_memory( sizeof *cfg->productions);
}


/** Destroys this CFG. */
void
destroy_cfg( CFG* cfg)
{
	destroy_dictionary( &cfg->symbols);
	cfg->max = cfg->count = 0;
	free( cfg->productions);
}


/** Allots a production. */
static Production*
allot_production( CFG* cfg)
{
	if( cfg->count+1 > cfg->max) {
		cfg->max *= 2;
		cfg->productions = reallocate_memory( cfg->productions,
				cfg->max * sizeof *cfg->productions);
		assume( cfg->productions != NULL, __FUNCTION__);
	}
	return &cfg->productions[cfg->count++];
}


/** Reads the right-hand-side of a production from the specified file.
 */
static void
read_rhs( CFG* cfg, Production* production, FILE* file)
{
	char* symbol;
	production->count = 0;
	while( getc( file) != '\n' &&
			fscanf( file, "%as", &symbol) == 1) {
		int id = insert( &cfg->symbols, symbol);
		assumex( production->count < MAX_RHS,
				"limit of %d elements on the right-hand-side of a "
				"production exceeded", MAX_RHS);
		production->rhs[production->count++] = id;
		if( !contains_element( &cfg->nonterminal, id))
			add_element( &cfg->terminal, id);
	}
}


/** Reads a CFG from the specified file. */
void
read_cfg( CFG* cfg, FILE* file)
{
	char* symbol;
	clear_cfg( cfg);
	while( fscanf( file, "%as", &symbol) == 1) {
		int id = insert( &cfg->symbols, symbol);
		Production* production = allot_production( cfg);
		production->lhs = id;
		add_element( &cfg->nonterminal, id);
		remove_element( &cfg->terminal, id);
		read_rhs( cfg, production, file);
	}
}


/** Reads precedence information from the specified file. */
void
read_precedence( CFG* g, FILE* file)
{
	char* symbol;
	int precedence;
	while( fscanf( file, "%as %d", &symbol, &precedence) == 2) {
		int a = find( &g->symbols, symbol);
		assumex( a != 0, "unknown symbol %s", symbol);
		free( symbol);
		g->precedence[a] = precedence;
	}
}
