/*
 *  docbook-to-latex using dancer-XML parser
 *  Copyright (C) 2003 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#define _GNU_SOURCE
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <getopt.h>
#include "dancer-xml.h"
#include "config.h"

static int 
replace_external_entity(const char * s)
{
  if (!strcmp(s,"gt"))
    printf("$>$");
  else if (!strcmp(s,"lt"))
    printf("$<$");
  else if (!strcmp(s,"amp"))
    printf("\\&{}");
  else
    {
      fprintf(stderr, "unknown entity %s\n", s);
      exit (1);
    }
  return 0;
}


/**
 *   remove exceeding space, and remove special-chars
 */
static int
convert_print (const char * s /** The string to print */,  
	       int verbatim /** preserve carriage returns? */)
{
  const char * endptr = NULL;
  
  if (!s)
    {
      return 1;
    }
  
  while (*s && isspace(*s))
    {
      s++;
    }
  while (*s)
    {
      switch (*s) 
	{
	case '$':
	  printf ("\\${}");
	  break;
	case '_':
	  printf ("\\verb!_!");
	  break;
	case '\\':
	  printf ("\\verb'\\'");
	  break;
	case '~':
	  printf ("\\~\\ ");
	  break;
	case '\n':
	  if (verbatim) printf ("\\\\%%");
	  printf ("\n");
	  break;
	case '#':
	  printf ("$\\sharp$");
	  break;
	case '[':
	  printf ("{}[{}");
	  break;
	case ']':
	  printf ("{}]{}");
	  break;
	  
	case '&':
	  if (NULL!=(endptr = strchr (s, ';')))
	    {
	      char * tmps;
	      tmps = strndup(s+1,endptr-s-1);
	      replace_external_entity(tmps);
	      free(tmps);
	      s=endptr;
	    }
	  else
	    {
	      fprintf(stderr, "Unterminated & \n");
	      exit (1);
	    }
	  break;
	default:
	  putc(*s,stdout);
	}
      
      s++;
      
    }
  
  return 0;
}

static const char * getatrstring(dxml_element * e, const char * attr)
{
  dxml_attribute * a = e->element_attribute;
  while (a)
    {
      if (!strcmp(a->attribute_name,attr))
	return a->attribute_data;
      a=a->next;
    }
  return NULL;
}

static int get_paratype(dxml_element *b, int verbatim);
static int get_bookchapter(dxml_element *b, int paraenter);

static int
maybe_label(dxml_element *b)
{
  const char * s;
  if ((s = getatrstring(b, "id")))
    printf ("\\label{%s}\n", s);
  return 0;
}

static int
get_table(dxml_element *b_parent)
{
  int i;
  dxml_element *row, *entry, *b = b_parent -> child;
  int columns=atoi(getatrstring (dxml_get_element_byname(b, "tgroup"), "cols"));
  printf ("\\begin{table}\n"
	  "\\caption{%s}\n",
	  dxml_get_PCDATA_bysimplepath(b,"title"));
  maybe_label(b_parent);
  
  printf ("\\begin{tabular}{c");
  for (i=0; i< columns - 1 ; i++)
    printf ("|c");
  printf ("}\n\\hline\n\\hline\n");
  for (row = dxml_get_element_bysimplepath(b, "tgroup/tbody/row"); row; row = row->next )
    {
      for (entry = row->child; entry; entry = entry -> next)
	{
	  get_paratype(entry->child,0);
	  if (entry->next)
	    printf ("&");
	  
	}

      printf ("\\\\\n");
    }
  printf (
	  "\\hline\n\\hline\n"
	  "\\end{tabular}\n"
	  "\\end{table}\n"
	  );
  return 0;
}

static int get_itemizedlist(dxml_element *b)
{
  printf("\\begin{itemize}\n");
  while (b)
    {
      
      if (!strcasecmp("listitem", b->element_name))
	{
	  printf("\\item ");
	  get_bookchapter(b->child,0);
	  printf("\n");
	}
      else
	dxml_dump_element(b);
      
      b=b->next;
    }
  printf("\\end{itemize}\n");
  return 0;
}

/**
   Something that appears inside the para elements and outside the
   para elements.

   @return 1 on match.
 */
static int get_bothtypes(dxml_element *b, int inlinetype /** 1 if it is inside para*/)
{
  if (!strcasecmp("screen", b->element_name))
    {
      if (!inlinetype)
	puts("\n\n\n\n");	/* enter newline when not inline */
      puts("{\\tt %\n");
      get_paratype(b->child, 1);
      puts("}");
      if (!inlinetype)
	puts("\n\n\n\n");	/* enter newline when not inline */
    }
  else
    return 0;
  return 1;
}


/** 
 * handle para-type ones which can contain PCDATA
 */
static int
get_paratype(dxml_element *b, 
	     int verbatim/** whether it is in screen env. or not. */)
{
  while (b)
    {
      if (b->element_type == element_type_pcdata)
	{
	  convert_print (b->element_name, verbatim);
	}
      else if (get_bothtypes(b, 1))
	{
	}
      else if ((!strcasecmp("command", b->element_name)) ||
	       (!strcasecmp("filename", b->element_name)) ||
	       (!strcasecmp("keysym", b->element_name)) ||
	       (!strcasecmp("keycombo", b->element_name)) ||
	       (!strcasecmp("otheraddr", b->element_name)) ||
	       (!strcasecmp("prompt", b->element_name)) ||
	       (!strcasecmp("address", b->element_name)))
	{
	  printf ("{\\tt ");
	  get_paratype(b->child, verbatim);
	  printf ("} ");
	}
      else if ((!strcasecmp("option", b->element_name)) ||
	       (!strcasecmp("parameter", b->element_name)))
	{
	  printf ("{\\it ");
	  get_paratype(b->child, verbatim);
	  printf ("} ");
	}
      else if (!strcasecmp("footnote", b->element_name))
	{
	  printf ("\\footnote{");
	  get_bookchapter(b->child,0);
	  printf ("} ");
	}
      else if (!strcasecmp("itemizedlist", b->element_name))
	{
	  get_itemizedlist(b->child);
	}
      else if ((!strcasecmp("xref", b->element_name))||
	       (!strcasecmp("link", b->element_name)))
	{
	  printf ("\\ref{%s}", 
		  getatrstring(b, "linkend"));
	  get_paratype(b->child, verbatim);
	}
      else if (!strcasecmp("ulink", b->element_name))
	{
	  get_paratype(b->child, verbatim);
	  printf ("(\\verb!%s!)", 
		  getatrstring(b, "url"));	  
	}
      else
	{
	  fprintf(stderr, "Unknown tag: %s\n", b->element_name);
	  dxml_dump_element(b);
	}
      
      
      b=b->next;
    }
  return 0;
}



/* handle chapter and sect* */
static int get_bookchapter(dxml_element *b, int paraenter)
{
  while (b)
    {
      if (b->element_type != element_type_element)
	{
	  exit (1);
	}
      else if (!strcasecmp("sect1", b->element_name))
	{
	  printf ("\n\\section{");
	  convert_print(dxml_get_PCDATA_bysimplepath(b->child, "title"),0);
	  printf("}\n\n");
	  maybe_label(b);
	  get_bookchapter(b->child,1);
	}
      else if (!strcasecmp("sect2", b->element_name))
	{
	  printf ("\n\\subsection{");
	  convert_print(dxml_get_PCDATA_bysimplepath(b->child, "title"),0);
	  printf("}\n\n");
	  maybe_label(b);
	  get_bookchapter(b->child,1);
	}
      else if (!strcasecmp("sect3", b->element_name))
	{
	  printf ("\n\\subsubsection{");
	  convert_print(dxml_get_PCDATA_bysimplepath(b->child, "title"),0);
	  printf("}\n\n");
	  maybe_label(b);
	  get_bookchapter(b->child,1);
	}
      else if (!strcasecmp("title", b->element_name))
	{
	}
      else if (get_bothtypes(b,0))
	{
	}
      else if (!strcasecmp("para", b->element_name))
	{
	  if (paraenter)puts("\n\n");
	  get_paratype(b->child, 0);
	  if (paraenter)puts("\n\n");
	}
      else if (!strcasecmp("itemizedlist", b->element_name))
	{
	  get_itemizedlist(b->child);
	}
      else if (!strcasecmp("table", b->element_name))
	{
	  get_table(b);
	}
      else
	{
	  fprintf(stderr, "Unknown tag: %s\n", b->element_name);
	  dxml_dump_element(b->child);
	}
      
      b=b->next;
    }
  return 0;
}

static int recursebook(dxml_element * b)
{
  dxml_element * tmp;
  for (tmp = dxml_get_element_byname(b, "chapter"); tmp; tmp=tmp->next)
    {
      printf ("\n\\chapter{");
      convert_print(dxml_get_PCDATA_bysimplepath(tmp->child, "title"),0);
      printf("}\n\n");
      maybe_label(tmp);
      get_bookchapter (tmp->child,1);
    }
  return 0;
}


int
main(int argc, char ** argv)
{
  dxml_element * tmp ;
  int c;
  const char * papersize="letter";
  const char * articletype="report";

  while ((c = getopt(argc, argv, "p")) != -1)
    {
      switch (c)
	{
	case 'p':
	  articletype="jreport";
	  papersize="a4j";
	  break;
	case '?':
	  return 1;
	default:
	  fprintf(stderr, "Unknown option -%c used\n", c);
	  return 1;
	}
    }

  printf (
	  "%s%s\n",
	  "%%Generated by dxml-db2latex",
	  VERSION);

  tmp = dxml_read_xml(stdin);
  if (tmp)
    {
      char * title = NULL;
      dxml_element * authorlink;
      

      printf (
	      "\\documentclass[%s]{%s}\n"
	      "\\usepackage{moreverb}\n"
	      "\\date{}\n"
	      "\\author{",
	      papersize, articletype);
      
      {
	int notfirst = 0;
	for (authorlink = dxml_get_element_bysimplepath(tmp,"book/bookinfo/authorgroup/author");
	     authorlink; 
	     authorlink = dxml_get_element_byname(authorlink->next, "author"))
	  {
	    if (notfirst) 
	      printf(", ");
	    else
	      notfirst = 1;
	    
	    convert_print(dxml_get_PCDATA_bysimplepath(authorlink, "author/firstname"),0);
	    printf(" ");
	    
	    convert_print(dxml_get_PCDATA_bysimplepath(authorlink, "author/surname"),0);
	  }
      }
      
      printf ("}\n");

   
      printf (
	      "\\title{");
      title=dxml_get_PCDATA_bysimplepath(tmp,"book/title");
      if (!title)
	title=dxml_get_PCDATA_bysimplepath(tmp,"book/bookinfo/title");
      if (title)
	convert_print(title,0);
      
      printf (
	      "}\n"
	      "\\begin{document}\n"
	      "\\maketitle\n"
	      "\\tableofcontents\n"
	      );
      
      recursebook(dxml_get_element_byname(tmp, "book")->child);
      printf (
	      "\\end{document}\n");
    }
  else
    {
      printf ("Something wicked happened!\n");
      return 1;
    }

  return 0;  
}
