/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_preferences_sql.pike,v 1.19.2.1 2004/03/19 10:09:04 vida Exp $
 */

#include <module.h>
#include <camas/globals.h>
inherit "module";

constant cvs_version="$Id: camas_preferences_sql.pike,v 1.19.2.1 2004/03/19 10:09:04 vida Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Preferences SQL";
constant module_doc  = "SQL preferences module for CAMAS. This module will take care of retrieving "
                       "and storing preferences from a SQL database for other CAMAS modules. "
                       "Currently only tested on MySQL.<br />"
                       "<i>Note</i>: If you load several preferences modules, Camas will store "
                       "information on each of them allowing you to migrate easily among them. "
                       "It will also try to load the preferences for each of them in the order of "
                       "the priority of each module. In this mode, it will stop after the first successfull "
                       "module. <br />";
constant module_unique = 1;
constant thread_safe = 1;		// I think this module should be :)

private mapping(string:object) sql_cons;
private string table;

// some optimizations for the preferences we already know about
constant properties2type = ([ 
  "layout": "varchar(50)",
  "replymsgprefix": "varchar(5)",
  "sortorder": "varchar(25)",
  "autobcc": "varchar(63)", 
  "delaycalc": "varchar(25)",
  "sortcolumn": "varchar(25)",
  "delaytime": "smallint(5) unsigned",
  "showlinkforalternatives": "tinyint(1) unsigned",
  "bcc2admin":  "tinyint(1) unsigned",
  "language": "varchar(50)",
  "replyincludemsg": "tinyint(1) unsigned",
	"replyontop": "tinyint(1) unsigned",
  "draftsfolder": "varchar(100)",
  "sentfolder": "varchar(100)",
  "answeredfolder": "varchar(100)",
  "trashfolder": "varchar(100)",
  "showinlineimages": "tinyint(1) unsigned",
  "showhtml": "tinyint(1) unsigned",
  "showtext": "tinyint(1) unsigned",
  "showhiddenheaders": "tinyint(1) unsigned",
  "addressbook": "text",
  "autologout": "smallint(5) unsigned",
  "saveattachments": "tinyint(1) unsigned",
  "filterbook": "varchar(100)",
  "autofilter": "varchar(100)",
  "blindsend": "tinyint(1) unsigned",
  "defaultpostloginscreen": "varchar(15)",
  "mailpath": "varchar(100)",
  "visiblemail": "smallint(5) unsigned",
  "organization": "varchar(100)",
  "address": "varchar(100)",
  "name" : "varchar(100)",
  "feedimapcache": "tinyint(1) unsigned",
]);

void create()
{
  #ifdef CAMAS_DEBUG
  defvar("debug",0,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
         "error log ?");
  #endif
  string sqlhelp = "Specifies the default host to use for SQL-queries.\n"
         "This argument can also be used to specify which SQL-server to "
         "use by specifying an \"SQL-URL\":<ul>\n"
         "<pre>[<i>sqlserver</i>://][[<i>user</i>][:<i>password</i>]@]"
         "[<i>host</i>[:<i>port</i>]]/<i>database</i></pre></ul><br>\n"
         "Valid values for \"sqlserver\" depend on which "
         "sql-servers your pike has support for, but the following "
         "might exist: msql, mysql, odbc, oracle, postgres.<br />\n"
         "<i>Note</i>: You can specify several SQL servers, see Multiple server"
         " startegy. <br />Comma separated values.";
  defvar("sqlserver", "mysql://localhost", "SQL server host",
         TYPE_STRING, sqlhelp);
  defvar("strategy", "failover", "Multiple server strategy",
         TYPE_STRING_LIST, "If you defined several SQL server, this module"
         " will can use these strategies: <ul><ol><b>Failover</b>: In this mode "
         "the module will always query the first server in the list unless there "
         "is an error. In this case it will try the next server.</ol>"
         "<ol><b>Load balancing</b>: The module will query the server in a random "
         "order. If a server is not available, it will retry with another random choosen "
         "one</ol></ul>", ({ "failover", "load balancing" }));
  defvar("autocreate", 1, "Auto create SQL table and fields",
         TYPE_FLAG, "If set to yes, the table will be automatically created. "
         "The fields used to store Camas preferences will also be created "
         "if needed by Camas core.");
  defvar("autodelete", 0, "Auto delete SQL fields",
         TYPE_FLAG, "Camas core contains all the fields needed to store "
         "preferences. If a fields is in the SQL table and not in Camas core "
         "while this option is set to yes, to will be automatically deleted");
  defvar("table", "CamasPreferences", "Table name",
         TYPE_STRING, "The table where to put preferences. This table "
         "will be automatically created and fields will be automatically "
         "created if new preferences are added to CAMAS core.");
  defvar("loginfield", "login", "The name of the login field", TYPE_STRING,
         "The login field will be used to save the login id of the user "
	 "Its value depends on your configuration of Camas auth");
  defvar("loginfieldtype", "varchar(32)", "The type of the login field", TYPE_STRING,
         "This is the internal type of the SQL login field. You can set "
	 "your own value if you need optimization or you can leave the default "
	 "one (32 characters) if you don't know. This field will be the primary key "
	 "and an index of the table");
  defvar("logoutsave", 1, "Always save preferences on logout",
         TYPE_FLAG, "If set the preferences will always be saved for this module "
	 "when the user logout. This is useful for migrating the preferences "
	 "from other method like IMAP to this one");
}

void connect(object conf)
{
  mixed global_err;
  array(string) servers;
#if constant(thread_create)
  object lock = conf->get_provider ("camas_main")->global_lock->lock();
#endif
  global_err = catch {
    sql_cons = ([ ]);
  };
#if constant(thread_create)
  destruct(lock);
#endif
  if(global_err)
    report_error("error in camas_preferences_sql.pike: %s\n", describe_backtrace(global_err));
  if(stringp(QUERY(sqlserver)))
    servers = QUERY(sqlserver) / ",";
  foreach(servers, string server)
  {
    server = String.trim_all_whites(server);
#if constant(thread_create)
    object lock = conf->get_provider ("camas_main")->global_lock->lock();
#endif  
    mixed err;
    global_err = catch {
      err = catch {
         sql_cons += ([ server: Sql.Sql(server) ]);
       };
       if(err)
       {
         if(sql_cons[server])
           destruct (sql_cons[server]);
         sql_cons[server] = 0;
         report_error("Camas Preferences: Fail to connect to %s: %s\n",
             server, describe_backtrace(err));
       }
       table = QUERY(table);
     };
#if constant(thread_create)
    destruct(lock);
#endif
    if(global_err)
      report_error("error in camas_preferences_sql.pike: %s\n", describe_backtrace(global_err));
    if(!err)
    {
      object sql_con = sql_cons[server];
      array prefproperties = indices(conf->get_provider ("camas_main")->prefproperties);
      err = catch {
        if(QUERY(autocreate))
        {
          create_table(sql_con);
          create_fields(prefproperties, sql_con);
        }
        if(QUERY(autodelete))
          delete_fields(prefproperties, sql_con);
      };
      if(err)
        report_error("Problem initializing database: %s\n", describe_backtrace(err));
    }
  }
  random_seed(getpid() + time());
}

void start(int cnt, object conf)
{
  connect(conf);
}

string status()
{
  string status = "";
  foreach(indices(sql_cons), string server)
  {
    object sql_con = sql_cons[server];
    if(!sql_con)
      status += "<strong><font color=\"red\">Not connected to database on " + server + 
        " <br /></font></strong>";
    else
    {
      status += sprintf("Connected to %s<br />", sql_con->host_info());
      if(is_table_exist(sql_con))
      {
        status += "<br /><table border=\"1\"><tr><th>Fields in the database</th></tr>";
        array(string) fields = get_current_fields(sql_con);
        foreach(fields, string field)
          status += "<tr><td>" + field + "</td></tr>";
        status += "</table>";
      }
      status += sprintf("<br />Number of entry in the database: %d<br />", nb_entry(sql_con));
    }
  }
  return status;
}

string query_provides()
{
  return("camas_preferences");
}

void stop()
{
  foreach(values(sql_cons), object sql_con) 
    if(sql_con)
      destruct(sql_con);
}

// the number of entries in the database
private int nb_entry(object sql_con)
{
  if(!is_table_exist(sql_con))
    return 0;
  return (int)sql_con->query(sprintf("SELECT count(*) as count from %s", table))[0]["count"];
}

// is the current table exist ?
private int is_table_exist(object sql_con)
{
  array(string) tables = sql_con->list_tables(table);
  if(sizeof(tables) == 0)
   return 0;
  return 1;
}

// create table if it doesn't exist
private void create_table(object sql_con)
{
  if(!is_table_exist(sql_con))
  {
    CDEBUG(sprintf("Table %s doesn't exist, creating it\n", table));
    sql_con->query(sprintf("CREATE table %s ( %s %s NOT NULL, "
          "PRIMARY KEY(%s), INDEX(%s))\n", table, QUERY(loginfield), 
	  QUERY(loginfieldtype), QUERY(loginfield), QUERY(loginfield)));
  }
}

// get all fields from the current table
private array(string) get_current_fields(object sql_con)
{
  array(mapping(string:string)) long_fields = sql_con->list_fields(table);
  array(string) current_fields = ({ });
  foreach(long_fields, mapping field)
    current_fields += ({ field["name"] });
  return current_fields;
}

// create fields if needed
private void create_fields(array(string) prop_fields, object sql_con)
{
  array(string) current_fields = get_current_fields(sql_con);
  array(string) needed_fields = prop_fields -  current_fields;
  if(sizeof(needed_fields) > 0)
  {
    CDEBUG(sprintf("Creating fields %s\n", needed_fields * ","));
    foreach(sort(needed_fields), string field)
    {
      if(field != QUERY(loginfield))
      {
        string type;
	if(properties2type[field])
	  type = properties2type[field];
	else
	  type = "varchar(255)";
	sql_con->query(sprintf("ALTER TABLE %s ADD COLUMN %s %s", 
	  table, field, type));
      }
    }
  }
}

// delete fields if uneccessary
private void delete_fields(array(string) prop_fields, object sql_con)
{
  array(string) current_fields = get_current_fields(sql_con);
  array(string) notneeded_fields = current_fields - prop_fields;
  if(sizeof(notneeded_fields) > 1)
  {
    CDEBUG(sprintf("Deleting fields %s\n", notneeded_fields * ","));
    foreach(notneeded_fields, string field)
    {
      if(field != "login")
        sql_con->query(sprintf("ALTER TABLE %s DROP COLUMN %s", table, field));
    }
  }
}

// does the user has an entry in the table ?
private int is_user_exists(string user, object sql_con)
{
  string query = sprintf("SELECT %s FROM %s WHERE %s='%s'", QUERY(loginfield),
    table, QUERY(loginfield), user);
  array(mapping(string:string)) res =  sql_con->query(query);
  if(sizeof(res) > 0)
    return 1;
  return 0;
}

// returns a sql server
string get_server(int i)
{
  switch(QUERY(strategy))
  {
    case "load balancing":
      return indices(sql_cons)[random(sizeof(sql_cons))];
    case "failover": 
      if(i < sizeof(sql_cons))
        return indices(sql_cons)[i];
    default: 
      return indices(sql_cons)[0];
  }
}

// automatically reconnected from disconnected SQL connection
// if needed
void reconnect(object sql_con, string server)
{
  int err = catch(sql_con->query(sprintf("SELECT %s from %s LIMIT 1",
    QUERY(loginfield), table)));
  if(!sql_con || err)
    sql_con = Sql.Sql(server);
}

/*
 * What we provide here
 */

//
//! method: int version(void)
//!  Give the CAMAS_PREFERENCES api version
//!  supported by the module
//! returns:
//!  the version of the API
//
int version()
{
  return 1;
}

//
//! method: mixed set_preferences(object id)
//!  Save the preferences into this module method
//! returns:
//!  a void by default
//!  an array containing the IMAP commands if using the 
//!  the IMAP module
//
mixed set_preferences(object id)
{
  mapping prefproperties = CAMAS_MODULE->prefproperties;
  string query;
  for(int i = 0; i < sizeof(sql_cons); i++)
  {
    string server = get_server(i);
    object sql_con = sql_cons[server];
    reconnect(sql_con, server);
    mixed err = catch {
      CDEBUG(sprintf("querying on host %O\n", server));
      if(!sql_con)
        throw (({ "Can't connect\n", backtrace() }));
      int user_exists = is_user_exists(CSESSION->login, sql_con);
      int first = 1;
      if(!user_exists)
        query = sprintf("INSERT INTO %s SET %s='%s',", table, QUERY(loginfield), CSESSION->login);
      else
        query = sprintf("UPDATE %s SET", table);
      
      foreach (indices(prefproperties), string prop) 
      {
        if (CSESSION[prop] && CSESSION["usersetup"+prop])
        {
          string value = CAMAS.Tools.encode_pref((string)(CSESSION)[prop]);
          if(!first)
            query += ",";
          else
            first = 0;
          query += sprintf(" %s='%s' ", prop, string_to_utf8(value));
        }
      }
      if(user_exists)
        query += sprintf("WHERE %s='%s'", QUERY(loginfield), CSESSION->login);
      CDEBUG(sprintf("set_preferences query=%s\n", query));
      sql_con->query(query);
    };
    if(err)
    {
      CDEBUG(sprintf("error querying server %s: %s\n", server, describe_backtrace(err)));
    }
    else
      break;
  }
}

//
//! method: mixed get_preferences(object id)
//!  Get the preferences from this module method
//! returns:
//!  a void by default
//!  an array containing the IMAP commands if using the 
//!  the IMAP module
//
mixed get_preferences(object id)
{
  mapping prefproperties = CAMAS_MODULE->prefproperties;
  string query = sprintf("SELECT * FROM %s WHERE %s='%s'",
    table, QUERY(loginfield), CSESSION->login);
  for(int i = 0; i < sizeof(sql_cons); i++)
  {
    string server = get_server(i);
    object sql_con = sql_cons[server];
    reconnect(sql_con, server);
    mixed err = catch {
      CDEBUG(sprintf("querying %O on host %O\n", query, server));
      if(!sql_con)
        throw (({ "Can't connect\n", backtrace() }));
      array(mapping(string:string)) res =  sql_con->query(query);
      if(sizeof(res) > 0)
      {
        mapping(string:string) result = res[0];
        foreach (indices(prefproperties), string prop) 
        {
          if (CSESSION[prop] && CSESSION["usersetup"+prop] && result[prop])
          {
            string pref;
            // some data might not be utf8 encoded if they are not written by us
            mixed err = catch { pref = utf8_to_string(result[prop]); };
            if(err)
              pref = result[prop];
            CSESSION[prop] = CAMAS.Tools.decode_pref(pref);
            CDEBUG(sprintf("CSESSION[%O]=%O\n", prop, CAMAS.Tools.decode_pref(pref)));
          }
        }
        CSESSION->prefsloaded = 1;
      }
    };
    if(err)
    {
      CDEBUG(sprintf("error querying server %s: %s\n", server, describe_backtrace(err)));
    }
    else
      break;
  }
}

//
//! method: mixed save_on_logout(id)
//!  Do we have to save the preferences on each user logout ?
//! returns:
//!  a void by default
//!  an array containing the IMAP commands if using the 
//!  the IMAP module
//
mixed save_on_logout(object id)
{
  if(QUERY(logoutsave))
  {
    CDEBUG("Saving preferences on logout\n");
    return set_preferences(id);
  }
}

/*
 * If you visit a file that doesn't contain these lines at its end, please     
 * cut and paste everything from here to that file.                            
 */                                                                            

/*
 * Local Variables:                                                            
 * c-basic-offset: 2                                                           
 * End:                                                                        
 *                                                                             
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! Debug the call / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: sqlserver
//!  type: TYPE_STRING
//!  name: SQL server host
//
//! defvar: strategy
//! If you defined several SQL server, this module will can use these strategies: <ul><ol><b>Failover</b>: In this mode the module will always query the first server in the list unless there is an error. In this case it will try the next server.</ol><ol><b>Load balancing</b>: The module will query the server in a random order. If a server is not available, it will retry with another random choosen one</ol></ul>
//!  type: TYPE_STRING_LIST
//!  name: Multiple server strategy
//
//! defvar: autocreate
//! If set to yes, the table will be automatically created. The fields used to store Camas preferences will also be created if needed by Camas core.
//!  type: TYPE_FLAG
//!  name: Auto create SQL table and fields
//
//! defvar: autodelete
//! Camas core contains all the fields needed to store preferences. If a fields is in the SQL table and not in Camas core while this option is set to yes, to will be automatically deleted
//!  type: TYPE_FLAG
//!  name: Auto delete SQL fields
//
//! defvar: table
//! The table where to put preferences. This table will be automatically created and fields will be automatically created if new preferences are added to CAMAS core.
//!  type: TYPE_STRING
//!  name: Table name
//
//! defvar: loginfield
//! The login field will be used to save the login id of the user Its value depends on your configuration of Camas auth
//!  type: TYPE_STRING
//!  name: The name of the login field
//
//! defvar: loginfieldtype
//! This is the internal type of the SQL login field. You can set your own value if you need optimization or you can leave the default one (32 characters) if you don't know. This field will be the primary key and an index of the table
//!  type: TYPE_STRING
//!  name: The type of the login field
//
//! defvar: logoutsave
//! If set the preferences will always be saved for this module when the user logout. This is useful for migrating the preferences from other method like IMAP to this one
//!  type: TYPE_FLAG
//!  name: Always save preferences on logout
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

