
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "brightoninternals.h"

static int hex2num(char);
static int xpmchar2num(brightonBitmap *,char);
static int convertindex(brightonBitmap *, char *, int);
static int convertcolor(char *);

#define BUFSIZE 8192

#define CTAB_SIZE 256

brightonBitmap *
xpmread(brightonWindow *bwin, char *filename)
{
	int color, total, width = 0, height = 0, colors = 0, bpcolor = 0, i = 1, j;
	int innerstatic = -1, outerstatic = -1;
	FILE *fd;
	char line[BUFSIZE];
	int *colormap;
	brightonBitmap *bitmap;

	if ((fd = fopen(filename, "r")) <= 0)
		return(0);

	//printf("xpmread(%s)\n", filename);

	while ((fgets(line, BUFSIZE, fd)) != 0)
	{
		/*
		 * We are looking for numbers:
		 */
		if (!isdigit(line[1]))
			continue;
		else {
//printf("read line %s\n", line);
			/*
			 * Should be able to get width, height, colors and planes from this
			 * line of text.
			 */

			while (isdigit(line[i]))
				width = width * 10 + line[i++] - '0';
			if (line[i++] != ' ')
				return(0);

			while (isdigit(line[i]))
				height = height * 10 + line[i++] - '0';
			if (line[i++] != ' ')
				return(0);

			while (isdigit(line[i]))
				colors = colors * 10 + line[i++] - '0';
			if (line[i++] != ' ')
				return(0);

			while (isdigit(line[i]))
				bpcolor = bpcolor * 10 + line[i++] - '0';

			if (line[i] != '"')
			{
				while (line[i] == ' ')
					i++;
				innerstatic = 0;

				while (isdigit(line[i]))
					innerstatic = innerstatic * 10 + line[i++] - '0';

				if (line[i] != '"')
				{
					while (line[i] == ' ')
						i++;
					outerstatic = 0;

					while (isdigit(line[i]))
						outerstatic = outerstatic * 10 + line[i++] - '0';
				}

				if (line[i] != '"')
					return(0);
			}

			break;
		}
	}
//printf("found %i %i %i %i, malloc %i, (%i,%i)\n",
//width, height, colors, bpcolor, width * height, innerstatic, outerstatic);

	bitmap = brightonCreateBitmap(bwin, width, height);
	bitmap->pixels = (int *) brightonmalloc(width * height * sizeof(int));
	bitmap->width = width;
	bitmap->height = height;
	bitmap->ncolors = colors;
	bitmap->ctabsize = colors;
	bitmap->uses = 0;

	bitmap->istatic = innerstatic;
	if (outerstatic == -1)
		bitmap->ostatic = width > height? height / 2: width /2;
	else
		bitmap->ostatic = outerstatic;

	colormap = (int *) brightonmalloc(colors * sizeof(int));
	if (bitmap->colormap)
		brightonfree(bitmap->colormap);
	bitmap->colormap = colormap;

	/*
	 * We now have some reasonable w, h, c and p. Need to parse c color lines.
	 * We have to build a table of the character indeces from the XPM file. To
	 * do this we need a character table and index. Rather than define them in
	 * this file I am going to re-use some parts of the bitmap structure.
	 */
	bitmap->name = brightonmalloc(CTAB_SIZE * sizeof(char));
	for (i = 0; i < colors;i++)
	{
		short r, g, b;

		if (fgets(line, BUFSIZE, fd) == 0)
		{
//printf("short read\n");
			return(bitmap);
		}

//printf("read line %s%s\n", line, &line[4 + bpcolor]);
		if (((line[bpcolor + 1] != '\t') && (line[bpcolor + 1] != ' '))
			|| (line[bpcolor + 2] != 'c'))
		{
//printf("bad parse\n");
			return(bitmap);
		}

		if (strncmp("None", &line[4 + bpcolor], 4) == 0)
		{
			color = convertindex(bitmap, &line[1], bpcolor);
			colormap[i] = brightonGetGCByName(bwin, "Blue");
			continue;
		}

		/*
		 * We need to make a new convertindex that builds its own index table
		 * based on the characters in the xpm file.
		 */
		color = convertindex(bitmap, &line[1], bpcolor);
//printf("color index is %i of %i\n", color, i);

/*
		if (color != i)
		{
//printf("bad match\n");
			return(bitmap);
		}
*/

		if ((color = convertcolor(&line[bpcolor + 4])) < 0)
		{
			line[strlen(line) - 3] = '\0';
			colormap[i] = brightonGetGCByName(bwin, &line[bpcolor + 4]);
		} else {
			/*
			 * Call the color manager to get a GC with this color for eventual
			 * rendering.
			 *
			 * XPM has color from 0 to 255. getGC wants 16 bit values.
			 */
			r = (color >> 16) * 256;
			g = ((color >> 8) & 0xff) * 256;
			b = (color & 0x00ff) * 256;
			/*
			 * Brighton color manager is going to find this color, and return
			 * it if found, return a new GC if there is space, or a best match
			 * if not.
			 */
			colormap[i] = brightonGetGC(bwin, r, g, b);
		}
		//printf("color %i is %x\n", i, color);
	}
//printf("done colors\n");
	for (i = 0; i < height;i++)
	{
		if (fgets(line, BUFSIZE, fd) == 0)
		{
			return(bitmap);
		}
		if (line[0] != '\"')
			continue;
		for (j = 0; j < width * bpcolor; j+=bpcolor)
		{
			color = convertindex(bitmap, &line[j + 1], bpcolor);
			if ((color < 0) || (color > colors))
			{
//printf("too many colors\n");
				return(bitmap);
			} else {
				//printf("color at %i is %i, %x\n",
				//	i * width + j / bpcolor, color, colormap[color]);
				bitmap->pixels[i * width + j / bpcolor] = colormap[color];
			}
		}
	}

	brightonfree(bitmap->name);

	bitmap->name = (char *) brightonmalloc(strlen(filename) + 1);
	sprintf(bitmap->name, "%s", filename);

//	bitmap->colormap = colormap;

	bitmap->uses = 1;

//printf("done xpmread of %s\n", bitmap->name);
	return(bitmap);
}

static int
hex2num(char c)
{
	if (isxdigit(c))
	{
		switch (c) {
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
				return(c - '0');
			case 'a':
			case 'b':
			case 'c':
			case 'd':
			case 'e':
			case 'f':
				return(c - 'a' + 10);
			default:
				return(c - 'A' + 10);
		}
	}
	return(-1);
}

static int
convertcolor(char *line)
{
	int i = 0, color = 0, digit;

	if (strcmp(line, "none") == 0)
		return(0);
	else if (strcmp(line, "None") == 0)
		return(0);

	if (line[0] != '#')
		return(-1);

	line++;

	while (line[i] != '"')
	{
		if ((digit = hex2num(line[i])) < 0)
			return(0);

		color = color * 16 + digit;
		i++;
	}

	return(color);
}

static int
convertindex(brightonBitmap *bitmap, char *line, int bpc)
{
	int cindex = 0, color = 0, i = 0, j = 1;

	while (i < bpc)
	{
		cindex = cindex + xpmchar2num(bitmap, line[i]) * j;
		j += 91;
		i++;
	}

	return(cindex);
}

int
xpmchar2num(brightonBitmap *bitmap, char c)
{
	int i;

	for (i = 0; i < bitmap->uses; i++)
		if (c == bitmap->name[i])
			return(i);

//printf("", bitmap->uses, i);
	/*
	 * If we got here then the character has not been assigned yet.
	 */
	bitmap->name[bitmap->uses] = (int) c;
	bitmap->uses++;
}

writeLine(int fd, char *line)
{
	return(write(fd, line, strlen(line)));
}

#define SAVE_IMAGE
#ifdef SAVE_IMAGE

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

brightonXpmWrite(brightonWindow *bwin, char *file)
{
	int fd, x, y, ncolors, bpc, color, lindex;
	int colors[256], cindex = 0, ccount = 0;
	long points[bwin->width][bwin->height];
	char line[(bwin->width + 10)];
	char cstring[16], filename[64];

	sprintf(filename, "/tmp/%s.xpm", bwin->template->name);

	if ((fd = open(filename, O_WRONLY|O_CREAT|O_TRUNC, 0770)) < 0)
		return;

	writeLine(fd, "/* XPM */\n");
	writeLine(fd, "static char * brighton_xpm[] = {\n");

	/*
	 * We need to scan the image to see how many colors it implements. From this
	 * we should build a character table, and then we should scan through the
	 * image printing these characters.
	 */
	for (y = 0; y < bwin->render->height; y++)
	{
		for (x = 0; x < bwin->render->width; x++)
		{
			color = bwin->render->pixels[x + y * bwin->render->width];

			for (cindex = 0; cindex < ccount; cindex++)
			{
				if (color == colors[cindex])
				{
					points[x][y] = cindex;
					break;
				}
			}
			if (cindex == ccount)
			{
				colors[cindex] = color;
				ccount++;
			}
			points[x][y] = cindex;
		}
	}

	//printf("Found %i colors\n", ccount);

	sprintf(line, "\"%i %i %i %i\"\n", bwin->width, bwin->height, ccount, 1);
	writeLine(fd, line);

	for (cindex = 0; cindex < ccount; cindex++)
	{
		brightonSprintColor(bwin, cstring, colors[cindex]);
		sprintf(line, "\"%c	c %s\"\n", cindex + 35, cstring);
		writeLine(fd, line);
	}

	for (y = 0; y < bwin->height; y++)
	{
		lindex = 1;
		sprintf(line, "\"");
		for (x = 0; x < bwin->width; x++)
		{
			sprintf(&line[lindex], "%c", points[x][y] + 35);
			lindex++;
		}
		sprintf(&line[lindex], "\"\n");
		writeLine(fd, line);
	}

	writeLine(fd, "};\n\n");

	close(fd);
printf("Image written to %s, used %i colors\n", filename, ccount);
}
#endif /* SAVE_IMAGE */

