<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    wml-defs.php                                            */
/* Author:      Paul Waite                                              */
/* Description: Definitions for WAP WML generation                      */
/*                                                                      */
/* ******************************************************************** */
/** @package wml */

/** Button widgets */
include_once("button-defs.php");
/** Form elements */
include_once("form-defs.php");

//-----------------------------------------------------------------------
/**
* WMLelement class
* Manage WML elements. This is a virtual class used for all WML elements.
* @package wml
*/
class WMLelement extends RenderableObject {
  /** The content of the element */
  var $body = "";
  // ....................................................................
  /**
  * Constructor
  * Create a new element object.
  * @param string $body Initial element body content
  */
  function WMLelement($body="") {
    $this->body = $body;
  } // WMLelement
  // ....................................................................
  /**
  * Insert content
  * Append more WML content to the body. This just concatenates
  * the given string onto the existing content.
  * @param string $wml WML string to append
  */
  function insert($content) {
    $this->body .= $content;
    return $this;
  } // insert
  // ....................................................................
  /**
  * Insert paragraph
  * Append content to the body inside a paragraph.
  * @param string $wml WML string to append inside paragraph tags
  */
  function insert_paragraph($content) {
    $this->insert("<p>$content</p>");
    return $this;
  } // insert_paragraph
  // ....................................................................
  /**
  * Insert paragraph small text
  * Append content to the body inside a paragraph and with small text
  * tags around it. Refinement of the standard insert_paragraph() method.
  * @param string $wml WML string to append inside paragraph tags as small text
  */
  function insert_para($content) {
    $this->insert_paragraph("<small>$content</small>");
    return $this;
  } // insert_para
  // ....................................................................
  /**
  * Insert paragraph small text scrubbed
  * Append content to the body inside a paragraph and with small text
  * tags around it. Just like insert_para() except we also scrub the
  * content to make sure it is clean.
  * @param string $wml WML string to append inside paragraph tags as small text and scrubbed
  */
  function insert_para_scrubbed($content) {
    $this->insert_para( scrub($content) );
    return $this;
  } // insert_para_scrubbed
} // WMLelement

//-----------------------------------------------------------------------
/**
* WMLfieldset class
* Manage a set of fields.
* @package wml
*/
class WMLfieldset extends WMLelement {
  /** Title of the set of fields */
  var $title = "";
  /** Array of fields */
  var $fields;
  /** Total number of fields */
  var $fields_total = 0;
  // ....................................................................
  /**
  * Constructor
  * Create a new fieldset object.
  * @param string $title Title of this set of fields
  */
  function WMLfieldset($title="") {
    $this->title = $title;
  } // WMLfieldset
  // ....................................................................
  /**
  * Add field
  * Add a field object to the fields in this fieldset.
  * @param object $field Field to add to the fieldset
  */
  function add_field($field) {
    $this->fields[$field->name] = $field;
    $this->fields_total += 1;
    return $this;
  } // add_field
  // ....................................................................
  /**
  * Return WML content
  * Use render() to render this element in your page.
  * Returns a string which is the WML for the fieldset. We render the
  * fieldset tags, and we iterate over the fields to fill in the
  * content of the fieldset.
  * NOTE: The field which is added is a field of a standard type such
  * as form_textfield, form_combofield etc. @see form-defs.php.
  * @return string The WML for the element.
  */
  function wml() {
    $wml = "<fieldset";
    if ($this->title != "") $wml .= " title=\"" . $this->title . "\"";
    $wml .= ">";
    reset($this->fields);
    while (list($fieldname, $fieldobj) = each($this->fields)) {
      $wml .= $fieldobj->wml() . "<br/>";
    }
    $wml .= "</fieldset>";
    return $wml;
  } // wml
  // ....................................................................
  /**
  * Return HTML content
  * Use render() to render this element in your page.
  * NOTE; For html this is meaningless. Just use the BACK button.
  * @return string The HTML for the element.
  */
  function html() {
    $form = new form($this->title);
    reset($this->fields);
    while (list($fieldname, $fieldobj) = each($this->fields)) {
      $form->add($fieldobj);
    }
    $submit = new submit_button("Submit");
    $form->add_button($submit);
    return $form->render();
  } // html
} // WMLfieldset

//-----------------------------------------------------------------------
/**
* WMLdo_element class
* Do Element Class: Extends element to make a virtual class which
* has descendants which are contained by a 'do' construct.
* A Do element is essentially a kind of clickable link which does
* something when the user clicks on it. What it does depends on
* "type" of the DO. This virtual class is inherited by child
* classes which implement the various different types of DO.
* @package wml
*/
class WMLdo_element extends WMLelement {
  /** Type of DO element: 'accept', 'prev', 'help', 'reset', 'options', or 'delete' */
  var $type;
  /** Name of the DO element */
  var $name = "";
  /** Label for the Do element */
  var $label = "";
  /** True if optional */
  var $optional = false;
  // ....................................................................
  /**
  * Constructor
  * Create a new do element object.
  * @param string $type        Type of do element.
  * @param string $name        Name of this Do element
  * @param string $label       Label for this DO element
  * @param bool   $optional    True if DO element is optional
  * @param string $ontimerhref URL for ontimer event to redirect to
  */
  function WMLdo_element($type, $name, $label="", $optional=false, $ontimerhref="") {
    $this->WMLelement("");
    $this->type = $type;
    $this->name = $name;
    $this->label = chop($label);
    $this->optional = $optional;
    $this->ontimerhref = $ontimerhref;
  } // WMLdo_element
  // ....................................................................
  /**
  * Produce main WML for DO element
  * This is a common utility method to render the DO WML once the
  * details (body) has been filled in by the child class of this one.
  * @return string WML for the DO element
  */
  function wml() {
    $wml = "";
    $wml .= "<do type=\"" . $this->type . "\" name=\"" . $this->name . "\"";
    if ($this->label != "") $wml .= " label=\"" . $this->label . "\"";
    if ($this->optional) $wml .= " optional=\"true\"";
    $timerhref = href_rewrite($this->ontimerhref);
    if ($this->ontimerhref != "") $wml .= " ontimer=\"$timerhref\"";
    $wml .= ">";
    $wml .= $this->body;
    $wml .= "</do>";
    return $wml;
  } // wml
  // ....................................................................
  /**
  * Produce main HTML for DO element
  * @return string HTML for the DO element
  */
  function html() {
    return $this->body;
  } // html
} // WMLdo_element
//-----------------------------------------------------------------------
/**
* WMLprev class
* Extends the Do Element Class. Provide the Do element of type = 'prev'.
* @package wml
*/
class WMLprev extends WMLdo_element {
  /**
  * Constructor
  * Create a new prev element object.
  * @param string $label  Label for this PREV element
  * @param string $body   Body content
  */
  function WMLprev($label, $body="") {
    $this->WMLdo_element("prev", "prev", $label);
    if ($body != "") $this-insert($body);
  } // WMLprev
  // ....................................................................
  /**
  * Return WML content
  * Use render() to render this element in your page.
  * @return string The WML for the element.
  */
  function wml() {
    global $RESPONSE;
    $wml = "";
    $wml .= "<prev";
    if ($this->body != "" || isset($RESPONSE->session_id)) {
      $wml .= ">" . $this->body;
      if (isset($RESPONSE->session_id)) {
        $cookie = $RESPONSE->cookiename;
        $sessid = $RESPONSE->session_id;
        $wml .= "<setvar name=\"$cookie\" value=\"$sessid\"/>";
      }
      $wml .= "</prev>";
    }
    else $wml .= "/>";

    // Make this our body, and render using the
    // parent 'do' wml rendering routine..
    $this->body = $wml;
    return WMLdo_element::wml();
  } // wml
  // ....................................................................
  /**
  * Return HTML content
  * Use render() to render this element in your page.
  * NOTE; For html this is meaningless. Just use the BACK button.
  * @return string The HTML for the element.
  */
  function html() {
    $html = "";
    return $html;
  } // html
} // WMLprev

//-----------------------------------------------------------------------
/**
* WMLrefresh class
* Extends the Do Element Class. Provide the Do element of type = 'refresh'.
* @package wml
*/
class WMLrefresh extends WMLdo_element {
  /**
  * Constructor
  * Create a new refresh DO element object.
  * @param string $label  Label for this DO element
  * @param string $body   Body content
  */
  function WMLrefresh($label, $body="") {
    $this->WMLdo_element("refresh", "refresh", $label);
    if ($body != "") $this-insert($body);
  } // WMLrefresh
  // ....................................................................
  /**
  * Return WML content
  * Use render() to render this element in your page.
  * @return string The WML for the element.
  */
  function wml() {
    global $RESPONSE;
    $wml = "";
    $wml .= "<refresh";
    if ($this->body != "" || isset($RESPONSE->session_id)) {
      $wml .= ">" . $this->body;
      if (isset($RESPONSE->session_id)) {
        $cookie = $RESPONSE->cookiename;
        $sessid = $RESPONSE->session_id;
        $wml .= "<setvar name=\"$cookie\" value=\"$sessid\"/>";
      }
      $wml .= "</refresh>";
    }
    else $wml .= "/>";

    // Make this our body, and render using the
    // parent 'do' wml rendering routine..
    $this->body = $wml;
    return WMLdo_element::wml();
  } // wml
  // ....................................................................
  /**
  * Return HTML content
  * Use render() to render this element in your page.
  * NOTE; For html this is meaningless. Just use the REFRESH button.
  * @return string The HTML for the element.
  */
  function html() {
    $html = "";
    return $html;
  } // html
} // WMLrefresh

//-----------------------------------------------------------------------
/**
* WMLgo class
* Extends the Do Element Class. Provide the Do element of type = 'accept'.
* This is like the HTML submit button which delivers form content back
* to the server after the user has entered details.
* @package wml
*/
class WMLgo extends WMLdo_element {
  /** URL to GO to */
  var $href = "";
  /** Method, eg. "post" */
  var $method;
  /** Array of fields to post */
  var $postfields;
  // ....................................................................
  /**
  * Constructor
  * Create a new go DO element object.
  * @param string $name   Name for this GO element
  * @param string $label  Label for this GO element
  * @param string $href   URL to go to when clicked by user
  * @param string $method Method, eg. "post"
  */
  function WMLgo($name, $label, $href, $method="") {
    $this->WMLdo_element("accept", $name, $label);
    $this->href = $href;
    $this->method = chop($method);
  } // WMLgo
  // ....................................................................
  /**
  * Define a field to post with this GO element. When fields are
  * defined like this the method is automatically switched to
  * being "post" for the GO element.
  * @param string $name   Name of the field to post
  * @param string $value  Value in the field to post
  */
  function postfield($name, $value="") {
    if ($value == "field_value") $value = "\$" . $name;
    $this->body .= "<postfield name=\"$name\" value=\"$value\"/>";
    $this->postfields[$name] = $value;
    $this->method = "post";
    return $this;
  } // postfield
  // ....................................................................
  /**
  * Return WML content
  * Use render() to render this element in your page.
  * @return string The WML for the element.
  */
  function wml() {
    $wml = "";
    $href = href_rewrite($this->href);
    $wml .= "<go href=\"$href\"";
    if ($this->method != "") $wml .= " method=\"" . $this->method . "\"";
    $wml .= ">\n";
    $wml .= $this->body;
    $wml .= "</go>";

    // Make this our body, and render using the
    // parent 'do' wml rendering routine..
    $this->body = $wml;
    return WMLdo_element::wml();
  } // wml
  // ....................................................................
  /**
  * Return HTML content
  * Use render() to render this element in your page.
  * @return string The HTML for the element.
  */
  function html() {
    $html = "";
    $href = $this->href;
    if (isset($this->postfields) && count($this->postfields) > 0) {
      if (!strstr($href, "?")) {
        $href .= "?";
      }
      $qstr = "";
      while (list($fieldname, $fieldval) = each($this->postfields)) {
        if ($qstr != "") $qstr .= "&";
        $qstr .= "$fieldname=$fieldval";
      }
      $qstr = urlencode($qstr);
      $href .= $qstr;
    }
    $golink = new Link($href, $this->label);
    $hml .= "<p>" . $golink->render() . "</p>";
    return $html;
  } // html
} // WMLgo

//-----------------------------------------------------------------------
/**
* WMLanchor class
* Provides a standard link on a WML page. It is usually an underlined
* bit of text which the user can click on to go to the given URL.
* This object contains a GO element to do the work of jumping to
* the given URL.
* @package wml
*/
class WMLanchor extends WMLelement {
  /** Text to display for the anchor/link */
  var $text;
  /** Title of the anchor */
  var $title;
  /** Go object associated with this link */
  var $go;
  // ....................................................................
  /**
  * Constructor
  * Create a new anchor (clickable link)
  * @param string $href   URL to go to when clicked by user
  * @param string $text   Text to display to the user
  * @param string $title  Title for the anchor
  */
  function WMLanchor($href="", $text="Go", $title="") {
    $this->WMLelement("");
    $go = new WMLgo($href);
    $this->text = $text;
    $this->title = $title;
  } // WMLanchor
  // ....................................................................
  /**
  * Insert content into the anchor
  * @param string $wml Content to append to the anchor
  */
  function insert($wml) {
    $this->go = $this->go->insert($wml);
    return $this;
  } // insert
  // ....................................................................
  /**
  * Define a variable
  * This variable will be sent to the server when the user clicks
  * the link.
  * @param string $varname Name of the variable
  * @param string $value   Value of the variable
  */
  function set_variable($varname, $value) {
    $this->insert("<setvar name=\"$varname\" value=\"$value\"/>");
    return $this;
  } // set_variable
  // ....................................................................
  /**
  * Return WML content
  * Use render() to render this element in your page.
  * @return string The WML for the element.
  */
  function wml() {
    $wml = "";
    $wml .= "<anchor ";
    if ($this->title != "") $wml .= " title=\"" . $this->title . "\"";
    $wml .= ">";
    $wml .= $this->go->wml();
    $wml .= "</anchor>";
    return $wml;
  } // wml
  // ....................................................................
  /**
  * Return HTML content
  * Use render() to render this element in your page.
  * @return string The HTML for the element.
  */
  function html() {
    $html = $this->go->html();
    return $html;
  } // html
} // WMLanchor

//-----------------------------------------------------------------------
/**
* WMLcontainer class
* Container Class: Extends element class to make a virtual class for
* descendants which are container of basic elements. Examples of these
* would be a WMLtemplate, or a WMLcard.
* @package wml
*/
class WMLcontainer extends WMLelement {
  /** URL to go to when timer expires */
  var $ontimerhref;
  /** Timer expiry interval (in 10ths of seconds) */
  var $timerval;
  // ....................................................................
  /**
  * Constructor
  * Create a new container object.
  * @param string $body  Initial body content
  */
  function WMLcontainer($body="") {
    $this->WMLelement($body);
    $this->ontimerhref = "";
  } // WMLcontainer
  // ....................................................................
  /**
  * Activate a timer
  * Sets up the parameters for a timer.
  * @param string  $href  URL to go to on timer expiry
  * @param integer $secs  Timer expiry time, in seconds
  */
  function set_ontimer($href, $secs=1) {
    $this->ontimerhref = $href;
    $this->timerval = $secs * 10;
  } // set_ontimer
  // ....................................................................
  /**
  * Insert GO element into container
  * @param string $name   Name for this GO element
  * @param string $label  Label for this GO element
  * @param string $href   URL to go to when clicked by user
  */
  function insert_go($name, $label, $href) {
    $d = new WMLgo($name, $label, $href);
    $this->insert($d->render());
  } // insert_go
  // ....................................................................
  /**
  * Insert PREV element into container
  * @param string $label  Label for this PREV element
  */
  function insert_prev($label) {
    $d = new WMLprev($label);
    $this->insert($d->render());
  } // insert_prev
  // ....................................................................
  /**
  * Insert REFRESH element into container
  * @param string $label  Label for this REFRESH element
  */
  function insert_refresh($label) {
    $d = new WMLrefresh($label);
    $this->insert($d->render());
  } // insert_refresh
} // WMLcontainer

//-----------------------------------------------------------------------
/**
* WMLtemplate class
* Extends container class to make a template. These are useful when
* you have a bunch of cards which you want to all have a common
* set of options.
* @package wml
*/
class WMLtemplate extends WMLcontainer {
  /**
  * Constructor
  * Create a new template object.
  * @param string $body  Initial body content
  */
  function WMLtemplate($body="") {
    $this->WMLcontainer($body);
  } // WMLtemplate
  // ....................................................................
  /**
  * Return WML content
  * Use render() to render this element in your page.
  * Returns a string which is the WML for the template.
  * @return string The WML for the element.
  */
  function wml() {
    $wml = "";
    $wml .= "<template";
    $timerhref = href_rewrite($this->ontimerhref);
    if ($timerhref != "") $wml .= " ontimer=\"$timerhref\"";
    $wml .= ">";
    if ($timerhref != "") $wml .= "<timer value=\"$this->timerval\"/>";
    $wml .= $this->body;
    $wml .= "</template>";
    return $wml;
  } // wml
  // ....................................................................
  /**
  * Return HTML content
  * Use render() to render this element in your page.
  * @return string The HTML for the element.
  */
  function html() {
    return $this->body;
  } // html
} // WMLtemplate

//-----------------------------------------------------------------------
/**
* WMLcard class
* Manage WML cards.
* These are effectively the "webpages" of the WAP world.
* @package wml
*/
class WMLcard extends WMLcontainer {
  /** Unique card identifier string */
  var $card_id = "";
  /** Card title/banner */
  var $title = "";
  /** True if card has its own context */
  var $newcontext = false;
  /** True if the card is "ordered" */
  var $ordered = false;
  // ....................................................................
  /**
  * Constructor
  * Create a new WML card.
  * @param string $card_id    The unique card identifier string
  * @param string $title      The card title string
  * @param string $newcontext True if card has a new context when shown
  * @param string $ordered    True if card is ordered
  */
  function WMLcard($card_id="main", $title="", $newcontext=false, $ordered=false) {
    $this->WMLcontainer("");
    $this->card_id = $card_id;
    $this->title = chop($title);
    $this->newcontext = $newcontext;
    $this->ordered = $ordered;
    $this->ontimerhref = "";
  } // WMLcard
  // ....................................................................
  /**
  * Return WML content
  * Use render() to render this element in your page.
  * Returns a string which is the WML for the card. This is basically
  * just a case of rendering the various properties like card_id etc.
  * as they were defined for the card.
  * @return string The WML for the element.
  */
  function wml() {
    global $RESPONSE;
    // Deal with OpenWave phones not showing a card title..
    if (isset($RESPONSE)
      && $RESPONSE->browser_type == BROWSER_TYPE_WMLUP
      && $this->title != "") {
      $s = "<p><small><b>" . $this->title . "</b></small></p>";
      $this->body = $s . $this->body;
    }
    $wml = "";
    $timerhref = href_rewrite($this->ontimerhref);
    $wml .= "<card";
    if ($this->title != "")   $wml .= " title=\"$this->title\"";
    if ($this->card_id != "") $wml .= " id=\"$this->card_id\"";
    if ($timerhref != "")     $wml .= " ontimer=\"$timerhref\"";
    if ($this->newcontext)    $wml .= " newcontext=\"true\"";
    if ($this->ordered)       $wml .= " ordered=\"true\"";
    $wml .= ">";
    if ($timerhref != "")     $wml .= "<timer value=\"$this->timerval\"/>";
    $wml .= $this->body;
    $wml .= "</card>";
    return $wml;
  } // wml
  // ....................................................................
  /**
  * Return HTML content
  * Use render() to render this element in your page.
  * @return string The HTML for the element.
  */
  function html() {
    $html = "";
    if ($this->title != "") {
      $html .= "<h3>$this->title</h3>";
    }
    if ($this->ontimerhref != "") {
      $timerlink = new Link($this->ontimerhref, "$this->ontimerhref (" . $this->timerval/10 . " secs)");
      $hml .= "<p>" . $timerlink->render() . "</p>";
    }
    $html .= $this->body;
    return $html;
  } // html
} // WMLcard

// ----------------------------------------------------------------------
/**
* WMLDeck class
* Manage WML Decks of cards.
* The main element in WML content - the deck which
* holds all the cards.
* @package wml
*/
class WMLdeck extends RenderableObject {
  /** Optional template */
  var $template;
  /** Array of card objects */
  var $cards;
  /** Length of content in bytes */
  var $wml_len = 0;
  // ....................................................................
  /**
  * Constructor
  * Create a new WML deck.
  * @param object $card A card to put in the deck
  */
  function WMLdeck($card="") {
    if ($card != "") {
      $this->insert_card($card);
    }
  } // WMLdeck
  // ....................................................................
  /**
  * Length (size) of deck
  * Return the length of the WML. NOTE: this should only be called
  * *after* the wml() function has been called, or it will return zero.
  * @return integer The size of the deck of WML
  */
  function length() {
    return $this->wml_len;
  } // length
  // ....................................................................
  /**
  * Calculate the length (size) of deck
  * Calculates the WML length from scratch.
  * @return integer The calculated size of the deck of WML
  */
  function calculate_length() {
    $this->wml_len = strlen($this->wml());
    return $this->wml_len;
  } // calculate_length
  // ....................................................................
  /**
  * Define template
  * defines the template for the deck.
  * @param object $template The template object for the WML deck
  */
  function insert_template($template) {
    if (is_object($template)) {
      $this->template = $template;
    }
    return $this;
  } // insert_template
  // ....................................................................
  /**
  * Add card
  * Inserts a ready-made card object into the deck.
  * @param object $card The card object to add to the WML deck
  */
  function insert_card($card) {
    $this->cards[$card->card_id] = $card;
    return $this;
  } // insert_card
  // ....................................................................
  /**
  * New card
  * Creates a card and inserts the new card object into the deck.
  * @param string $card_id    The unique card identifier string
  * @param string $title      The card title string
  * @param string $newcontext True if card has a new context when shown
  * @param string $ordered    True if card is ordered
  */
  function new_card($card_id, $title="", $newcontext=false, $ordered=false) {
    $this->cards[$card_id] = new WMLcard($card_id, $title, $newcontext, $ordered);
    return $this;
  } // new_card
  // ....................................................................
  /**
  * Return complete WML response content including DOCTYPE, <wml> tags,
  * and the content-length header.
  * @return string The complete WML for this deck
  */
  function response() {
    $s = "";
    $s .= "<?xml version=\"1.0\"?>\n";
    $s .= "<!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" ";
    $s .= "\"http://www.wapforum.org/DTD/wml_1.1.xml\">\n";
    $s .= "<wml>";
    $s .= $this->wml();
    $s .= "</wml>";

    // Set the length..
    $this->wml_len = strlen($wml);

    // Output the header for wml length..
    Header("Content-Length: " . $this->wml_len);
    return $s;
  } // response
  // ....................................................................
  /**
  * Return WML content for this deck.
  * Use render() to render this element in your page.
  * Returns a string which is the WML for the deck. First we render any
  * template and then we iterate over all the cards defined for the deck.
  * @return string The WML for the deck of cards complete.
  */
  function wml() {
    $s = "";
    if (isset($this->template)) {
      $s .= $this->template->wml();
    }
    foreach($this->cards as $card) {
      $s .= $card->wml();
    }
    return $s;
  } // wml
  // ....................................................................
  /**
  * Return HTML content
  * Use render() to render this element in your page.
  * Returns a string which is the HTML for the deck.
  * @return string The HTML for the element.
  */
  function html() {
    // Define a webpage..
    $page = new webpage("WML Viewer");
    // Render all cards..
    foreach ($this->cards as $card) {
      $page->add( $card->html() );
    }
    // Return the content of the whole page..
    return $page->render();
  } // html
} // WMLdeck

// ----------------------------------------------------------------------
/**
* Scrubs the given string to make it acceptable to touchy WML parsers
* in cellphones.
* @param string $s String to scrub
* @return string Scrubbed version of given string
*/
function scrub($s) {
  global $RESPONSE;
  $res = strip_tags($s);
  $res = ereg_replace( "[\r\n ]+", " ", $res);
  $res = ereg_replace( "&([#0-9a-z]+);", "aMpErSaNd-\\1;", $res);
  $res = ereg_replace( "&", "&amp;", $res);
  $res = ereg_replace( "aMpErSaNd-", "&", $res);
  if ($RESPONSE->browser == BROWSER_PHONE) {
    $res = ereg_replace( "&copy;", "(c)", $res);
    $res = ereg_replace( "[$]", "$$", $res);
  }
  return $res;
} // scrub

// ----------------------------------------------------------------------
/**
* Re-writes the given HREF, appending session ID if found. Appends the
* session properly when pre-existing query parameters are already on
* the href string. Omits the session id if the string contains the
* pattern "logoff", since this is assumed to mean that the session is
* being terminated. Also omits it if already present.
* @param string $href HREF to rewrite
* @return string Re-written HREF
*/
function href_rewrite($href) {
  global $RESPONSE;
  if ($href != "" && isset($RESPONSE) && isset($RESPONSE->session_id)) {
    $cookie = $RESPONSE->cookiename;
    $sessid = $RESPONSE->session_id;
    if (($cookie != "")
      && !stristr($href, $cookie)
      && !stristr($href, "logoff")) {
      if ((strtolower(substr($href,0,4)) != "http") && (substr($href,0,1) != "#")) {
        if (strstr($href, "?")) $href .= "&amp;$cookie=$sessid";
        else $href .= "?$cookie=$sessid";
      }
    }
  }
  return $href;
} // href_rewrite
// ----------------------------------------------------------------------
?>