/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import nz.net.catalyst.ELog;
import nz.net.catalyst.Log;
import nz.net.catalyst.RtException;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.IndexWriter;


/**
 * This class manages access to the Lucene IndexWriter.  Only one
 * IndexWriter can be opened at a time.  If an attempt is made to open
 * another, an IOException results.  <p>
 *
 * This class will issue an IndexWriter with the required Analyzer.
 * If the Analyzer is the same as on a previous call, then the it is
 * possible that an already-open IndexWriter will be retuned.  <p>
 *
 * If an open IndexWriter remains idle for a time, then it will be
 * closed.
 */

public class WriterControl implements IPackage
{
  /**
   * System property that determines how frequently the Writer monitor
   * will wake up and check for locks held too long.  The property name is:
   *
   * <pre>  nz.net.catalyst.lucene.server.WriterCheck </pre>
   *
   * The value is the number of seconds between checks.
   */
  private static final String WRITER_CHECK_PROPERTY = PACKAGE + "WriterCheck";

  /**
   * Default number of seconds between checks for Writer Locks being
   * held too long. Value is 15.
   */
  private static final int DEFAULT_WRITER_CHECK = 15;

  /**
   * System property that determines how long an idle IndexWriter open
   * in the hope that another user will want to Index a document with
   * it.  Holding it open means that it does not flush its current
   * updates to disk.  The property name is:
   *
   * <pre>  nz.net.catalyst.lucene.server.CloseWait </pre>
   *
   * The value is the number of seconds an IndexWriter can remain idle
   * before being closed.
   */
  private static final String CLOSE_WAIT_PROPERTY = PACKAGE + "CloseWait";

  /**
   * Default number of seconds befoer closing an idle
   * IndexWriter. Value is 30.
   */
  private static final int DEFAULT_CLOSE_WAIT = 30;

  /**
   * System property that determines how frequently the Writer monitor
   * will wake up and attempt to open an IndexWriter if it is being
   * locked by another process.  The property name is:
   *
   * <pre>  nz.net.catalyst.lucene.server.LockWait </pre>
   *
   * The value is the number of seconds between checks.
   */
  private static final String WAIT_SECONDS_PROPERTY = PACKAGE + "LockWait";

  /**
   * Default number of seconds between attempts to open the
   * IndexWriter if it is locked by another process. Value is 5.
   */
  private static final int DEFAULT_WAIT_SECONDS = 5;

  /**
   * For each index directory, contains the WriterControl object
   * which is controlling access to its IndexWriter.  <p>
   *
   * Key is File(directory), Value is WriterControl
   */

  private static final Map writers = new HashMap();

  static
  {
    startMonitor();
  }

  // The directory containing the index.  This will be unique across all
  // Writer Control instances.
  private final File directory;

  // The Lucene IndexWriter.  Will be null if there is not an open one
  // for this directory.
  private IndexWriter indexWriter;

  // The thread currently using the IndexWriter.  May be non-null when
  // indexWriter is null if the last user was forced off and the
  // IndexWriter was force-closed.  If null, the IndexWriter is idle.
  private Thread user;

  // Last activity on the IndexWriter (either getIndexWriter() or
  // release() ).
  private long timeStamp;  // time last issued or released.

  // Analyzer associated with the IndexWriter.  Any request for an
  // IndexWriter must have the same Analyzer as was used for the open
  // IndexWriter, otherwise the IndexWriter is closed and a new one
  // opened.
  private Analyzer analyzer;

  // The number of threads whcih are waiting to use the IndexWriter.
  private int waiters = 0;

  // Set up Control for a particular directory.
  private WriterControl(File directory)
  {
    this.directory = directory;
  }

  private static WriterControl getInstance(File directory) throws IOException
  {
    if (!directory.exists() && !directory.mkdirs())
      throw new IOException("Unable to create directory: " + directory);

    directory = directory.getCanonicalFile();

    synchronized(writers)
    {
      WriterControl writerControl = (WriterControl)writers.get(directory);
      if (writerControl == null)
      {
        writerControl = new WriterControl(directory);
        writers.put(directory, writerControl);
      }
      return writerControl;
    }
  }

  /**
   * Obtain an WriterControl instance for use with a particular
   * directory and Analyzer.  The instance returned will be contain an
   * IndexWriter which is open and ready for use. <p>
   *
   * If this thread had previously called closeWriter() with the same
   * directory and with the lock flag set, then the same WriterControl
   * instance will be returned.
   *
   * @param directory The directory to contain the index.  Will be
   *                  created if it does not exist (if possible).
   * @param analyzer  The Lucene Analyser to be passed to the
   *                  IndexWriter.
   *
   * @return A WriterControl containing an open IndexWriter.
   *
   * @throws IOException if the directory was invalid or the
   *                     IndexWriter could not be opened
   */

  static WriterControl getWriterControl(File directory,Analyzer analyzer)
    throws IOException
  {
    return getInstance(directory).getWriterControl(analyzer);
  }

  /**
   * Make sure that there is no open IndexWriter in the directory as
   * we are just about to execute a command against a Reader that
   * needs exclusive access.
   *
   * @param directory The directory to contain the index.  Will be
   *                  created if it does not exist (if possible).
   *
   * @param lock Whether to keep the lock on the IndexWriter for later opening.
   *
   * @throws IOException If the IndexWriter failed to close (or there
   * is a problem with the directory.
   */

  static WriterControl closeWriter(File directory, boolean lock)
    throws IOException
  {
    return getInstance(directory).closeWriter(lock);
  }


  /**
   * We're just about to execute a query.  Make sure that any idle IndexWriter
   * is closed so that we get the latest documents.
   *
   * @param directory The directory to contain the index.  Will be
   *                  created if it does not exist (if possible).
   *
   * @throws IOException If the IndexWriter failed to close (or there
   * is a problem with the directory.
   */

  static void closeIdleWriter(File directory)
    throws IOException
  {
    getInstance(directory).closeIdleWriter();
  }

  /**
   * Suspend the thread until no other thread has locked the
   * IndexWriter.  This is determined by checking the value of the
   * <user> variable.
   */

  private void waitForWriter(String debugMessage) throws IOException
  {
    Thread myThread = Thread.currentThread();
    long before = System.currentTimeMillis();
    boolean waited = false;

    if (user != myThread)
    {
      try
      {
        ++waiters;

        while (user != null)
        {
          waited = true;

          //  Someone else is using it: We're going to have to wait.
          try
          {
            wait();
          }
          catch (InterruptedException e)
          {
            throw new IOException(e.toString());
          }
        }
      }
      finally
      {
        --waiters;
      }
    }
    if (waited)
    {
      long duration = System.currentTimeMillis() - before;
      Log.debug(myThread.getName() + " waited " + duration +
                " ms to " + debugMessage);
    }
  }

  IndexWriter getIndexWriter()
  {
    return indexWriter;
  }

  /**
   * Obtain a WriterControl instance containing an IndexWriter
   * instance for use with a particular Analyzer.  The IndexWriter
   * will be open and ready for use.
   *
   * @param analyzer  The Lucene Analyser to be passed to the
   *                  IndexWriter.
   *
   * @return A WriterControl containing an open IndexWriter
   *
   * @throws IOException if the IndexWriter could not be opened
   */

  private synchronized WriterControl getWriterControl(Analyzer analyzer)
    throws IOException
  {
    waitForWriter("get IndexWriter");

    // Prevent anyone else from stealing IndexWriter if we're forced
    // to wait for another process to release it.

    user = Thread.currentThread();

    // Our turn to use the IndexWriter

    try
    {
      if (indexWriter == null)
      {
        // Easy: There is no currently defined IndexWriter for this directory:
        indexWriter = openIndex(directory, analyzer);
        Log.debug("Opened new IndexWriter in " + directory);
        this.analyzer = analyzer;
      }
      else if (this.analyzer != analyzer)
      {
        // If the Analyzer is different, we must reopen the IndexWriter.
        indexWriter.close();
        // In case open fails, clear out old value explicitly.
        indexWriter = null;
        indexWriter = openIndex(directory, analyzer);
        Log.debug("Closed and reopened IndexWriter in " + directory);
        this.analyzer = analyzer;
      }
      else
        Log.debug("Reused IndexWriter in " + directory);

      timeStamp = System.currentTimeMillis();
      return this;
    }
    finally
    {
      // If we didn't, in the end, succeed in obtaining the
      // IndexWriter, then relinquish our lock.

      if (indexWriter == null)
        user = null;
    }
  }

  /**
   * Make sure that there is no open IndexWriter in the directory as
   * we are just about to execute a command against a Reader that
   * needs exclusive access.
   *
   * @param lock Whether to keep the lock on the IndexWriter for later opening.
   *
   * @throws IOException If the IndexWriter failed to close
   */
  private synchronized WriterControl closeWriter(boolean lock)
    throws IOException
  {
    waitForWriter("close IndexWriter");

    // Our turn to manipulate the IndexWriter

    if (indexWriter != null)
    {
      indexWriter.close();
      indexWriter = null;
      timeStamp = System.currentTimeMillis();
      Log.debug("Closed IndexWriter in " + directory);

    }

    if (lock)
    {
      user = Thread.currentThread();
      return this;
    }
    user = null;
    notifyAll();
    return null;
  }

  /**
   * We're just about to execute a query.  Make sure that any idle IndexWriter
   * is closed so that we get the latest documents.
   *
   * @throws IOException If the IndexWriter failed to close
   */

  synchronized void closeIdleWriter() throws IOException
  {
    if (indexWriter != null)
    {
      if (user == null)
      {
        try
        {
          indexWriter.close();
          Log.debug("Closed idle IndexWriter in " + directory);
        }
        catch (IOException e)
        {
          e.printStackTrace();
        }
        indexWriter = null;
      }
      else
        Log.debug("IndexWriter not idle in " + directory);
    }
  }

  /**
   * This method will attempt to open an IndexWriter.  If it fails due
   * to a Lock failure, then it will retry for a little while.  Since
   * this class is serializing all access to IndexWriters within this
   * process, then a Lock Failure should only happen if another Lucene
   * process is accessing the same Index. <p>
   *
   * This method will either succeed in opening the IndexWriter or
   * throw an IOException
   *
   * @param create Whether to open the index in CREATE mode or not.
   *
   * @throws IOException if the IndexWriter could not be opened.
   */

  private IndexWriter openIndex(File directory, Analyzer analyzer)
    throws IOException
  {
    int waitSeconds = Integer.getInteger(WAIT_SECONDS_PROPERTY,
                                         DEFAULT_WAIT_SECONDS).intValue();
    if (waitSeconds < 0)
      waitSeconds = DEFAULT_WAIT_SECONDS;

    long endTime = System.currentTimeMillis() + waitSeconds * 1000L;

  openLoop:
    for(;;)
    {
      try
      {
        try
        {
          // Try to open an existing IndexWriter.
          IndexWriter indexWriter = new IndexWriter(directory, analyzer, false);
          Log.debug("Opened IndexWriter in: " + directory);
          return indexWriter;
        }
        catch (FileNotFoundException e)
        {
          // This seems to mean that the Index doesn't exist yet.
          // Open a new one.
          Log.log(ELog.NOTICE, "Failed to open existing Lucene index in: " +
                  directory);
          IndexWriter indexWriter = new IndexWriter(directory, analyzer, true);
          Log.debug("Created IndexWriter in: " + directory);
          return indexWriter;
        }
      }
      catch (IOException e)
      {
        Log.log(ELog.ERROR, "Failed to open Lucene index in: " +
                           directory + ": " + e.toString());
        String error = e.getMessage();
        if (error.startsWith("Index locked for write"))
        {
          // Another process has the Index Writer...

          long now = System.currentTimeMillis();
          if (now >= endTime)
            throw e;   // Give up -- waited too long!

          // Wait a moment, and try to lock again.  We relinquish
          // synchronized lock while waiting but no other thread can
          // take away the IndexWriter while we're sleeping.
          long sleepTime = Math.min(200L, endTime - now);
          try
          {
            wait(sleepTime);
          }
          catch (InterruptedException ie)
          {
            // Ignore!
          }
        }
        else
          throw e;  // Some other error
      }
    }
  }

  /**
   * Release an IndexWriter after use.  This method <i>MUST</i> be
   * called in order for other threads to have access to the
   * IndexWriter as well.  <p>
   *
   * Only the thread which obtained the IndexWriter may release it.
   * Any attempts to release it by other threads will throw an
   * exception <p>
   *
   * (Sigh) There's no point in holding open the IndexWriter because
   * we <i>always</i> need to close it in order to attempt to delete
   * the next insert!
   *
   * @param indexWriter the IndexWriter allocated to the current
   * thread by a previous call to {@link #getIndexWriter()}
   */

  synchronized void release()
  {
    if (user != Thread.currentThread())
      throw new RtException("Attempt to release an IndexWriter not owned by this thread (probably previously released)");

    user = null;
    if (indexWriter != null)
    {
      try
      {
        indexWriter.close();
        Log.debug("Closed IndexWriter in " + directory);
      }
      catch (IOException e)
      {
        e.printStackTrace();
      }
      indexWriter = null;
    }

    timeStamp = System.currentTimeMillis();
    notifyAll();
  }

  /**
   * Check whether any idle IndexWriters should be closed.  Also check
   * that they haven't been held in use for too long.
   *
   * @param now The current time
   * @param idleTime The permitted idle time limit
   * @param useLimit The permitted use duration.
   */

  private synchronized void check(long now, long idleTime, long useLimit)
  {
    if (waiters > 0)
    {
      Log.debug("There " + (waiters == 1
                            ? "is 1 waiter"
                            : "are " + waiters + " waiters") +
                " for " + directory + " and " +
                (indexWriter == null ? "no" : "an") +
                " IndexWriter and " + (user == null ? "no" : "one") +
                " user");
    }

    if (indexWriter == null)
      return;

    if (user == null)
    {
      if (timeStamp + idleTime <= now)
      {
        // IndexWrite has been idle too long.  Close it.
        Log.debug("Closing IndexWriter in: " + directory);
        try
        {
          indexWriter.close();
        }
        catch (IOException e)
        {
          e.printStackTrace();
        }
        indexWriter = null;
      }

      return;
    }

    if (timeStamp + useLimit <= now)
    {
      // Somebody has been hanging on to the IndexWriter for too long.
      // Try and force them off.  This will get messy: If they are
      // stuck in a synchronized IndexWriter method call, we will
      // block (so use a sub-thread).  If another user comes along and
      // tries to open an IndexWriter before the close succeeds, they
      // will get rebuffed because of the IndexWriter lock.  If the
      // IndexWriter won't close, we're screwed!

	  // wr#31987 - No longer force the close of overused IndexWriters as
	  //            this has been the probable cause of index corruption
	  //            due to lengthy optimize operations.  Optimizes are done
	  //            periodically during indexing as well as on demand.
	  
	  java.util.Date useStart = new java.util.Date(timeStamp);
	  
	  Log.warn("useLimit exceeded for IndexWriter");
	  Log.warn(" UserThread: " + user.getName());
	  Log.warn(" Directory:  " + directory.toString());
	  Log.warn(" useLimit:   " + useLimit + " ms");
	  Log.warn(" useStart:   " + useStart);	  
	  Log.warn(" Probable long optimize command, worse-case: runaway thread.");
	  
/*	   
      final IndexWriter temp = indexWriter;
      indexWriter = null; // This makes it unused!
      user = null;
      Thread closer = new Thread(){
          public void run()
          {
            try
            {
              temp.close();
              Log.debug("Force-close succeeded of IndexWriter: " + directory);
            }
            catch (IOException e)
            {
              Log.log(ELog.ERROR, "Force-close failed of IndexWriter: " + directory + ":");
              e.printStackTrace();
            }
          }
        };
      closer.setDaemon(true);
      closer.start();
      notifyAll();
*/      
    }
  }

  private static void startMonitor()
  {

    Thread monitor = new Thread(new Runnable() {
        public void run(){monitor();}
      }, "WriterControl");
    monitor.setDaemon(true);

    // Use reduced priority for polling loop.
    int priority = monitor.getPriority() - 1;
    if (priority < Thread.MIN_PRIORITY)
      ++priority;
    monitor.setPriority(priority);
    monitor.start();
  }

  private static void checkWriters(long useLimit)
  {
    WriterControl[] wc;

    synchronized (writers)
    {
      wc = new WriterControl[writers.size()];
      writers.values().toArray(wc);
    }

    long idleTime = 1000L * Integer.getInteger(CLOSE_WAIT_PROPERTY,
                                               DEFAULT_CLOSE_WAIT).intValue();
    long now = System.currentTimeMillis();

    for (int i = 0; i < wc.length; ++i)
      wc[i].check(now, idleTime, useLimit);
  }

  private static void monitor()
  {
    try
    {
      for(;;)
      {
        int sleepSeconds = Integer.getInteger(WRITER_CHECK_PROPERTY,
                                              DEFAULT_WRITER_CHECK).intValue();
        if (sleepSeconds < 1)
          sleepSeconds = 1;

        long sleepTime = sleepSeconds * 1000L;

        Thread.sleep(sleepTime);
        checkWriters(sleepTime);
      }
    }
    catch(InterruptedException e)
    {
      // Don't know how this happened, but I'm going to kill this thread!
    }
    finally
    {
      // If thread dies for any reason, start a new instance.
      startMonitor();
    }
  }

  public static void Start(String[] args)
  {
    // Nothing to do -- static initialiser does it.
  }
}
