/* Segment.h, Uros Platise Dec 1998 */

#ifndef __Segment
#define __Segment

#include <vector>
#include "Global.h"
#ifdef WIN32
using namespace std;
#endif

#define SEG_ROOT	0

#define SEG_NOTPLACED	-1
#define SEG_UNKNOWNSIZE	-1
#define SEG_UNKNOWNALIGN -1

#define SEG_MAXNAMELEN	32
#define SEG_LABELLEN	256
#define MAX_SEGMENTS	1024

/* All symbols that are defined as extern and have no prototype in
   current source, should point to segment 0 due to incRef func. */
#define SEGNUMBER_UNDEFINED 0


/*
  Segment Translation Table
  
  Different object files have different segment. 
  Linker's job is to synchronize all objects throught
  the Segment Translation Table.
*/
class TSegTable{
private:
  int CRef;
  struct TSegTableRec{
    int newNo;		/* new segment number */
    long rel;		/* relative offset per file per segment */
    TSegTableRec():newNo(0),rel(0){}
  };  
public:
  TSegTableRec seg[MAX_SEGMENTS];
  TSegTable(){};
  ~TSegTable(){};
  friend TPt<TSegTable>;
};
typedef TPt<TSegTable> PSegTable;


/* Segment Dependencies

   If a label declared in segment A is called in segment B then
   segment B depends on segment A. That is why segment A must
   not be removed.
*/   
class TSegDep{
private:
  static const int LOG_BITPR; 	/* log of bits per record -- 
                                           this may vary in the future */
  static const int NAMES_PER_LINE;	/* Used by Print() function */			      
  typedef unsigned int TSegBitMap;

/* Array Declaration - Same to all Objects */
  TSegBitMap* segbitmap;
  int max_segs;
  int max_byte;
  
/* Diagnostic */
  int count;			/* number of 1s in segbitmap array */
  int last_byte;		/* last used byte in the segbitmap */
  bool mapped;
  
private:
  int CRef;
public:
  friend TPt<TSegDep>;
  
public:
  TSegDep(int _max_segs = MAX_SEGMENTS):
    count(0),last_byte(0),mapped(false),CRef(0){
    max_byte = (max_segs = _max_segs)>>LOG_BITPR;
    segbitmap = new TSegBitMap[max_byte];
    for (int i=0;i<max_byte;i++){segbitmap[i]=0;}	/* clear array */
  }
  ~TSegDep(){delete[] segbitmap;}
  
  void Add(int seg_no);
  void Remove(int seg_no);
    /* level defines the text shift from the left border */
  void Print(int level) const; 
  void Merge(const TSegDep &src);
  void MapSegments(const TSegTable& segtbl);
  void SaveObj();
  void LoadObj();
  void LoadObj(const TSegTable& segtbl);
  
  bool Empty() const {return count==0;}	/* says true if segment is abs. free */
  int Count() const {return count;}	/* counts number of deps */
  bool Mapped() const {return mapped;}
  int LastByte() const {return last_byte;}
};

typedef TPt<TSegDep> PSegDep;	/* Smart Pointer Class */


/* Segment Record

   This structure provide all necessary segment properties.
   Segment tree is built upon array of these records.
*/
struct TSegmentRec{  
  enum TFlags  {None=0,Removable=1,Abstract=2};
  enum TStatus {Ok=0,OutofRange=1,Collision=2,Removed=4};
  enum TMirror {NoMirror=0, NoCompression, RLE1};

  char name [SEG_MAXNAMELEN];           /* original name of the segment */
  int  mirrorSegNo;		 	/* mirror segment */  
  char label [SEG_LABELLEN];            /* label used for PC */
  
  vector< TSegmentRec* > segRef;        /* segment reference are childs! */
  TSegmentRec* parent;			/* parent */  
  
  long size;                            /* maximum segment size */
  long PC;                              /* running counter and tmp size */
  long sumSize;                         /* size of this and all sub segs. */
  long abs;                             /* when placed, this value is set.*/
  int align;				/* alignment (default to one byte) */
    
  TFlags flags;                         /* segment attributes */
  TStatus status;			/* segment status */
  TMirror mirrorType;			/* mirror segments may be compressed */
  bool update_mirror_no;		/* if true, mirror updates needs to be
  					   updated after loading all segments
					   per file. */
  TSegDep dep;
					    
  /* note: abs (addr) and size are assigned -1 value to inform linker
     that these two values are to be calculated! */  
  TSegmentRec(){clear();}
      
  void clear(){
    parent=NULL; flags=None; status=Ok;
    size=SEG_UNKNOWNSIZE; abs=SEG_NOTPLACED; mirrorType=NoMirror;    
    PC=sumSize=0;
    align=SEG_UNKNOWNALIGN;
    name[0]=label[0]=0;     
    update_mirror_no=false;
  }
   
  /* sort refs by ABSOLUTE number */
  void sortRefs();
  inline long start(){return abs+PC;}
  inline long end(){return abs+sumSize;}
};
  

/*
  Segment Sort Record
  
  Used by fitter only to ease some operations.
*/
struct TSSRec{
  TSegmentRec* ref;
  long start,size;

  TSSRec():ref(NULL),start(0){}  
  TSSRec(long _size, TSegmentRec* _ref):ref(_ref),size(_size){}
  TSSRec(long _start, long _end):start(_start){size=_end-start;}

  inline long end(){return start+size;}
  inline bool operator<(const TSSRec& ssrec){return size<ssrec.size;}
};    
  

/*
  Class Segment
  
  Core of the segment mechanism.
*/
class TSegment{
private:
  typedef vector< TSegmentRec* >::const_iterator TsegRecCI;
  typedef vector< TSegmentRec* >::iterator TsegRecI;

  TSegmentRec segRec [MAX_SEGMENTS];  /* segment entries */
  int segUsageCnt;                    /* segment usage counter */
  TSegmentRec* cseg;		      /* current segment in use */
  bool fitted;
  bool errorActive;
  
  void close();			      /* close current segment */
  bool set(const char* name);         /* set new segment (ret: true if new)*/
  void setPrevious();		      /* trace back */
  void update(const TSegmentRec& tmpSeg);
  long parseValue();
  void parseSegmentName(bool mirror=false);
  void removeUnused();
  void findSum(TSegmentRec* segp, int level=0);    
                                      /* stores info in .sumSize record */
  void CalcMirrors();		      /* calculate mirror sizes */
  void CopyMirrorChildren(TSegmentRec* p);
  void fit(TSegmentRec* segp);
  void saveSegmentTree(TSegmentRec* p);
  void ReportSegmentTree(TSegmentRec* p, int level);
  void ReportOverall(TSegmentRec* p);  
  
public:
  TSegment();
  ~TSegment(){}
  
  bool parse();				/* return true, if sth was done */  
  void fitter();
  void adjustSegments();
  void MapSegmentDependencies();
  void saveSegments();
  void loadSegment(int segNo, bool end_of_load = false);
  void Report();
  
  char* getPC(char* PC_str);		/* returns pointer to PC_str */
  bool isEnabled(int segNo);		/* true if segment is active */
  bool isAbstract(int segNo);
    
  void incPC(long relative);  
  
  int getSegNo(){return cseg-segRec;}	/* returns current segment number */
  void SetSeg(int seg_no); 		/* sets active segment (pass2 only) */
  void AddDep(int seg_no, int current_seg_no);
  void MergeDep(int seg_no, const TSegDep& src);
  
  const char* TellBaseSegment(int segNo) const;
  int   GetRootsSegments(unsigned child) const;
  int   TellNoSegments() const;		    
  long  TellSize(int segNo) const {return segRec[segNo].PC;}
  long  TellAbsolute(int segNo) const {return segRec[segNo].abs;}
  const char* TellSegmentName(int segNo) const;
  int   TellMirror(int segNo) const;
  int   TellAlign(int segNo) const;
};
extern TSegment segment;

#endif

