/*

*************************************************************************

ArmageTron -- Just another Tron Lightcycle Game in 3D.
Copyright (C) 2000  Manuel Moos (manuel@moosnet.de)

**************************************************************************

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
***************************************************************************

*/

#ifndef ArmageTron_PLAYER_H
#define ArmageTron_PLAYER_H

#define MAX_INSTANT_CHAT 12
#define MAX_PLAYERS 4

#include "uInput.h"
#include "tList.h"
#include "tString.h"
#include "eCamera.h"
#include "eNetGameObject.h"
#include "tCallbackString.h"
#include "nSpamProtection.h"

#define PLAYER_CONFITEMS 40

extern signed short int thetopscore;
extern unsigned short int hudfpscount;
extern unsigned short int alivepeople;

class tConfItemBase;
class uAction;
class tOutput;
class eTeam;
class eVoter;

class ePlayer: public uPlayerPrototype{
	friend class eMenuItemChat;
	static uActionPlayer s_chat;

	tConfItemBase *configuration[PLAYER_CONFITEMS];
	int            CurrentConfitem;
	void   StoreConfitem(tConfItemBase *c);
	void   DeleteConfitems();
 public:
	tString    name;
	REAL	   rubberstatus;
	bool       centerIncamOnTurn;
	bool       wobbleIncam;
	bool       autoSwitchIncam;
	bool       spectate;

	bool 		nameTeamAfterMe; // player prefers to call his team after his name
	int			favoriteNumberOfPlayersPerTeam;

	eCamMode startCamera;
	bool     allowCam[10];
	int      startFOV;

	tCHECKED_PTR(eCamera)           cam;
	tCONTROLLED_PTR(ePlayerNetID) netPlayer;

	int rgb[3]; // our color

	tString instantChatString[MAX_INSTANT_CHAT];
	// instant chat macros

	static uActionPlayer *se_instantChatAction[MAX_INSTANT_CHAT];

	ePlayer();
	virtual ~ePlayer();

	virtual const char *Name() const{return name;}
 
	virtual bool Act(uAction *act,REAL x);

	int ID(){return id;};
#ifndef DEDICATED
	void Render();
#endif
  
	static ePlayer * PlayerConfig(int p);

	static bool PlayerIsInGame(int p);

	static rViewport * PlayerViewport(int p);

	static void Init();
	static void Exit();
};

// the class that identifies players across the network
class ePlayerNetID: public nNetObject{
	friend class ePlayer;
	friend class eTeam;
	friend class eNetGameObject;
	friend class tControlledPTR< ePlayerNetID >;

	int listID;                          // ID in the list of all players
	int teamListID;                      // ID in the list of the team

	bool							silenced_;		// flag indicating whether the player has been silenced

	REAL							timeJoinedTeam; // the time the player joined the team he is in now
	tCONTROLLED_PTR(eTeam)			nextTeam;		// the team we're in ( logically )
	tCONTROLLED_PTR(eTeam)			currentTeam;	// the team we currently are spawned for
	tCONTROLLED_PTR(eVoter)			voter_;			// voter assigned to this player

	tCHECKED_PTR(eNetGameObject) object; // the object this player is 
	// controlling

	int score; // points made so far

	int favoriteNumberOfPlayersPerTeam;		// join team if number of players on it is less than this; create new team otherwise
	bool nameTeamAfterMe; 					// player prefers to call his team after his name
	bool auth;            					// is this user valid?
	bool greeted;        					// did the server already greet him?
	bool disconnected;   					// did he disconnect from the game?
  
	static void SwapPlayersNo(int a,int b); // swaps the players a and b

	ePlayerNetID& operator= (const ePlayerNetID&); // forbid copy constructor

	bool			chatting_; // are we currently chatting?
	int				chatFlags_; // different types of reasons for beeing chatting
 public:
	enum			ChatFlags
		{
			ChatFlags_Chat = 1,
			ChatFlags_Away = 2,
			ChatFlags_Menu = 4,
			ChatFlags_Console = 4
		};

	int    pID;
	tString name;
	REAL	rubberstatus;
	tArray<tString> lastSaid;
	tArray<double> lastSaidTimes;
//	void SetLastSaid(tString ls);
	unsigned short r,g,b; // our color

	unsigned short pingCharity; // max ping you are willing to take over

	REAL ping;

	double lastSync;

	nSpamProtection chatSpam_;

	ePlayerNetID(int p=-1);
	ePlayerNetID(nMessage &m);
	virtual ~ePlayerNetID();

	virtual bool ActionOnQuit();
	virtual void ActionOnDelete();

	// chatting
	bool IsChatting() { return chatting_; }
	void SetChatting ( ChatFlags flag, bool chatting );
	
	// team management
	eTeam* NextTeam()    const { return nextTeam; }				// return the team I will be next round
	eTeam* CurrentTeam() const { return currentTeam; }		// return the team I am in
	void FindDefaultTeam();									// look for a good default team for us
	void SetTeamForce(eTeam* team );                 	// register me in the given team without checks
	void SetTeam(eTeam* team);          	// register me in the given team (callable on the server)
	void SetTeamWish(eTeam* team); 				// express the wish to be part of the given team (always callable)
	void UpdateTeamForce();							// update team membership without checks
	void UpdateTeam();							// update team membership

	void CreateNewTeam(); 	    				// create a new team and join it (on the server)
	void CreateNewTeamWish();	 				// express the wish to create a new team and join it
	virtual void ReceiveControlNet(nMessage &m);// receive the team control wish

  
	// print out an understandable name in to s
	virtual void 			PrintName(tString &s) const;

	virtual bool 			AcceptClientSync() const;
	virtual void 			WriteSync(nMessage &m);
	virtual void 			ReadSync(nMessage &m);
	virtual nDescriptor&	CreatorDescriptor() const;
	virtual void			InitAfterCreation();
	virtual bool			ClearToTransmit(int user) const;
	
	virtual void 			NewObject(){}        				// called when we control a new object
	virtual void 			RightBeforeDeath(int triesLeft){} 	// is called right before the vehicle gets destroyed.


	void RemoveFromGame();

	void ControlObject(eNetGameObject *c);
	void ClearObject();

	void Greet();
	void Auth(); 										// make the authentification valid
	bool IsAuth() const; 								// is the authentification valid?
	bool IsActive() const { return !disconnected; }

	bool IsSilenced( void ) const { return silenced_; }
	void SetSilenced( bool silenced ) { silenced_ = silenced; }
	bool& AccessSilenced( void ) { return silenced_; }

	void CreateVoter();						// create our voter or find it
	static void SilenceMenu();				// menu where you can silence players
	static void PoliceMenu();				// menu where you can silence and kick players

	virtual bool IsHuman() const { return true; }

	void Activity(); // call it if this player just showed some activity.

	eNetGameObject *Object() const;

	void SetRubber(REAL rubber2);
	void AddScore(int points, const tOutput& reasonwin, const tOutput& reasonloose);
	int Score()const {return score;}
	int TotalScore() const;

	static void SortByScore(); // brings the players into the right order
	static tString Ranking( int MAX=12, bool cut = true );     // returns a ranking list
	static void  ResetScore();  // resets the ranking list

	static void DisplayScores(); // display scores on the screen

	void GreetHighscores(tString &s); // tell him his positions in the 
	// highscore lists (defined in game.cpp)

	static void Update();           // creates ePlayerNetIDs for new players 
	// and destroys those of players that have left

	static void CompleteRebuild(); // same as above, but rebuilds every ePlayerNetID.

	static void ThrowOutDisconnected(); // get rid of everyone that disconnected from the game

	void GetScoreFromDisconnectedCopy(); // get the player's data from the previous login

	void Chat(const tString &s);

	virtual void Color( REAL&r, REAL&g, REAL&b ) const;
	virtual void TrailColor( REAL&r, REAL&g, REAL&b ) const;

private:
	void			MyInitAfterCreation();

			// private:
//	virtual void AddRef();
//	virtual void Release();
};

extern tList<ePlayerNetID> se_PlayerNetIDs;
extern int    sr_viewportBelongsToPlayer[MAX_VIEWPORTS];

void se_ChatState( ePlayerNetID::ChatFlags flag, bool cs);

void se_SaveToScoreFile(const tOutput &o);

tString & operator << (tString &s,const ePlayer &p);
tString & operator << (tString &s,const ePlayerNetID &p);

extern int pingCharity;

void se_AutoShowScores(); // show scores based on automated decision
void se_UserShowScores(bool show); // show scores based on user input
void se_SetShowScoresAuto(bool a); // disable/enable auto show scores

//Password stuff
void se_DeletePasswords();
extern int se_PasswordStorageMode; // 0: store in memory, -1: don't store, 1: store on file

tOutput& operator << (tOutput& o, const ePlayerNetID& p);

// greeting callback
class eCallbackGreeting: public tCallbackString
{
	static eCallbackGreeting *anchor;
	static ePlayerNetID* greeted;

 public:
	static tString Greet(ePlayerNetID* player);
	static ePlayerNetID* Greeted(){return greeted;}

	eCallbackGreeting(STRINGRETFUNC* f);
};

extern int se_SpamMaxLen;	// maximal length of chat message

#endif

