//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it
//  and/or modify it under the terms of the GNU General
//  Public License as published by the Free Software
//  Foundation; either version 2 of the License, or
//  (at your option) any later version.
//==============================================

#ifndef BACKEND_SUBALBUM_H
#define BACKEND_SUBALBUM_H

//define representative image sizes
#define SMALL 1
#define MEDIUM 2
#define LARGE 3

//--------------------
//forward declarations
class QString;
class QPixmap;
class QDomNode;
class Album;
class Photo;
class StatusWidget;
class PhotoPreviewWidget;
//--------------------

//=====================================
/*! \brief A subalbum contains photos.
 *
 *  Specific contents:
 * <ul>
 *  <li>Photos
 *  <li>Name
 *  <li>Description
 *  <li>Representative image
 *  <li>Subalbum number
 *  <li>Number of Photos
 * </ul>
 */
//=====================================

class Subalbum
{
//------------------------------------------------------
public:

  ///Sets default information
  ///@number is the Subalbum number
  Subalbum(Album* albm, int number);

  ///Frees photos
  ~Subalbum();

  ///Sets the Subalbum Name
  void setName(QString val);

  ///Gets the Subalbum Name
  QString getName();

  ///Sets the Subalbum description
  void setDescription(QString val);

  ///Gets the Subalbum description
  QString getDescription();

  ///gets a sized representative image
  QPixmap* getRepresentativeImage(int size);

  ///sets a sized representative image
  void setRepresentativeImage(QString imageFilename);

  ///Adds a new photo to the Subalbum and appends it to the end, returns
  ///TRUE if successful
  bool addPhoto(QString fileName, bool replaceDescription = false, Photo* newPhoto = NULL);

  ///Lazily adds a new photo to the subalbum without rescaling from scrath, returns TRUE if successful
  bool lazyAddPhoto(QString imageName,
                             QString slideshowName,
                             QString thumbnailName,
                             Photo* newPhoto);

  ///Adds a preexisting photo object to the Subalbum, appending it to the end
  void addPhoto(Photo* newPhoto);

  ///Removes a specified photo without deleting the object
  void photoMoved(Photo* val);

  ///Removes a specified photo
  void removePhoto(Photo* val);

  ///Returns pointer to prev subalbum
  Subalbum* getPrev();
  
  ///Returns pointer to next subalbum
  Subalbum* getNext();

  ///Sets pointer of prev subalbum
  void setPrev(Subalbum* val);
  
  ///Sets pointer of next subalbum
  void setNext(Subalbum* val);

  ///Returns first photo in subalbum
  Photo* getFirst();

  ///Returns last photo in subalbum
  Photo* getLast();

  ///Exports subalbum to xml
  void exportToXML(StatusWidget* status, QTextStream& stream);

  ///Builds subalbum from XML DOM node
  void importFromDisk(QDomNode* root, int subalbumNum,
                                     StatusWidget* status, QString dirName, bool disableCheckPhotoMods);

  ///Syncs photo ordering with front end gui ordering
  void syncPhotoList(PhotoPreviewWidget* item);

  ///Returns subalbum number
  int getSubalbumNumber();

  ///Sets the subalbum number to newVal
  void setSubalbumNumber(int newVal);

  ///Returns the number of photos in the subalbum
  int getNumPhotos();

  ///Returns the number of loaded photos in subalbum
  int getNumLoadedPhotos();

  //Resets the number of loaded photos in the subalbum to the number of photos in the subalbum
  void resetNumLoadedPhotos();

  //Sets the subalbum as modified
  void setModified();
  
  ///returns the album pointer
  Album* getAlbum();
//------------------------------------------------------
private:
  ///Subalbum Number
  int number;

  ///Number of photos in subalbum
  int numPhotos;

  ///Number of photos in subalbum when last loaded
  int loadedPhotos;

  ///Short Name for subalbum
  QString name;

  ///Longer description of subalbum
  QString description;

  //Small, medium, and large representative images
  QPixmap* smallRepresentativeImage;
  QPixmap* mediumRepresentativeImage;
  QPixmap* largeRepresentativeImage;

  ///Pointer to first photo
  Photo* firstPhoto;

  ///Pointer to last photo
  Photo* lastPhoto;

  ///Pointer to prev subalbum
  Subalbum* prevSubalbum;
  
  ///Pointer to next subalbum
  Subalbum* nextSubalbum;

  ///Pointer to album subalbum is in
  Album* albm;
//------------------------------------------------------
};

#endif //BACKEND_SUBALBUM_H
