/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <stdio.h>
#include "stars.h"
#include "gui.h"
#include "glade/glade.h"
#include "coordutils.h"
#include "configure.h"
#include "printgui.h"

int selectorlock = 0;

struct classtablerow
{
	GtkWidget *namelabel, *minspin, *maxspin, *unitslabel;
	GtkAdjustment *minadj, *maxadj;
};

struct classparamvalues
{
	int numext;
	struct classheader *ch;
	float *max, *min, *oldmax, *oldmin;
};

struct selectorinfo
{
	GtkWidget *sel, *classlist, *catlist, *classvis, *catvis;
	GtkWidget *classtext, *cattext, *applyall, *classtable;
	struct viewer *v;
	int *iscatvis, *oldcatvis;
	int *isclassvis, *oldclassvis;
	int numcatrows, numclassrows;
	int catrow, classrow;
	struct classheader *ch;
	GList *tablerows;
	GList *classvalues;
} *si = 0;

void hideselector()
{
	selectorlock = 0;

	si->numcatrows = 0;
	g_free(si->iscatvis);
	g_free(si->oldcatvis);
	si->iscatvis = 0;
	si->oldcatvis = 0;

	gtk_widget_hide(si->sel);
}

void hideviewerselector(struct viewer *v)
{
	if(selectorlock == 0) return;
	if(v == si->v) hideselector();
}

void deleteclassvalues(struct selectorinfo *si)
{
	GList *l;
	struct classparamvalues *cv;

	while((l = si->classvalues) != 0)
	{
		cv = (struct classparamvalues *)(l->data);
		g_free(cv->min);
		g_free(cv->max);
		g_free(cv->oldmin);
		g_free(cv->oldmax);
		g_free(cv);
		si->classvalues = g_list_remove(si->classvalues, cv);
	}

	si->classvalues = 0;
}

void getclassvalues(struct selectorinfo *si, struct classheader *ch)
{
	int i;
	GList *l, *l2;
	struct classparamvalues *cv;
	struct classtablerow *tr;

//	printf("getclassvalues\n");

	for(l = si->classvalues; l != 0; l = l->next)
	{
		cv = (struct classparamvalues *)(l->data);
		if(cv->ch == ch)
		{
			l2 = si->tablerows;
			for(i = 0; i < cv->numext; i++)
			{
				if(l2 == 0)
				{
					printf("Null in getclassvalues\n");
					return;
				}
				tr = (struct classtablerow *)(l2->data);

				cv->min[i] =
					((GtkAdjustment *)(tr->minadj))->value; 
				cv->max[i] =
					((GtkAdjustment *)(tr->maxadj))->value; 

				l2 = l2->next;
			}
			return;
		}
	}

	printf("Problem in getclassvalues!\n");
}

void setclassvalues(struct selectorinfo *si, struct classheader *ch)
{
	int i;
	GList *l, *l2;
	struct classparamvalues *cv;
	struct classtablerow *tr;

	for(l = si->classvalues; l != 0; l = l->next)
	{
		cv = (struct classparamvalues *)(l->data);
		if(cv->ch == ch)
		{
			l2 = si->tablerows;
			for(i = 0; i < cv->numext; i++)
			{
				if(l2 == 0)
				{
					printf("Null in setclassvalues\n");
					return;
				}
				tr = (struct classtablerow *)(l2->data);

				((GtkAdjustment *)(tr->minadj))->value 
					= cv->min[i];
				((GtkAdjustment *)(tr->maxadj))->value 
					= cv->max[i];
				gtk_adjustment_value_changed(
					GTK_ADJUSTMENT(tr->minadj));
				gtk_adjustment_value_changed(
					GTK_ADJUSTMENT(tr->maxadj));

				l2 = l2->next;
			}
			return;
		}
	}

	printf("Problem in setclassvalues!\n");
}

void replaceclassvalues(struct selectorinfo *si)
{
	int i, index, id;
	GList *l2 = 0;
	GList *l;
	struct classheader *ch;
	struct classparamvalues *cv;

	id = si->v->viewerID;

	getclassvalues(si, si->ch);

//	printf("*** replace ***  id = %d\n", id);

	l2 = si->classvalues;
	for(l2 = si->classvalues, l = classlist; l != 0; 
		l = l->next, l2 = l2->next)
	{
		if(l2 == 0)
		{	
			printf("Problem in replaceclassvalues\n");
			return;
		}

		index = 0;

		ch = (struct classheader *)(l->data);
		cv = (struct classparamvalues *)(l2->data);

		if(cv->numext > 0)
		{
			if(ch->hasmag)
			{
//				printf("c: %s, mag limints: %f %f\n",
//					ch->fullname, cv->min[index],
//					cv->max[index]);
				ch->magmin[id] = cv->min[index];
				ch->magmax[id] = cv->max[index];
				index++;
			}

			for(i = 0; i < ch->numext; i++)
			{
				ch->extensions[i].min[id] = cv->min[index];
				ch->extensions[i].max[id] = cv->max[index];
				index++;
			}
		}
	}
}

void retrieveclassvalues(struct selectorinfo *si)
{
	int i, index, id;
	GList *l2 = 0;
	GList *l;
	struct classheader *ch;
	struct classparamvalues *cv;

	if(si->classvalues != 0) deleteclassvalues(si);


	id = si->v->viewerID;

	for(l = classlist; l != 0; l = l->next)
	{
		index = 0;

		ch = (struct classheader *)(l->data);

		cv = g_new(struct classparamvalues, 1);
		cv->numext = ch->numext;
		cv->ch = ch;
		if(ch->hasmag) cv->numext++;

		if(cv->numext > 0)
		{
			cv->min = g_new(float, cv->numext);
			cv->max = g_new(float, cv->numext);
			cv->oldmin = g_new(float, cv->numext);
			cv->oldmax = g_new(float, cv->numext);

			if(ch->hasmag)
			{
				cv->oldmin[index] = cv->min[index] 
						  = ch->magmin[id];
				cv->oldmax[index] = cv->max[index] 
						  = ch->magmax[id];
				index++;
			}

			for(i = 0; i < ch->numext; i++)
			{
				cv->oldmin[index] = cv->min[index] 
						  = ch->extensions[i].min[id];
				cv->oldmax[index] = cv->max[index] 
						  = ch->extensions[i].max[id];
				index++;
			}
		}
		else
		{
			cv->min = cv->max = 0;
			cv->oldmin = cv->oldmax = 0;
		}
		l2 = g_list_append(l2, cv);
	}

	si->classvalues = l2;
}

void setclasstablerow(struct classtablerow *tr, const char *name,
	const char *units, float min, float max)
{
	gtk_label_set_text(GTK_LABEL(tr->namelabel), name);
	gtk_label_set_text(GTK_LABEL(tr->unitslabel), units);
	GTK_ADJUSTMENT(tr->minadj)->lower = -10000.0; // min;
	GTK_ADJUSTMENT(tr->maxadj)->lower = -10000.0; //min;
	GTK_ADJUSTMENT(tr->minadj)->upper = 10000.0; //max;
	GTK_ADJUSTMENT(tr->maxadj)->upper = 10000.0; //max;

	gtk_adjustment_changed(GTK_ADJUSTMENT(tr->minadj));
	gtk_adjustment_changed(GTK_ADJUSTMENT(tr->maxadj));

	gtk_widget_show(tr->namelabel);
	gtk_widget_show(tr->minspin);
	gtk_widget_show(tr->maxspin);
	gtk_widget_show(tr->unitslabel);
}

void fillclasstablerows(struct selectorinfo *si, struct classheader *ch)
{
	int i;
	GList *l;
	struct classtablerow *tr;

	for(l = si->tablerows; l != 0; l = l->next)
	{
		tr = (struct classtablerow *)(l->data);
		gtk_widget_hide(tr->namelabel);
		gtk_widget_hide(tr->minspin);
		gtk_widget_hide(tr->maxspin);
		gtk_widget_hide(tr->unitslabel);
	}
	
	setclassvalues(si, ch);

	l = si->tablerows;

	if(ch->hasmag)
	{
		setclasstablerow((struct classtablerow *)(l->data),
			"Magnitude", "Magnitudes", ch->defmagmin,
			ch->defmagmax);
		l = l->next;
	}

	for(i = 0; i < ch->numext; i++)
	{
		setclasstablerow((struct classtablerow *)(l->data), 
			ch->extensions[i].name,
			ch->extensions[i].units,
			ch->extensions[i].defaultmin,
			ch->extensions[i].defaultmax);
		l = l->next;
	}
}

void fillclassinfo(struct selectorinfo *si, struct viewer *v)
{
	gchar *text[3];
	struct classheader *ch;
	GList *l;
	int classrow = 0;

	retrieveclassvalues(si);

	si->v = v;
	si->ch = 0;

	si->classrow = 0;
	si->numclassrows = g_list_length(classlist);
	si->isclassvis   = g_new(gint, si->numclassrows);
	si->oldclassvis  = g_new(gint, si->numclassrows);
	
	gtk_clist_freeze(GTK_CLIST(si->classlist));
	gtk_clist_clear(GTK_CLIST(si->classlist));
	
	for(l = classlist; l != 0; l = l->next)
	{
		ch = (struct classheader *)(l->data);
		text[0] = g_strdup(ch->fullname);
		text[1] = g_strdup(ch->class);
		if((ch->classmask & si->v->andmask) == si->v->showmask)
		{
			text[2] = g_strdup("YES");
			si->oldclassvis[classrow] = si->isclassvis[classrow] =1;
		}
		else 
		{
			text[2] = g_strdup("NO");
			si->oldclassvis[classrow] = si->isclassvis[classrow] =0;
		}
		
		gtk_clist_append(GTK_CLIST(si->classlist), text);
		
		gtk_clist_set_row_data(GTK_CLIST(si->classlist), 
			classrow++, ch);

		g_free(text[0]);
		g_free(text[1]);
		g_free(text[2]);
	}
	gtk_clist_thaw(GTK_CLIST(si->classlist));
	gtk_clist_select_row(GTK_CLIST(si->classlist), 0, 0);
}

void fillcatinfo(struct selectorinfo *si, struct viewer *v)
{
	gchar *text[3];
	struct catalog *cat;
	GList *l;
	int catrow = 0;

	si->v = v;

	si->catrow = 0;
	si->numcatrows = g_list_length(catalogs);
	si->iscatvis   = g_new(gint, si->numcatrows);
	si->oldcatvis  = g_new(gint, si->numcatrows);

	gtk_clist_freeze(GTK_CLIST(si->catlist));
	gtk_clist_clear(GTK_CLIST(si->catlist));

	for(l = catalogs; l != 0; l = l->next) if(l->data != 0)
	{
		cat = (struct catalog *)(l->data);
		text[0] = g_strdup(cat->name);
		text[1] = g_strdup(cat->type);
		if((cat->catmask & si->v->andmask) == si->v->showmask)
		{
			text[2] = g_strdup("YES");
			si->oldcatvis[catrow] = si->iscatvis[catrow] = 1;
		}
		else 
		{
			text[2] = g_strdup("NO");
			si->oldcatvis[catrow] = si->iscatvis[catrow] = 0;
		}

		gtk_clist_append(GTK_CLIST(si->catlist), text);
		
		gtk_clist_set_row_data(GTK_CLIST(si->catlist), catrow++, cat);

		g_free(text[0]);
		g_free(text[1]);
		g_free(text[2]);
	}
	gtk_clist_thaw(GTK_CLIST(si->catlist));
	gtk_clist_select_row(GTK_CLIST(si->catlist), 0, 0);
}

void refillselectorcatinfo(struct catalog *cat)
{
	int i, catrow;
	gchar *text[3];
	gint *cvis, *ocvis;

	if(!si) return;

	catrow = si->numcatrows;

	si->numcatrows++;

	cvis  = g_new(gint, si->numcatrows);
	ocvis = g_new(gint, si->numcatrows);

	for(i = 0; i < catrow; i++)
	{
		cvis[i]  = si->iscatvis[i];
		ocvis[i] = si->oldcatvis[i];
	}
	g_free(si->iscatvis);
	g_free(si->oldcatvis);
	si->iscatvis  = cvis;
	si->oldcatvis = ocvis;

	text[0] = g_strdup(cat->name);
	text[1] = g_strdup(cat->type);
	if((cat->catmask & si->v->andmask) == si->v->showmask)
	{
		text[2] = g_strdup("YES");
		si->oldcatvis[catrow] = si->iscatvis[catrow] = 1;
	}
	else 
	{
		text[2] = g_strdup("NO");
		si->oldcatvis[catrow] = si->iscatvis[catrow] = 0;
	}

	gtk_clist_append(GTK_CLIST(si->catlist), text);
	
	gtk_clist_set_row_data(GTK_CLIST(si->catlist), catrow, cat);

	g_free(text[0]);
	g_free(text[1]);
	g_free(text[2]);
	
}

void catalog_vis_click(GtkWidget *widget, struct selectorinfo *si)
{
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(si->catvis)))
	{
		si->iscatvis[si->catrow] = 1;
		gtk_clist_set_text(GTK_CLIST(si->catlist), 
			si->catrow, 2, "YES");
	}
	else
	{
		si->iscatvis[si->catrow] = 0;
		gtk_clist_set_text(GTK_CLIST(si->catlist), 
			si->catrow, 2, "NO");
	}
}

void class_vis_click(GtkWidget *widget, struct selectorinfo *si)
{
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(si->classvis)))
	{
		si->isclassvis[si->classrow] = 1;
		gtk_clist_set_text(GTK_CLIST(si->classlist), 
			si->classrow, 2, "YES");
	}
	else
	{
		si->isclassvis[si->classrow] = 0;
		gtk_clist_set_text(GTK_CLIST(si->classlist), 
			si->classrow, 2, "NO");
	}
}

void catalog_row_select(GtkWidget *widget, gint row, gint column,
                             GdkEventButton *event, struct selectorinfo *si)
{
	struct catalog *cat;
	
	if(si == 0) return;
	if(si->iscatvis == 0) return;
	si->catrow = row;
	if(si->iscatvis[row]) 
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(si->catvis),
			TRUE);
	else gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(si->catvis),
			FALSE);

	gtk_text_set_point(GTK_TEXT(si->cattext), 0);
	gtk_text_forward_delete(GTK_TEXT(si->cattext), 
		gtk_text_get_length(GTK_TEXT(si->cattext)));
	cat = (struct catalog *)gtk_clist_get_row_data(
		GTK_CLIST(si->catlist), row);
	if(cat != 0)
		gtk_text_insert(GTK_TEXT(si->cattext), 0, 0, 0, 
			cat->description, -1);

	if(column == 2)
		gtk_signal_emit_by_name(GTK_OBJECT(si->catvis), "clicked", si);
}

void class_row_select(GtkWidget *widget, gint row, gint column,
                             GdkEventButton *event, struct selectorinfo *si)
{
	struct classheader *ch;
	
	if(si == 0) return;
	if(si->isclassvis == 0) return;
	si->classrow = row;

	if(si->ch != 0) getclassvalues(si, si->ch);

	if(si->isclassvis[row]) 
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(si->classvis),
			TRUE);
	else gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(si->classvis),
			FALSE);

	gtk_text_set_point(GTK_TEXT(si->classtext), 0);
	gtk_text_forward_delete(GTK_TEXT(si->classtext), 
		gtk_text_get_length(GTK_TEXT(si->classtext)));
	ch = (struct classheader *)gtk_clist_get_row_data(
		GTK_CLIST(si->classlist), row);

	si->ch = ch;

	if(ch != 0)
	{
		gtk_text_insert(GTK_TEXT(si->classtext), 0, 0, 0, 
			ch->fullname, -1);
		fillclasstablerows(si, ch);
	}
	
	if(column == 2)
		gtk_signal_emit_by_name(GTK_OBJECT(si->classvis), 
			"clicked", si);
}

void try_click(GtkWidget *widget, struct selectorinfo *si)
{
	int i, amask, omask, doall = 0;
	struct catalog *cat;

	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(si->applyall)))
	{
		omask = 0x0F;
		amask = 0xF0;
		doall = 1;
	}
	else
	{
		omask = si->v->showmask;
		amask = si->v->showmask ^ 0xFF;
	}
	
	for(i = 0; i < si->numcatrows; i++)
	{
		cat = (struct catalog *)gtk_clist_get_row_data(
			GTK_CLIST(si->catlist), i);

		if(si->iscatvis[i]) cat->catmask |= omask;
		else cat->catmask &= amask;
	}

	replaceclassvalues(si);

	if(doall) 
	{
		applyclasslimitsall(0);
		paintall();
	}
	else 
	{
		applyclasslimits(0, si->v->viewerID);
		paint(si->v);
	}
}

void revert_click(GtkWidget *widget, struct selectorinfo *si)
{
	int i;

	gtk_clist_freeze(GTK_CLIST(si->catlist));
	for(i = 0; i < si->numcatrows; i++) 
	{
		si->iscatvis[i] = si->oldcatvis[i];
		if(si->oldcatvis[i] == 0)
			gtk_clist_set_text(GTK_CLIST(si->catlist), i, 2, "NO");
		else gtk_clist_set_text(GTK_CLIST(si->catlist), i, 2, "YES");
	}	
	gtk_clist_thaw(GTK_CLIST(si->catlist));
}

void cancel_click(GtkWidget *widget, struct selectorinfo *si)
{
	hideselector();
}

void accept_click(GtkWidget *widget, struct selectorinfo *si)
{
	try_click(widget, si);
	hideselector();
}

struct selectorinfo *getselectorinfo()
{
	int i, trows;
	static GladeXML *selectorxml = 0;
	struct selectorinfo *si;
	struct classtablerow *tr;
	GtkWidget *trybutton, *revertbutton, *cancelbutton, *acceptbutton;
	gfloat climb_rate;
	guint digits;

	if(selectorxml == 0) selectorxml = glade_xml_new(PREFIXDIR "/share/stars/glade/selector.glade", 0);
	if(selectorxml == 0)
	{
		printf("Bad XML!\n");
		return 0;
	}

	si = g_new(struct selectorinfo, 1);
	si->catrow = 0;
	si->iscatvis = 0;
	si->oldcatvis = 0;
	si->tablerows = 0;
	si->classvalues = 0;

	si->sel = glade_xml_get_widget(selectorxml, "Selector");
	si->classlist = glade_xml_get_widget(selectorxml, "classlist");
	si->classtext = glade_xml_get_widget(selectorxml, "classtext");
	si->classvis  = glade_xml_get_widget(selectorxml, "classvis");
	si->catlist   = glade_xml_get_widget(selectorxml, "catlist");
	si->cattext   = glade_xml_get_widget(selectorxml, "cattext");
	si->catvis    = glade_xml_get_widget(selectorxml, "catvis");
	si->applyall  = glade_xml_get_widget(selectorxml, "applyall");
	si->classtable= glade_xml_get_widget(selectorxml, "classtable");

	trybutton    = glade_xml_get_widget(selectorxml, "trybutton");
	revertbutton = glade_xml_get_widget(selectorxml, "revertbutton");
	cancelbutton = glade_xml_get_widget(selectorxml, "cancelbutton");
	acceptbutton = glade_xml_get_widget(selectorxml, "acceptbutton");

	if(si->classlist == 0 || si->classtext == 0 || si->classvis == 0 ||
	   si->catlist == 0 || si->cattext == 0 || si->catvis == 0 ||
	   si->classtable == 0)
	{
		g_free(si);
		return 0;
	}

	if(trybutton == 0 || revertbutton == 0)
	{
		g_free(si);
                return 0;
        }

	trows = maxclassnumext()+2;
	if(trows > GTK_TABLE(si->classtable)->nrows)
		gtk_table_resize(GTK_TABLE(si->classtable), trows, 3);


	tr = g_new(struct classtablerow, 1);
	tr->namelabel  = glade_xml_get_widget(selectorxml, "mlabel");
	tr->minspin    = glade_xml_get_widget(selectorxml, "minspin");
	tr->maxspin    = glade_xml_get_widget(selectorxml, "maxspin");
	tr->unitslabel = glade_xml_get_widget(selectorxml, "ulabel");
	tr->minadj     = GTK_SPIN_BUTTON(tr->minspin)->adjustment;
	tr->maxadj     = GTK_SPIN_BUTTON(tr->maxspin)->adjustment;

	si->tablerows = g_list_append(si->tablerows, tr);

	climb_rate = GTK_SPIN_BUTTON(tr->minspin)->climb_rate;
	digits = GTK_SPIN_BUTTON(tr->minspin)->digits;

	for(i = 0; i < trows - 2; i++)
	{
		tr = g_new(struct classtablerow, 1);

		tr->minadj = (GtkAdjustment *)gtk_adjustment_new(
				1.0, 0.0, 16.0, 0.5, 0.5, 0.5);
		tr->maxadj = (GtkAdjustment *)gtk_adjustment_new(
				13.0, 0.0, 16.0, 0.5, 0.5, 0.5);

		tr->namelabel  = gtk_label_new("empty");
		tr->minspin    = gtk_spin_button_new(tr->minadj,
					climb_rate, digits);
		tr->maxspin    = gtk_spin_button_new(tr->maxadj,
					climb_rate, digits);
		tr->unitslabel = gtk_label_new("empty");

		si->tablerows = g_list_append(si->tablerows, tr);

		gtk_table_attach_defaults(GTK_TABLE(si->classtable),
			tr->namelabel, 0, 1, 2+i, 3+i);
		gtk_table_attach_defaults(GTK_TABLE(si->classtable),
			tr->minspin, 1, 2, 2+i, 3+i);
		gtk_table_attach_defaults(GTK_TABLE(si->classtable),
			tr->maxspin, 2, 3, 2+i, 3+i);
		gtk_table_attach_defaults(GTK_TABLE(si->classtable),
			tr->unitslabel, 3, 4, 2+i, 3+i);

		gtk_widget_show(tr->namelabel);
		gtk_widget_show(tr->minspin);
		gtk_widget_show(tr->maxspin);
		gtk_widget_show(tr->unitslabel);
	}

	gtk_text_set_line_wrap(GTK_TEXT(si->classtext), TRUE);
	gtk_text_set_line_wrap(GTK_TEXT(si->cattext), TRUE);

	gtk_signal_connect(GTK_OBJECT(si->catlist), "select_row",
		GTK_SIGNAL_FUNC(catalog_row_select), si);

	gtk_signal_connect(GTK_OBJECT(si->catvis), "clicked",
		GTK_SIGNAL_FUNC(catalog_vis_click), si);

	gtk_signal_connect(GTK_OBJECT(si->classlist), "select_row",
		GTK_SIGNAL_FUNC(class_row_select), si);

	gtk_signal_connect(GTK_OBJECT(si->classvis), "clicked",
		GTK_SIGNAL_FUNC(class_vis_click), si);

	gtk_signal_connect(GTK_OBJECT(trybutton), "clicked",
		GTK_SIGNAL_FUNC(try_click), si);

	gtk_signal_connect(GTK_OBJECT(revertbutton), "clicked",
		GTK_SIGNAL_FUNC(revert_click), si);

	gtk_signal_connect(GTK_OBJECT(cancelbutton), "clicked",
		GTK_SIGNAL_FUNC(cancel_click), si);

	gtk_signal_connect(GTK_OBJECT(si->sel), "delete_event",
		GTK_SIGNAL_FUNC(cancel_click), si);

	gtk_signal_connect(GTK_OBJECT(acceptbutton), "clicked",
		GTK_SIGNAL_FUNC(accept_click), si);

	return si;
}

/* This is the wrong way to do it */
void objectselector(GtkWidget *widget, struct viewer *v)
{
	if(catalogs == 0)
	{
		printf("No catalogs!\n");
		return;
	}	

	if(selectorlock == 1) return;
	selectorlock = 1;
	if(si == 0) si = getselectorinfo(v);
	if(si == 0)
	{
		printf("Bad XML file\n");
		return;
	}
	
	fillcatinfo(si, v);
	fillclassinfo(si, v);

	gtk_widget_show(si->sel);
}

/* This is the right way */
void configure(GtkWidget *widget, struct viewer *v)
{
	showconfigwindow(v);
}

void zoom_in(GtkObject *obj)
{
	struct viewer *v = (struct viewer *)obj;

	setzoom(v, v->view->zoom*1.5);
}

void zoom_out(GtkObject *obj)
{
	struct viewer *v = (struct viewer *)obj;
	double z = v->view->zoom/1.5;

	if(z < 1.0) z = 1.0;
	
	setzoom(v, z);
}

void zoom_1(GtkObject *obj)
{
	struct viewer *v = (struct viewer *)obj;

	setzoom(v, 1.0);
}

void makenewviewer(GtkObject *obj)
{
	newviewer();
}

void scrollwin(GtkAdjustment *adj, struct viewer *v)
{
	setcenter(v, GTK_ADJUSTMENT(v->raadjust)->value*PI/12.0,
		(90.0-GTK_ADJUSTMENT(v->decadjust)->value)*PI/180.0);
	printf("New Center: %f %f\n", GTK_ADJUSTMENT(v->raadjust)->value,
	(90.0-GTK_ADJUSTMENT(v->decadjust)->value));
	paint(v);
}

GtkWidget *newbuttonbar(struct viewer *v)
{
        GtkWidget *bar;
	GtkWidget *button;

        bar = gtk_hbox_new(FALSE, 0);

	button = gtk_button_new_with_label("Zoom in");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(zoom_in), (GtkObject *)v);
	gtk_widget_show(button);
	gtk_box_pack_start(GTK_BOX(bar), button, FALSE, FALSE, 0);

	button = gtk_button_new_with_label("Zoom out");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(zoom_out), (GtkObject *)v);
	gtk_widget_show(button);
	gtk_box_pack_start(GTK_BOX(bar), button, FALSE, FALSE, 0);

	button = gtk_button_new_with_label("Zoom = 1");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(zoom_1), (GtkObject *)v);
	gtk_widget_show(button);
	gtk_box_pack_start(GTK_BOX(bar), button, FALSE, FALSE, 0);

	button = gtk_button_new_with_label("new");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(makenewviewer), 0);
	gtk_widget_show(button);
	gtk_box_pack_start(GTK_BOX(bar), button, FALSE, FALSE, 0);

	button = gtk_button_new_with_label("Select");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(objectselector), v);
	gtk_widget_show(button);
	gtk_box_pack_start(GTK_BOX(bar), button, FALSE, FALSE, 0);

	button = gtk_button_new_with_label("Print");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(printviewgui), v);
	gtk_widget_show(button);
	gtk_box_pack_start(GTK_BOX(bar), button, FALSE, FALSE, 0);

	button = gtk_button_new_with_label("Config");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(configure), v);
	gtk_widget_show(button);
	gtk_box_pack_start(GTK_BOX(bar), button, FALSE, FALSE, 0);

	gtk_widget_show(bar);

        return bar;
}

