/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.treedisplay;

import java.awt.*;
import java.awt.geom.*;
import javax.swing.tree.TreeSelectionModel;

/**
 * Routines for drawing a tree in a simple manner. Nodes are drawn as
 * red rectangles and edges are straight lines.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */
public class DefaultTreeRenderer implements TreeRenderer {

	// inherits doc comment
	public void drawNode(TreeLayoutNode node, Graphics g, TreeSelectionModel m)
	{
		Graphics2D g2d = (Graphics2D) g;
		Rectangle rect = new Rectangle(node.getX(), node.getY(), node.getWidth(), node.getHeight());
		Object data = node.getData();

		if (m.getLeadSelectionPath() != null &&
			m.getLeadSelectionPath().getLastPathComponent() == data)
		{
			g2d.setPaint(new Color(153,153,204));
			g2d.fillRect(node.getX() - 3, node.getY() - 3, node.getWidth() + 7, node.getHeight() + 7);
		}

		g2d.setPaint(Color.red);
		g2d.fill(rect);
 		g2d.setStroke(new BasicStroke());
 		g2d.setColor(Color.black);
 		g2d.draw(rect);
	}

	// inherits doc comment
	public void drawEdge(TreeLayoutEdge edge, Graphics g, TreeSelectionModel m)
	{
		Graphics2D g2d = (Graphics2D) g;
		int epsilon = 5;
		TreeLayoutNode parent = edge.getParent();
		TreeLayoutNode child = edge.getChild();
		Point p, c;

		switch (edge.getTreeLayoutModel().getOrientation())
		{
			case TreeDisplay.TOP:
				p = new Point(parent.getX() + (parent.getWidth() / 2), parent.getY() + parent.getHeight());
				c = new Point(child.getX() + (child.getWidth() / 2), child.getY());
				break;

			case TreeDisplay.BOTTOM:
				p = new Point(parent.getX() + (parent.getWidth() / 2), parent.getY());
				c = new Point(child.getX() + (child.getWidth() / 2), child.getY() + child.getHeight());
				break;

			case TreeDisplay.LEFT:
				p = new Point(parent.getX() + parent.getWidth(), parent.getY() + (parent.getHeight() / 2));
				c = new Point(child.getX(), child.getY() + (child.getHeight() / 2));
				break;

			case TreeDisplay.RIGHT:
				p = new Point(parent.getX(), parent.getY() + (parent.getHeight() / 2));
				c = new Point(child.getX() + child.getWidth(), child.getY() + (child.getHeight() / 2));
				break;

			default:
				p = new Point(parent.getX() + (parent.getWidth() / 2), parent.getY() + parent.getHeight());
				c = new Point(child.getX() + (child.getWidth() / 2), child.getY());
				break;
		}

		// clipping
		Shape clip = g2d.getClipBounds();
		Rectangle rect = new Line2D.Double(p, c).getBounds();

		if (rect.width < epsilon)
			rect = new Rectangle(rect.x - 6, rect.y, rect.width + 13, rect.height);
		if (rect.height < epsilon)
			rect = new Rectangle(rect.x, rect.y - 6, rect.width, rect.height + 13);

		if (! clip.intersects(rect))
			return;

		// for drawing a straight line to child.. hmmm.. this is
		// making the assumption that child is direcly below us,
		// which is not always true. so give a little bit of leeway.
		// really only for TOP, BOTTOM orientations.
		if (parent.getChildCount() == 1 && Math.abs((parent.getX() + (parent.getWidth() / 2)) -
													(child.getX() + (child.getWidth() / 2))) < epsilon)
			c.x = p.x;

		g2d.setColor(Color.black);
		g2d.drawLine(p.x, p.y, c.x, c.y);
	}
	
}// DefaultTreeRenderer
